/** Declaration module describing the VS Code debug protocol
 */
export declare module DebugProtocol {
    /** Base class of requests, responses, and events. */
    interface ProtocolMessage {
        /** Sequence number */
        seq: number;
        /** One of "request", "response", or "event" */
        type: string;
    }
    /** Client-initiated request */
    interface Request extends ProtocolMessage {
        /** The command to execute */
        command: string;
        /** Object containing arguments for the command */
        arguments?: any;
    }
    /** Server-initiated event */
    interface Event extends ProtocolMessage {
        /** Type of event */
        event: string;
        /** Event-specific information */
        body?: any;
    }
    /** Server-initiated response to client request */
    interface Response extends ProtocolMessage {
        /** Sequence number of the corresponding request */
        request_seq: number;
        /** Outcome of the request */
        success: boolean;
        /** The command requested */
        command: string;
        /** Contains error message if success == false. */
        message?: string;
        /** Contains request result if success is true and optional error details if success is false. */
        body?: any;
    }
    /** Event message for "initialized" event type.
        This event indicates that the debug adapter is ready to accept configuration requests (e.g. SetBreakpointsRequest, SetExceptionBreakpointsRequest).
        A debug adapter is expected to send this event when it is ready to accept configuration requests.
        The sequence of events/requests is as follows:
        - adapters sends InitializedEvent (at any time)
        - frontend sends zero or more SetBreakpointsRequest
        - frontend sends one SetExceptionBreakpointsRequest (in the future 'zero or one')
        - frontend sends other configuration requests that are added in the future
        - frontend sends one ConfigurationDoneRequest
    */
    interface InitializedEvent extends Event {
    }
    /** Event message for "stopped" event type.
        The event indicates that the execution of the debugee has stopped due to a break condition.
        This can be caused by by a break point previously set, a stepping action has completed or by executing a debugger statement.
    */
    interface StoppedEvent extends Event {
        body: {
            /** The reason for the event (such as: 'step', 'breakpoint', 'exception', 'pause') */
            reason: string;
            /** The thread which was stopped. */
            threadId?: number;
            /** Additional information. E.g. if reason is 'exception', text contains the exception name. */
            text?: string;
        };
    }
    /** Event message for "exited" event type.
        The event indicates that the debugee has exited.
    */
    interface ExitedEvent extends Event {
        body: {
            /** The exit code returned from the debuggee. */
            exitCode: number;
        };
    }
    /** Event message for "terminated" event types.
        The event indicates that debugging of the debuggee has terminated.
    */
    interface TerminatedEvent extends Event {
        body?: {
            /** A debug adapter may set 'restart' to true to request that the front end restarts the session. */
            restart?: boolean;
        };
    }
    /** Event message for "thread" event type.
        The event indicates that a thread has started or exited.
    */
    interface ThreadEvent extends Event {
        body: {
            /** The reason for the event (such as: 'started', 'exited'). */
            reason: string;
            /** The identifier of the thread. */
            threadId: number;
        };
    }
    /** Event message for "output" event type.
        The event indicates that the target has produced output.
    */
    interface OutputEvent extends Event {
        body: {
            /** The category of output (such as: 'console', 'stdout', 'stderr', 'telemetry'). If not specified, 'console' is assumed. */
            category?: string;
            /** The output to report. */
            output: string;
            /** Optional data to report. For the 'telemetry' category the data will be sent to telemetry, for the other categories the data is shown in JSON format. */
            data?: any;
        };
    }
    /** Event message for "breakpoint" event type.
        The event indicates that some information about a breakpoint has changed.
    */
    interface BreakpointEvent extends Event {
        body: {
            /** The reason for the event (such as: 'changed', 'new'). */
            reason: string;
            /** The breakpoint. */
            breakpoint: Breakpoint;
        };
    }
    /** On error that is whenever 'success' is false, the body can provide more details.
     */
    interface ErrorResponse extends Response {
        body: {
            /** An optional, structured error message. */
            error?: Message;
        };
    }
    /** Initialize request; value of command field is "initialize".
    */
    interface InitializeRequest extends Request {
        arguments: InitializeRequestArguments;
    }
    /** Arguments for "initialize" request. */
    interface InitializeRequestArguments {
        /** The ID of the debugger adapter. Used to select or verify debugger adapter. */
        adapterID: string;
        /** If true all line numbers are 1-based (default). */
        linesStartAt1?: boolean;
        /** If true all column numbers are 1-based (default). */
        columnsStartAt1?: boolean;
        /** Determines in what format paths are specified. Possible values are 'path' or 'uri'. The default is 'path', which is the native format. */
        pathFormat?: string;
    }
    /** Response to Initialize request. */
    interface InitializeResponse extends Response {
        /** The capabilities of this debug adapter */
        body?: Capabilites;
    }
    /** ConfigurationDone request; value of command field is "configurationDone".
        The client of the debug protocol must send this request at the end of the sequence of configuration requests (which was started by the InitializedEvent)
    */
    interface ConfigurationDoneRequest extends Request {
        arguments?: ConfigurationDoneArguments;
    }
    /** Arguments for "configurationDone" request. */
    interface ConfigurationDoneArguments {
    }
    /** Response to "configurationDone" request. This is just an acknowledgement, so no body field is required. */
    interface ConfigurationDoneResponse extends Response {
    }
    /** Launch request; value of command field is "launch".
    */
    interface LaunchRequest extends Request {
        arguments: LaunchRequestArguments;
    }
    /** Arguments for "launch" request. */
    interface LaunchRequestArguments {
    }
    /** Response to "launch" request. This is just an acknowledgement, so no body field is required. */
    interface LaunchResponse extends Response {
    }
    /** Attach request; value of command field is "attach".
    */
    interface AttachRequest extends Request {
        arguments: AttachRequestArguments;
    }
    /** Arguments for "attach" request. */
    interface AttachRequestArguments {
    }
    /** Response to "attach" request. This is just an acknowledgement, so no body field is required. */
    interface AttachResponse extends Response {
    }
    /** Disconnect request; value of command field is "disconnect".
    */
    interface DisconnectRequest extends Request {
        arguments?: DisconnectArguments;
    }
    /** Arguments for "disconnect" request. */
    interface DisconnectArguments {
    }
    /** Response to "disconnect" request. This is just an acknowledgement, so no body field is required. */
    interface DisconnectResponse extends Response {
    }
    /** SetBreakpoints request; value of command field is "setBreakpoints".
        Sets multiple breakpoints for a single source and clears all previous breakpoints in that source.
        To clear all breakpoint for a source, specify an empty array.
        When a breakpoint is hit, a StoppedEvent (event type 'breakpoint') is generated.
    */
    interface SetBreakpointsRequest extends Request {
        arguments: SetBreakpointsArguments;
    }
    /** Arguments for "setBreakpoints" request. */
    interface SetBreakpointsArguments {
        /** The source location of the breakpoints; either source.path or source.reference must be specified. */
        source: Source;
        /** The code locations of the breakpoints. */
        breakpoints?: SourceBreakpoint[];
        /** Deprecated: The code locations of the breakpoints. */
        lines?: number[];
    }
    /** Response to "setBreakpoints" request.
        Returned is information about each breakpoint created by this request.
        This includes the actual code location and whether the breakpoint could be verified.
    */
    interface SetBreakpointsResponse extends Response {
        body: {
            /** Information about the breakpoints. The array elements correspond to the elements of the 'breakpoints' (or the deprecated 'lines) array. */
            breakpoints: Breakpoint[];
        };
    }
    /** SetFunctionBreakpoints request; value of command field is "setFunctionBreakpoints".
        Sets multiple function breakpoints and clears all previous function breakpoints.
        To clear all function breakpoint, specify an empty array.
        When a function breakpoint is hit, a StoppedEvent (event type 'function breakpoint') is generated.
    */
    interface SetFunctionBreakpointsRequest extends Request {
        arguments: SetFunctionBreakpointsArguments;
    }
    /** Arguments for "setFunctionBreakpoints" request. */
    interface SetFunctionBreakpointsArguments {
        /** The function names of the breakpoints. */
        breakpoints: FunctionBreakpoint[];
    }
    /** Response to "setFunctionBreakpoints" request.
        Returned is information about each breakpoint created by this request.
    */
    interface SetFunctionBreakpointsResponse extends Response {
        body: {
            /** Information about the breakpoints. The array elements correspond to the elements of the 'breakpoints' array. */
            breakpoints: Breakpoint[];
        };
    }
    /** SetExceptionBreakpoints request; value of command field is "setExceptionBreakpoints".
        Enable that the debuggee stops on exceptions with a StoppedEvent (event type 'exception').
    */
    interface SetExceptionBreakpointsRequest extends Request {
        arguments: SetExceptionBreakpointsArguments;
    }
    /** Arguments for "setExceptionBreakpoints" request. */
    interface SetExceptionBreakpointsArguments {
        /** Names of enabled exception breakpoints. */
        filters: string[];
    }
    /** Response to "setExceptionBreakpoints" request. This is just an acknowledgement, so no body field is required. */
    interface SetExceptionBreakpointsResponse extends Response {
    }
    /** Continue request; value of command field is "continue".
        The request starts the debuggee to run again.
    */
    interface ContinueRequest extends Request {
        arguments: ContinueArguments;
    }
    /** Arguments for "continue" request. */
    interface ContinueArguments {
        /** continue execution for this thread. */
        threadId: number;
    }
    /** Response to "continue" request. This is just an acknowledgement, so no body field is required. */
    interface ContinueResponse extends Response {
    }
    /** Next request; value of command field is "next".
        The request starts the debuggee to run again for one step.
        penDebug will respond with a StoppedEvent (event type 'step') after running the step.
    */
    interface NextRequest extends Request {
        arguments: NextArguments;
    }
    /** Arguments for "next" request. */
    interface NextArguments {
        /** Continue execution for this thread. */
        threadId: number;
    }
    /** Response to "next" request. This is just an acknowledgement, so no body field is required. */
    interface NextResponse extends Response {
    }
    /** StepIn request; value of command field is "stepIn".
        The request starts the debuggee to run again for one step.
        The debug adapter will respond with a StoppedEvent (event type 'step') after running the step.
    */
    interface StepInRequest extends Request {
        arguments: StepInArguments;
    }
    /** Arguments for "stepIn" request. */
    interface StepInArguments {
        /** Continue execution for this thread. */
        threadId: number;
    }
    /** Response to "stepIn" request. This is just an acknowledgement, so no body field is required. */
    interface StepInResponse extends Response {
    }
    /** StepOutIn request; value of command field is "stepOut".
        The request starts the debuggee to run again for one step.
        penDebug will respond with a StoppedEvent (event type 'step') after running the step.
    */
    interface StepOutRequest extends Request {
        arguments: StepOutArguments;
    }
    /** Arguments for "stepOut" request. */
    interface StepOutArguments {
        /** Continue execution for this thread. */
        threadId: number;
    }
    /** Response to "stepOut" request. This is just an acknowledgement, so no body field is required. */
    interface StepOutResponse extends Response {
    }
    /** Pause request; value of command field is "pause".
        The request suspenses the debuggee.
        penDebug will respond with a StoppedEvent (event type 'pause') after a successful 'pause' command.
    */
    interface PauseRequest extends Request {
        arguments: PauseArguments;
    }
    /** Arguments for "pause" request. */
    interface PauseArguments {
        /** Pause execution for this thread. */
        threadId: number;
    }
    /** Response to "pause" request. This is just an acknowledgement, so no body field is required. */
    interface PauseResponse extends Response {
    }
    /** StackTrace request; value of command field is "stackTrace".
        The request returns a stacktrace from the current execution state.
    */
    interface StackTraceRequest extends Request {
        arguments: StackTraceArguments;
    }
    /** Arguments for "stackTrace" request. */
    interface StackTraceArguments {
        /** Retrieve the stacktrace for this thread. */
        threadId: number;
        /** The maximum number of frames to return. If levels is not specified or 0, all frames are returned. */
        levels?: number;
    }
    /** Response to "stackTrace" request. */
    interface StackTraceResponse extends Response {
        body: {
            /** The frames of the stackframe. If the array has length zero, there are no stackframes available.
                This means that there is no location information available. */
            stackFrames: StackFrame[];
        };
    }
    /** Scopes request; value of command field is "scopes".
        The request returns the variable scopes for a given stackframe ID.
    */
    interface ScopesRequest extends Request {
        arguments: ScopesArguments;
    }
    /** Arguments for "scopes" request. */
    interface ScopesArguments {
        /** Retrieve the scopes for this stackframe. */
        frameId: number;
    }
    /** Response to "scopes" request. */
    interface ScopesResponse extends Response {
        body: {
            /** The scopes of the stackframe. If the array has length zero, there are no scopes available. */
            scopes: Scope[];
        };
    }
    /** Variables request; value of command field is "variables".
        Retrieves all children for the given variable reference.
    */
    interface VariablesRequest extends Request {
        arguments: VariablesArguments;
    }
    /** Arguments for "variables" request. */
    interface VariablesArguments {
        /** The Variable reference. */
        variablesReference: number;
    }
    /** Response to "variables" request. */
    interface VariablesResponse extends Response {
        body: {
            /** All children for the given variable reference */
            variables: Variable[];
        };
    }
    /** Source request; value of command field is "source".
        The request retrieves the source code for a given source reference.
    */
    interface SourceRequest extends Request {
        arguments: SourceArguments;
    }
    /** Arguments for "source" request. */
    interface SourceArguments {
        /** The reference to the source. This is the value received in Source.reference. */
        sourceReference: number;
    }
    /** Response to "source" request. */
    interface SourceResponse extends Response {
        body: {
            /** Content of the source reference */
            content: string;
        };
    }
    /** Thread request; value of command field is "threads".
        The request retrieves a list of all threads.
    */
    interface ThreadsRequest extends Request {
    }
    /** Response to "threads" request. */
    interface ThreadsResponse extends Response {
        body: {
            /** All threads. */
            threads: Thread[];
        };
    }
    /** Evaluate request; value of command field is "evaluate".
        Evaluates the given expression in the context of the top most stack frame.
        The expression has access to any variables and arguments that are in scope.
    */
    interface EvaluateRequest extends Request {
        arguments: EvaluateArguments;
    }
    /** Arguments for "evaluate" request. */
    interface EvaluateArguments {
        /** The expression to evaluate. */
        expression: string;
        /** Evaluate the expression in the scope of this stack frame. If not specified, the expression is evaluated in the global scope. */
        frameId?: number;
        /** The context in which the evaluate request is run. Possible values are 'watch' if evaluate is run in a watch, 'repl' if run from the REPL console, or 'hover' if run from a data hover. */
        context?: string;
    }
    /** Response to "evaluate" request. */
    interface EvaluateResponse extends Response {
        body: {
            /** The result of the evaluate. */
            result: string;
            /** If variablesReference is > 0, the evaluate result is structured and its children can be retrieved by passing variablesReference to the VariablesRequest */
            variablesReference: number;
        };
    }
    /** Information about the capabilities of a debug adapter. */
    interface Capabilites {
        /** The debug adapter supports the configurationDoneRequest. */
        supportsConfigurationDoneRequest?: boolean;
        /** The debug adapter supports functionBreakpoints. */
        supportsFunctionBreakpoints?: boolean;
        /** The debug adapter supports conditionalBreakpoints. */
        supportsConditionalBreakpoints?: boolean;
        /** The debug adapter supports a (side effect free) evaluate request for data hovers. */
        supportsEvaluateForHovers?: boolean;
        /** Available filters for the setExceptionBreakpoints request. */
        exceptionBreakpointFilters?: [{
            filter: string;
            label: string;
        }];
    }
    /** A structured message object. Used to return errors from requests. */
    interface Message {
        /** Unique identifier for the message. */
        id: number;
        /** A format string for the message. Embedded variables have the form '{name}'.
            If variable name starts with an underscore character, the variable does not contain user data (PII) and can be safely used for telemetry purposes. */
        format: string;
        /** An object used as a dictionary for looking up the variables in the format string. */
        variables?: {
            [key: string]: string;
        };
        /** if true send to telemetry */
        sendTelemetry?: boolean;
        /** if true show user */
        showUser?: boolean;
    }
    /** A Thread */
    interface Thread {
        /** Unique identifier for the thread. */
        id: number;
        /** A name of the thread. */
        name: string;
    }
    /** A Source is a descriptor for source code. It is returned from the debug adapter as part of a StackFrame and it is used by clients when specifying breakpoints. */
    interface Source {
        /** The short name of the source. Every source returned from the debug adapter has a name. When specifying a source to the debug adapter this name is optional. */
        name?: string;
        /** The long (absolute) path of the source. It is not guaranteed that the source exists at this location. */
        path?: string;
        /** If sourceReference > 0 the contents of the source can be retrieved through the SourceRequest. A sourceReference is only valid for a session, so it must not be used to persist a source. */
        sourceReference?: number;
        /** The (optional) origin of this source: possible values "internal module", "inlined content from source map", etc. */
        origin?: string;
        /** Optional data that a debug adapter might want to loop through the client. The client should leave the data intact and persist it across sessions. The client should not interpret the data. */
        adapterData?: any;
    }
    /** A Stackframe contains the source location. */
    interface StackFrame {
        /** An identifier for the stack frame. This id can be used to retrieve the scopes of the frame with the 'scopesRequest'. */
        id: number;
        /** The name of the stack frame, typically a method name */
        name: string;
        /** The optional source of the frame. */
        source?: Source;
        /** The line within the file of the frame. If source is null or doesn't exist, line is 0 and must be ignored. */
        line: number;
        /** The column within the line. If source is null or doesn't exist, column is 0 and must be ignored. */
        column: number;
    }
    /** A Scope is a named container for variables. */
    interface Scope {
        /** name of the scope (as such 'Arguments', 'Locals') */
        name: string;
        /** The variables of this scope can be retrieved by passing the value of variablesReference to the VariablesRequest. */
        variablesReference: number;
        /** If true, the number of variables in this scope is large or expensive to retrieve. */
        expensive: boolean;
    }
    /** A Variable is a name/value pair.
        If the value is structured (has children), a handle is provided to retrieve the children with the VariablesRequest.
    */
    interface Variable {
        /** The variable's name */
        name: string;
        /** The variable's value. For structured objects this can be a multi line text, e.g. for a function the body of a function. */
        value: string;
        /** If variablesReference is > 0, the variable is structured and its children can be retrieved by passing variablesReference to the VariablesRequest. */
        variablesReference: number;
    }
    /** Properties of a breakpoint passed to the setBreakpoints request.
    */
    interface SourceBreakpoint {
        /** The source line of the breakpoint. */
        line: number;
        /** An optional source column of the breakpoint. */
        column?: number;
        /** An optional expression for conditional breakpoints. */
        condition?: string;
    }
    /** Properties of a breakpoint passed to the setFunctionBreakpoints request.
    */
    interface FunctionBreakpoint {
        /** The name of the function. */
        name: string;
        /** An optional expression for conditional breakpoints. */
        condition?: string;
    }
    /** Information about a Breakpoint created in setBreakpoints or setFunctionBreakpoints.
    */
    interface Breakpoint {
        /** An optional unique identifier for the breakpoint. */
        id?: number;
        /** If true breakpoint could be set (but not necessarily at the desired location).  */
        verified: boolean;
        /** An optional message about the state of the breakpoint. This is shown to the user and can be used to explain why a breakpoint could not be verified. */
        message?: string;
        /** The source where the breakpoint is located. */
        source?: Source;
        /** The actual line of the breakpoint. */
        line?: number;
        /** The actual column of the breakpoint. */
        column?: number;
    }
}
