
                (function() {
                    var nodeEnv = typeof require !== 'undefined' && typeof process !== 'undefined';
                    var __module = nodeEnv ? module : {exports:{}};
                    var __filename = 'engine-dev/cocos2d/renderer/memop/timsort.js';
                    var __require = nodeEnv ? function (request) {
                        return require(request);
                    } : function (request) {
                        return __quick_compile_engine__.require(request, __filename);
                    };
                    function __define (exports, require, module) {
                        if (!nodeEnv) {__quick_compile_engine__.registerModule(__filename, module);}"use strict";

exports.__esModule = true;
exports["default"] = _default;
// reference: https://github.com/mziccard/node-timsort

/**
 * Default minimum size of a run.
 */
var DEFAULT_MIN_MERGE = 32;
/**
 * Minimum ordered subsequece required to do galloping.
 */

var DEFAULT_MIN_GALLOPING = 7;
/**
 * Default tmp storage length. Can increase depending on the size of the
 * smallest run to merge.
 */

var DEFAULT_TMP_STORAGE_LENGTH = 256;
/**
 * Pre-computed powers of 10 for efficient lexicographic comparison of
 * small integers.
 */

var POWERS_OF_TEN = [1e0, 1e1, 1e2, 1e3, 1e4, 1e5, 1e6, 1e7, 1e8, 1e9];
/**
 * Estimate the logarithm base 10 of a small integer.
 *
 * @param {number} x - The integer to estimate the logarithm of.
 * @return {number} - The estimated logarithm of the integer.
 */

function log10(x) {
  if (x < 1e5) {
    if (x < 1e2) {
      return x < 1e1 ? 0 : 1;
    }

    if (x < 1e4) {
      return x < 1e3 ? 2 : 3;
    }

    return 4;
  }

  if (x < 1e7) {
    return x < 1e6 ? 5 : 6;
  }

  if (x < 1e9) {
    return x < 1e8 ? 7 : 8;
  }

  return 9;
}
/**
 * Default alphabetical comparison of items.
 *
 * @param {string|object|number} a - First element to compare.
 * @param {string|object|number} b - Second element to compare.
 * @return {number} - A positive number if a.toString() > b.toString(), a
 * negative number if .toString() < b.toString(), 0 otherwise.
 */


function alphabeticalCompare(a, b) {
  if (a === b) {
    return 0;
  }

  if (~~a === a && ~~b === b) {
    if (a === 0 || b === 0) {
      return a < b ? -1 : 1;
    }

    if (a < 0 || b < 0) {
      if (b >= 0) {
        return -1;
      }

      if (a >= 0) {
        return 1;
      }

      a = -a;
      b = -b;
    }

    var al = log10(a);
    var bl = log10(b);
    var t = 0;

    if (al < bl) {
      a *= POWERS_OF_TEN[bl - al - 1];
      b /= 10;
      t = -1;
    } else if (al > bl) {
      b *= POWERS_OF_TEN[al - bl - 1];
      a /= 10;
      t = 1;
    }

    if (a === b) {
      return t;
    }

    return a < b ? -1 : 1;
  }

  var aStr = String(a);
  var bStr = String(b);

  if (aStr === bStr) {
    return 0;
  }

  return aStr < bStr ? -1 : 1;
}
/**
 * Compute minimum run length for TimSort
 *
 * @param {number} n - The size of the array to sort.
 */


function minRunLength(n) {
  var r = 0;

  while (n >= DEFAULT_MIN_MERGE) {
    r |= n & 1;
    n >>= 1;
  }

  return n + r;
}
/**
 * Counts the length of a monotonically ascending or strictly monotonically
 * descending sequence (run) starting at array[lo] in the range [lo, hi). If
 * the run is descending it is made ascending.
 *
 * @param {array} array - The array to reverse.
 * @param {number} lo - First element in the range (inclusive).
 * @param {number} hi - Last element in the range.
 * @param {function} compare - Item comparison function.
 * @return {number} - The length of the run.
 */


function makeAscendingRun(array, lo, hi, compare) {
  var runHi = lo + 1;

  if (runHi === hi) {
    return 1;
  } // Descending


  if (compare(array[runHi++], array[lo]) < 0) {
    while (runHi < hi && compare(array[runHi], array[runHi - 1]) < 0) {
      runHi++;
    }

    reverseRun(array, lo, runHi); // Ascending
  } else {
    while (runHi < hi && compare(array[runHi], array[runHi - 1]) >= 0) {
      runHi++;
    }
  }

  return runHi - lo;
}
/**
 * Reverse an array in the range [lo, hi).
 *
 * @param {array} array - The array to reverse.
 * @param {number} lo - First element in the range (inclusive).
 * @param {number} hi - Last element in the range.
 */


function reverseRun(array, lo, hi) {
  hi--;

  while (lo < hi) {
    var t = array[lo];
    array[lo++] = array[hi];
    array[hi--] = t;
  }
}
/**
 * Perform the binary sort of the array in the range [lo, hi) where start is
 * the first element possibly out of order.
 *
 * @param {array} array - The array to sort.
 * @param {number} lo - First element in the range (inclusive).
 * @param {number} hi - Last element in the range.
 * @param {number} start - First element possibly out of order.
 * @param {function} compare - Item comparison function.
 */


function binaryInsertionSort(array, lo, hi, start, compare) {
  if (start === lo) {
    start++;
  }

  for (; start < hi; start++) {
    var pivot = array[start]; // Ranges of the array where pivot belongs

    var left = lo;
    var right = start;
    /*
     *   pivot >= array[i] for i in [lo, left)
     *   pivot <  array[i] for i in  in [right, start)
     */

    while (left < right) {
      var mid = left + right >>> 1;

      if (compare(pivot, array[mid]) < 0) {
        right = mid;
      } else {
        left = mid + 1;
      }
    }
    /*
     * Move elements right to make room for the pivot. If there are elements
     * equal to pivot, left points to the first slot after them: this is also
     * a reason for which TimSort is stable
     */


    var n = start - left; // Switch is just an optimization for small arrays

    switch (n) {
      case 3:
        array[left + 3] = array[left + 2];

      /* falls through */

      case 2:
        array[left + 2] = array[left + 1];

      /* falls through */

      case 1:
        array[left + 1] = array[left];
        break;

      default:
        while (n > 0) {
          array[left + n] = array[left + n - 1];
          n--;
        }

    }

    array[left] = pivot;
  }
}
/**
 * Find the position at which to insert a value in a sorted range. If the range
 * contains elements equal to the value the leftmost element index is returned
 * (for stability).
 *
 * @param {number} value - Value to insert.
 * @param {array} array - The array in which to insert value.
 * @param {number} start - First element in the range.
 * @param {number} length - Length of the range.
 * @param {number} hint - The index at which to begin the search.
 * @param {function} compare - Item comparison function.
 * @return {number} - The index where to insert value.
 */


function gallopLeft(value, array, start, length, hint, compare) {
  var lastOffset = 0;
  var maxOffset = 0;
  var offset = 1;

  if (compare(value, array[start + hint]) > 0) {
    maxOffset = length - hint;

    while (offset < maxOffset && compare(value, array[start + hint + offset]) > 0) {
      lastOffset = offset;
      offset = (offset << 1) + 1;

      if (offset <= 0) {
        offset = maxOffset;
      }
    }

    if (offset > maxOffset) {
      offset = maxOffset;
    } // Make offsets relative to start


    lastOffset += hint;
    offset += hint; // value <= array[start + hint]
  } else {
    maxOffset = hint + 1;

    while (offset < maxOffset && compare(value, array[start + hint - offset]) <= 0) {
      lastOffset = offset;
      offset = (offset << 1) + 1;

      if (offset <= 0) {
        offset = maxOffset;
      }
    }

    if (offset > maxOffset) {
      offset = maxOffset;
    } // Make offsets relative to start


    var tmp = lastOffset;
    lastOffset = hint - offset;
    offset = hint - tmp;
  }
  /*
   * Now array[start+lastOffset] < value <= array[start+offset], so value
   * belongs somewhere in the range (start + lastOffset, start + offset]. Do a
   * binary search, with invariant array[start + lastOffset - 1] < value <=
   * array[start + offset].
   */


  lastOffset++;

  while (lastOffset < offset) {
    var m = lastOffset + (offset - lastOffset >>> 1);

    if (compare(value, array[start + m]) > 0) {
      lastOffset = m + 1;
    } else {
      offset = m;
    }
  }

  return offset;
}
/**
 * Find the position at which to insert a value in a sorted range. If the range
 * contains elements equal to the value the rightmost element index is returned
 * (for stability).
 *
 * @param {number} value - Value to insert.
 * @param {array} array - The array in which to insert value.
 * @param {number} start - First element in the range.
 * @param {number} length - Length of the range.
 * @param {number} hint - The index at which to begin the search.
 * @param {function} compare - Item comparison function.
 * @return {number} - The index where to insert value.
 */


function gallopRight(value, array, start, length, hint, compare) {
  var lastOffset = 0;
  var maxOffset = 0;
  var offset = 1;

  if (compare(value, array[start + hint]) < 0) {
    maxOffset = hint + 1;

    while (offset < maxOffset && compare(value, array[start + hint - offset]) < 0) {
      lastOffset = offset;
      offset = (offset << 1) + 1;

      if (offset <= 0) {
        offset = maxOffset;
      }
    }

    if (offset > maxOffset) {
      offset = maxOffset;
    } // Make offsets relative to start


    var tmp = lastOffset;
    lastOffset = hint - offset;
    offset = hint - tmp; // value >= array[start + hint]
  } else {
    maxOffset = length - hint;

    while (offset < maxOffset && compare(value, array[start + hint + offset]) >= 0) {
      lastOffset = offset;
      offset = (offset << 1) + 1;

      if (offset <= 0) {
        offset = maxOffset;
      }
    }

    if (offset > maxOffset) {
      offset = maxOffset;
    } // Make offsets relative to start


    lastOffset += hint;
    offset += hint;
  }
  /*
   * Now array[start+lastOffset] < value <= array[start+offset], so value
   * belongs somewhere in the range (start + lastOffset, start + offset]. Do a
   * binary search, with invariant array[start + lastOffset - 1] < value <=
   * array[start + offset].
   */


  lastOffset++;

  while (lastOffset < offset) {
    var m = lastOffset + (offset - lastOffset >>> 1);

    if (compare(value, array[start + m]) < 0) {
      offset = m;
    } else {
      lastOffset = m + 1;
    }
  }

  return offset;
}

var TimSort =
/*#__PURE__*/
function () {
  function TimSort(array, compare) {
    this.array = array;
    this.compare = compare;
    this.minGallop = DEFAULT_MIN_GALLOPING;
    this.length = array.length;
    this.tmpStorageLength = DEFAULT_TMP_STORAGE_LENGTH;

    if (this.length < 2 * DEFAULT_TMP_STORAGE_LENGTH) {
      this.tmpStorageLength = this.length >>> 1;
    }

    this.tmp = new Array(this.tmpStorageLength);
    this.stackLength = this.length < 120 ? 5 : this.length < 1542 ? 10 : this.length < 119151 ? 19 : 40;
    this.runStart = new Array(this.stackLength);
    this.runLength = new Array(this.stackLength);
    this.stackSize = 0;
  }
  /**
   * Push a new run on TimSort's stack.
   *
   * @param {number} runStart - Start index of the run in the original array.
   * @param {number} runLength - Length of the run;
   */


  var _proto = TimSort.prototype;

  _proto.pushRun = function pushRun(runStart, runLength) {
    this.runStart[this.stackSize] = runStart;
    this.runLength[this.stackSize] = runLength;
    this.stackSize += 1;
  }
  /**
   * Merge runs on TimSort's stack so that the following holds for all i:
   * 1) runLength[i - 3] > runLength[i - 2] + runLength[i - 1]
   * 2) runLength[i - 2] > runLength[i - 1]
   */
  ;

  _proto.mergeRuns = function mergeRuns() {
    while (this.stackSize > 1) {
      var n = this.stackSize - 2;

      if (n >= 1 && this.runLength[n - 1] <= this.runLength[n] + this.runLength[n + 1] || n >= 2 && this.runLength[n - 2] <= this.runLength[n] + this.runLength[n - 1]) {
        if (this.runLength[n - 1] < this.runLength[n + 1]) {
          n--;
        }
      } else if (this.runLength[n] > this.runLength[n + 1]) {
        break;
      }

      this.mergeAt(n);
    }
  }
  /**
   * Merge all runs on TimSort's stack until only one remains.
   */
  ;

  _proto.forceMergeRuns = function forceMergeRuns() {
    while (this.stackSize > 1) {
      var n = this.stackSize - 2;

      if (n > 0 && this.runLength[n - 1] < this.runLength[n + 1]) {
        n--;
      }

      this.mergeAt(n);
    }
  }
  /**
   * Merge the runs on the stack at positions i and i+1. Must be always be called
   * with i=stackSize-2 or i=stackSize-3 (that is, we merge on top of the stack).
   *
   * @param {number} i - Index of the run to merge in TimSort's stack.
   */
  ;

  _proto.mergeAt = function mergeAt(i) {
    var compare = this.compare;
    var array = this.array;
    var start1 = this.runStart[i];
    var length1 = this.runLength[i];
    var start2 = this.runStart[i + 1];
    var length2 = this.runLength[i + 1];
    this.runLength[i] = length1 + length2;

    if (i === this.stackSize - 3) {
      this.runStart[i + 1] = this.runStart[i + 2];
      this.runLength[i + 1] = this.runLength[i + 2];
    }

    this.stackSize--;
    /*
     * Find where the first element in the second run goes in run1. Previous
     * elements in run1 are already in place
     */

    var k = gallopRight(array[start2], array, start1, length1, 0, compare);
    start1 += k;
    length1 -= k;

    if (length1 === 0) {
      return;
    }
    /*
     * Find where the last element in the first run goes in run2. Next elements
     * in run2 are already in place
     */


    length2 = gallopLeft(array[start1 + length1 - 1], array, start2, length2, length2 - 1, compare);

    if (length2 === 0) {
      return;
    }
    /*
     * Merge remaining runs. A tmp array with length = min(length1, length2) is
     * used
     */


    if (length1 <= length2) {
      this.mergeLow(start1, length1, start2, length2);
    } else {
      this.mergeHigh(start1, length1, start2, length2);
    }
  }
  /**
   * Merge two adjacent runs in a stable way. The runs must be such that the
   * first element of run1 is bigger than the first element in run2 and the
   * last element of run1 is greater than all the elements in run2.
   * The method should be called when run1.length <= run2.length as it uses
   * TimSort temporary array to store run1. Use mergeHigh if run1.length >
   * run2.length.
   *
   * @param {number} start1 - First element in run1.
   * @param {number} length1 - Length of run1.
   * @param {number} start2 - First element in run2.
   * @param {number} length2 - Length of run2.
   */
  ;

  _proto.mergeLow = function mergeLow(start1, length1, start2, length2) {
    var compare = this.compare;
    var array = this.array;
    var tmp = this.tmp;
    var i = 0;

    for (i = 0; i < length1; i++) {
      tmp[i] = array[start1 + i];
    }

    var cursor1 = 0;
    var cursor2 = start2;
    var dest = start1;
    array[dest++] = array[cursor2++];

    if (--length2 === 0) {
      for (i = 0; i < length1; i++) {
        array[dest + i] = tmp[cursor1 + i];
      }

      return;
    }

    if (length1 === 1) {
      for (i = 0; i < length2; i++) {
        array[dest + i] = array[cursor2 + i];
      }

      array[dest + length2] = tmp[cursor1];
      return;
    }

    var minGallop = this.minGallop;

    while (true) {
      var count1 = 0;
      var count2 = 0;
      var exit = false;

      do {
        if (compare(array[cursor2], tmp[cursor1]) < 0) {
          array[dest++] = array[cursor2++];
          count2++;
          count1 = 0;

          if (--length2 === 0) {
            exit = true;
            break;
          }
        } else {
          array[dest++] = tmp[cursor1++];
          count1++;
          count2 = 0;

          if (--length1 === 1) {
            exit = true;
            break;
          }
        }
      } while ((count1 | count2) < minGallop);

      if (exit) {
        break;
      }

      do {
        count1 = gallopRight(array[cursor2], tmp, cursor1, length1, 0, compare);

        if (count1 !== 0) {
          for (i = 0; i < count1; i++) {
            array[dest + i] = tmp[cursor1 + i];
          }

          dest += count1;
          cursor1 += count1;
          length1 -= count1;

          if (length1 <= 1) {
            exit = true;
            break;
          }
        }

        array[dest++] = array[cursor2++];

        if (--length2 === 0) {
          exit = true;
          break;
        }

        count2 = gallopLeft(tmp[cursor1], array, cursor2, length2, 0, compare);

        if (count2 !== 0) {
          for (i = 0; i < count2; i++) {
            array[dest + i] = array[cursor2 + i];
          }

          dest += count2;
          cursor2 += count2;
          length2 -= count2;

          if (length2 === 0) {
            exit = true;
            break;
          }
        }

        array[dest++] = tmp[cursor1++];

        if (--length1 === 1) {
          exit = true;
          break;
        }

        minGallop--;
      } while (count1 >= DEFAULT_MIN_GALLOPING || count2 >= DEFAULT_MIN_GALLOPING);

      if (exit) {
        break;
      }

      if (minGallop < 0) {
        minGallop = 0;
      }

      minGallop += 2;
    }

    this.minGallop = minGallop;

    if (minGallop < 1) {
      this.minGallop = 1;
    }

    if (length1 === 1) {
      for (i = 0; i < length2; i++) {
        array[dest + i] = array[cursor2 + i];
      }

      array[dest + length2] = tmp[cursor1];
    } else if (length1 === 0) {
      throw new Error('mergeLow preconditions were not respected');
    } else {
      for (i = 0; i < length1; i++) {
        array[dest + i] = tmp[cursor1 + i];
      }
    }
  }
  /**
   * Merge two adjacent runs in a stable way. The runs must be such that the
   * first element of run1 is bigger than the first element in run2 and the
   * last element of run1 is greater than all the elements in run2.
   * The method should be called when run1.length > run2.length as it uses
   * TimSort temporary array to store run2. Use mergeLow if run1.length <=
   * run2.length.
   *
   * @param {number} start1 - First element in run1.
   * @param {number} length1 - Length of run1.
   * @param {number} start2 - First element in run2.
   * @param {number} length2 - Length of run2.
   */
  ;

  _proto.mergeHigh = function mergeHigh(start1, length1, start2, length2) {
    var compare = this.compare;
    var array = this.array;
    var tmp = this.tmp;
    var i = 0;

    for (i = 0; i < length2; i++) {
      tmp[i] = array[start2 + i];
    }

    var cursor1 = start1 + length1 - 1;
    var cursor2 = length2 - 1;
    var dest = start2 + length2 - 1;
    var customCursor = 0;
    var customDest = 0;
    array[dest--] = array[cursor1--];

    if (--length1 === 0) {
      customCursor = dest - (length2 - 1);

      for (i = 0; i < length2; i++) {
        array[customCursor + i] = tmp[i];
      }

      return;
    }

    if (length2 === 1) {
      dest -= length1;
      cursor1 -= length1;
      customDest = dest + 1;
      customCursor = cursor1 + 1;

      for (i = length1 - 1; i >= 0; i--) {
        array[customDest + i] = array[customCursor + i];
      }

      array[dest] = tmp[cursor2];
      return;
    }

    var minGallop = this.minGallop;

    while (true) {
      var count1 = 0;
      var count2 = 0;
      var exit = false;

      do {
        if (compare(tmp[cursor2], array[cursor1]) < 0) {
          array[dest--] = array[cursor1--];
          count1++;
          count2 = 0;

          if (--length1 === 0) {
            exit = true;
            break;
          }
        } else {
          array[dest--] = tmp[cursor2--];
          count2++;
          count1 = 0;

          if (--length2 === 1) {
            exit = true;
            break;
          }
        }
      } while ((count1 | count2) < minGallop);

      if (exit) {
        break;
      }

      do {
        count1 = length1 - gallopRight(tmp[cursor2], array, start1, length1, length1 - 1, compare);

        if (count1 !== 0) {
          dest -= count1;
          cursor1 -= count1;
          length1 -= count1;
          customDest = dest + 1;
          customCursor = cursor1 + 1;

          for (i = count1 - 1; i >= 0; i--) {
            array[customDest + i] = array[customCursor + i];
          }

          if (length1 === 0) {
            exit = true;
            break;
          }
        }

        array[dest--] = tmp[cursor2--];

        if (--length2 === 1) {
          exit = true;
          break;
        }

        count2 = length2 - gallopLeft(array[cursor1], tmp, 0, length2, length2 - 1, compare);

        if (count2 !== 0) {
          dest -= count2;
          cursor2 -= count2;
          length2 -= count2;
          customDest = dest + 1;
          customCursor = cursor2 + 1;

          for (i = 0; i < count2; i++) {
            array[customDest + i] = tmp[customCursor + i];
          }

          if (length2 <= 1) {
            exit = true;
            break;
          }
        }

        array[dest--] = array[cursor1--];

        if (--length1 === 0) {
          exit = true;
          break;
        }

        minGallop--;
      } while (count1 >= DEFAULT_MIN_GALLOPING || count2 >= DEFAULT_MIN_GALLOPING);

      if (exit) {
        break;
      }

      if (minGallop < 0) {
        minGallop = 0;
      }

      minGallop += 2;
    }

    this.minGallop = minGallop;

    if (minGallop < 1) {
      this.minGallop = 1;
    }

    if (length2 === 1) {
      dest -= length1;
      cursor1 -= length1;
      customDest = dest + 1;
      customCursor = cursor1 + 1;

      for (i = length1 - 1; i >= 0; i--) {
        array[customDest + i] = array[customCursor + i];
      }

      array[dest] = tmp[cursor2];
    } else if (length2 === 0) {
      throw new Error('mergeHigh preconditions were not respected');
    } else {
      customCursor = dest - (length2 - 1);

      for (i = 0; i < length2; i++) {
        array[customCursor + i] = tmp[i];
      }
    }
  };

  return TimSort;
}();
/**
 * Sort an array in the range [lo, hi) using TimSort.
 *
 * @param {array} array - The array to sort.
 * @param {number} lo - First element in the range (inclusive).
 * @param {number} hi - Last element in the range.
 * @param {function=} compare - Item comparison function. Default is alphabetical.
 */


function _default(array, lo, hi, compare) {
  if (!Array.isArray(array)) {
    throw new TypeError('Can only sort arrays');
  }
  /*
   * Handle the case where a comparison function is not provided. We do
   * lexicographic sorting
   */


  if (lo === undefined) {
    lo = 0;
  }

  if (hi === undefined) {
    hi = array.length;
  }

  if (compare === undefined) {
    compare = alphabeticalCompare;
  }

  var remaining = hi - lo; // The array is already sorted

  if (remaining < 2) {
    return;
  }

  var runLength = 0; // On small arrays binary sort can be used directly

  if (remaining < DEFAULT_MIN_MERGE) {
    runLength = makeAscendingRun(array, lo, hi, compare);
    binaryInsertionSort(array, lo, hi, lo + runLength, compare);
    return;
  }

  var ts = new TimSort(array, compare);
  var minRun = minRunLength(remaining);

  do {
    runLength = makeAscendingRun(array, lo, hi, compare);

    if (runLength < minRun) {
      var force = remaining;

      if (force > minRun) {
        force = minRun;
      }

      binaryInsertionSort(array, lo, lo + force, lo + runLength, compare);
      runLength = force;
    } // Push new run and merge if necessary


    ts.pushRun(lo, runLength);
    ts.mergeRuns(); // Go find next run

    remaining -= runLength;
    lo += runLength;
  } while (remaining !== 0); // Force merging of remaining runs


  ts.forceMergeRuns();
}

module.exports = exports["default"];
                    }
                    if (nodeEnv) {
                        __define(__module.exports, __require, __module);
                    }
                    else {
                        __quick_compile_engine__.registerModuleFunc(__filename, function () {
                            __define(__module.exports, __require, __module);
                        });
                    }
                })();
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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