
                (function() {
                    var nodeEnv = typeof require !== 'undefined' && typeof process !== 'undefined';
                    var __module = nodeEnv ? module : {exports:{}};
                    var __filename = 'engine-dev/cocos2d/core/load-pipeline/pipeline.js';
                    var __require = nodeEnv ? function (request) {
                        return require(request);
                    } : function (request) {
                        return __quick_compile_engine__.require(request, __filename);
                    };
                    function __define (exports, require, module) {
                        if (!nodeEnv) {__quick_compile_engine__.registerModule(__filename, module);}"use strict";

/****************************************************************************
 Copyright (c) 2013-2016 Chukong Technologies Inc.
 Copyright (c) 2017-2018 Xiamen Yaji Software Co., Ltd.

 https://www.cocos.com/

 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated engine source code (the "Software"), a limited,
  worldwide, royalty-free, non-assignable, revocable and non-exclusive license
 to use Cocos Creator solely to develop games on your target platforms. You shall
  not use Cocos Creator software for developing other software or tools that's
  used for developing games. You are not granted to publish, distribute,
  sublicense, and/or sell copies of Cocos Creator.

 The software or tools in this License Agreement are licensed, not sold.
 Xiamen Yaji Software Co., Ltd. reserves all rights not expressly granted to you.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 THE SOFTWARE.
 ****************************************************************************/
var js = require('../platform/js');

var LoadingItems = require('./loading-items');

var ItemState = LoadingItems.ItemState;

function flow(pipe, item) {
  var pipeId = pipe.id;
  var itemState = item.states[pipeId];
  var next = pipe.next;
  var pipeline = pipe.pipeline;

  if (item.error || itemState === ItemState.WORKING || itemState === ItemState.ERROR) {
    return;
  } else if (itemState === ItemState.COMPLETE) {
    if (next) {
      flow(next, item);
    } else {
      pipeline.flowOut(item);
    }
  } else {
    item.states[pipeId] = ItemState.WORKING; // Pass async callback in case it's a async call

    var result = pipe.handle(item, function (err, result) {
      if (err) {
        item.error = err;
        item.states[pipeId] = ItemState.ERROR;
        pipeline.flowOut(item);
      } else {
        // Result can be null, then it means no result for this pipe
        if (result) {
          item.content = result;
        }

        item.states[pipeId] = ItemState.COMPLETE;

        if (next) {
          flow(next, item);
        } else {
          pipeline.flowOut(item);
        }
      }
    }); // If result exists (not undefined, null is ok), then we go with sync call flow

    if (result instanceof Error) {
      item.error = result;
      item.states[pipeId] = ItemState.ERROR;
      pipeline.flowOut(item);
    } else if (result !== undefined) {
      // Result can be null, then it means no result for this pipe
      if (result !== null) {
        item.content = result;
      }

      item.states[pipeId] = ItemState.COMPLETE;

      if (next) {
        flow(next, item);
      } else {
        pipeline.flowOut(item);
      }
    }
  }
}
/**
 * !#en
 * A pipeline describes a sequence of manipulations, each manipulation is called a pipe.<br/>
 * It's designed for loading process. so items should be urls, and the url will be the identity of each item during the process.<br/>
 * A list of items can flow in the pipeline and it will output the results of all pipes.<br/>
 * They flow in the pipeline like water in tubes, they go through pipe by pipe separately.<br/>
 * Finally all items will flow out the pipeline and the process is finished.
 *
 * !#zh
 * pipeline 描述了一系列的操作，每个操作都被称为 pipe。<br/>
 * 它被设计来做加载过程的流程管理。所以 item 应该是 url，并且该 url 将是在处理中的每个 item 的身份标识。<br/>
 * 一个 item 列表可以在 pipeline 中流动，它将输出加载项经过所有 pipe 之后的结果。<br/>
 * 它们穿过 pipeline 就像水在管子里流动，将会按顺序流过每个 pipe。<br/>
 * 最后当所有加载项都流出 pipeline 时，整个加载流程就结束了。
 * @class Pipeline
 */

/**
 * !#en
 * Constructor, pass an array of pipes to construct a new Pipeline,
 * the pipes will be chained in the given order.<br/>
 * A pipe is an object which must contain an `id` in string and a `handle` function,
 * the id must be unique in the pipeline.<br/>
 * It can also include `async` property to identify whether it's an asynchronous process.
 * !#zh
 * 构造函数，通过一系列的 pipe 来构造一个新的 pipeline，pipes 将会在给定的顺序中被锁定。<br/>
 * 一个 pipe 就是一个对象，它包含了字符串类型的 ‘id’ 和 ‘handle’ 函数，在 pipeline 中 id 必须是唯一的。<br/>
 * 它还可以包括 ‘async’ 属性以确定它是否是一个异步过程。
 *
 * @method constructor
 * @param {Array} pipes
 * @example
 *  var pipeline = new Pipeline([
 *      {
 *          id: 'Downloader',
 *          handle: function (item, callback) {},
 *          async: true
 *      },
 *      {id: 'Parser', handle: function (item) {}, async: false}
 *  ]);
 */


var Pipeline = function Pipeline(pipes) {
  this._pipes = pipes;
  this._cache = js.createMap(true);

  for (var i = 0; i < pipes.length; ++i) {
    var pipe = pipes[i]; // Must have handle and id, handle for flow, id for state flag

    if (!pipe.handle || !pipe.id) {
      continue;
    }

    pipe.pipeline = this;
    pipe.next = i < pipes.length - 1 ? pipes[i + 1] : null;
  }
};

Pipeline.ItemState = ItemState;
var proto = Pipeline.prototype;
/**
 * !#en
 * Insert a new pipe at the given index of the pipeline. <br/>
 * A pipe must contain an `id` in string and a `handle` function, the id must be unique in the pipeline.
 * !#zh
 * 在给定的索引位置插入一个新的 pipe。<br/>
 * 一个 pipe 必须包含一个字符串类型的 ‘id’ 和 ‘handle’ 函数，该 id 在 pipeline 必须是唯一标识。
 * @method insertPipe
 * @param {Object} pipe The pipe to be inserted
 * @param {Number} index The index to insert
 */

proto.insertPipe = function (pipe, index) {
  // Must have handle and id, handle for flow, id for state flag
  if (!pipe.handle || !pipe.id || index > this._pipes.length) {
    cc.warnID(4921);
    return;
  }

  if (this._pipes.indexOf(pipe) > 0) {
    cc.warnID(4922);
    return;
  }

  pipe.pipeline = this;
  var nextPipe = null;

  if (index < this._pipes.length) {
    nextPipe = this._pipes[index];
  }

  var previousPipe = null;

  if (index > 0) {
    previousPipe = this._pipes[index - 1];
  }

  if (previousPipe) {
    previousPipe.next = pipe;
  }

  pipe.next = nextPipe;

  this._pipes.splice(index, 0, pipe);
};
/**
 * !#en
 * Insert a pipe to the end of an existing pipe. The existing pipe must be a valid pipe in the pipeline.
 * !#zh
 * 在当前 pipeline 的一个已知 pipe 后面插入一个新的 pipe。
 * @method insertPipeAfter
 * @param {Object} refPipe An existing pipe in the pipeline.
 * @param {Object} newPipe The pipe to be inserted.
 */


proto.insertPipeAfter = function (refPipe, newPipe) {
  var index = this._pipes.indexOf(refPipe);

  if (index < 0) {
    return;
  }

  this.insertPipe(newPipe, index + 1);
};
/**
 * !#en
 * Add a new pipe at the end of the pipeline. <br/>
 * A pipe must contain an `id` in string and a `handle` function, the id must be unique in the pipeline.
 * !#zh
 * 添加一个新的 pipe 到 pipeline 尾部。 <br/>
 * 该 pipe 必须包含一个字符串类型 ‘id’ 和 ‘handle’ 函数，该 id 在 pipeline 必须是唯一标识。
 * @method appendPipe
 * @param {Object} pipe The pipe to be appended
 */


proto.appendPipe = function (pipe) {
  // Must have handle and id, handle for flow, id for state flag
  if (!pipe.handle || !pipe.id) {
    return;
  }

  pipe.pipeline = this;
  pipe.next = null;

  if (this._pipes.length > 0) {
    this._pipes[this._pipes.length - 1].next = pipe;
  }

  this._pipes.push(pipe);
};
/**
 * !#en
 * Let new items flow into the pipeline. <br/>
 * Each item can be a simple url string or an object,
 * if it's an object, it must contain `id` property. <br/>
 * You can specify its type by `type` property, by default, the type is the extension name in url. <br/>
 * By adding a `skips` property including pipe ids, you can skip these pipe. <br/>
 * The object can contain any supplementary property as you want. <br/>
 * !#zh
 * 让新的 item 流入 pipeline 中。<br/>
 * 这里的每个 item 可以是一个简单字符串类型的 url 或者是一个对象,
 * 如果它是一个对象的话，他必须要包含 ‘id’ 属性。<br/>
 * 你也可以指定它的 ‘type’ 属性类型，默认情况下，该类型是 ‘url’ 的后缀名。<br/>
 * 也通过添加一个 包含 ‘skips’ 属性的 item 对象，你就可以跳过 skips 中包含的 pipe。<br/>
 * 该对象可以包含任何附加属性。
 * @method flowIn
 * @param {Array} items
 * @example
 *  pipeline.flowIn([
 *      'res/Background.png',
 *      {
 *          id: 'res/scene.json',
 *          type: 'scene',
 *          name: 'scene',
 *          skips: ['Downloader']
 *      }
 *  ]);
 */


proto.flowIn = function (items) {
  var i,
      pipe = this._pipes[0],
      item;

  if (pipe) {
    // Cache all items first, in case synchronous loading flow same item repeatly
    for (i = 0; i < items.length; i++) {
      item = items[i];
      this._cache[item.id] = item;
    }

    for (i = 0; i < items.length; i++) {
      item = items[i];
      flow(pipe, item);
    }
  } else {
    for (i = 0; i < items.length; i++) {
      this.flowOut(items[i]);
    }
  }
};
/*
 * !#en
 * Let new items flow into the pipeline and give a callback when the list of items are all completed. <br/>
 * This is for loading dependencies for an existing item in flow, usually used in a pipe logic. <br/>
 * For example, we have a loader for scene configuration file in JSON, the scene will only be fully loaded  <br/>
 * after all its dependencies are loaded, then you will need to use function to flow in all dependencies  <br/>
 * found in the configuration file, and finish the loader pipe only after all dependencies are loaded (in the callback).
 * !#zh
 * 让新 items 流入 pipeline 并且当 item 列表完成时进行回调函数。<br/>
 * 这个 API 的使用通常是为了加载依赖项。<br/>
 * 例如：<br/>
 * 我们需要加载一个场景配置的 JSON 文件，该场景会将所有的依赖项全部都加载完毕以后，进行回调表示加载完毕。
 * @method flowInDeps
 * @deprecated since v1.3
 * @param {Array} urlList
 * @param {Function} callback
 * @return {Array} Items accepted by the pipeline
 */


proto.flowInDeps = function (owner, urlList, callback) {
  var deps = LoadingItems.create(this, function (errors, items) {
    callback(errors, items);
    items.destroy();
  });
  return deps.append(urlList, owner);
};

proto.flowOut = function (item) {
  if (item.error) {
    delete this._cache[item.id];
  } else if (!this._cache[item.id]) {
    this._cache[item.id] = item;
  }

  item.complete = true;
  LoadingItems.itemComplete(item);
};
/**
 * !#en
 * Copy the item states from one source item to all destination items. <br/>
 * It's quite useful when a pipe generate new items from one source item,<br/>
 * then you should flowIn these generated items into pipeline, <br/>
 * but you probably want them to skip all pipes the source item already go through,<br/>
 * you can achieve it with this API. <br/>
 * <br/>
 * For example, an unzip pipe will generate more items, but you won't want them to pass unzip or download pipe again.
 * !#zh
 * 从一个源 item 向所有目标 item 复制它的 pipe 状态，用于避免重复通过部分 pipe。<br/>
 * 当一个源 item 生成了一系列新的 items 时很有用，<br/>
 * 你希望让这些新的依赖项进入 pipeline，但是又不希望它们通过源 item 已经经过的 pipe，<br/>
 * 但是你可能希望他们源 item 已经通过并跳过所有 pipes，<br/>
 * 这个时候就可以使用这个 API。
 * @method copyItemStates
 * @param {Object} srcItem The source item
 * @param {Array|Object} dstItems A single destination item or an array of destination items
 */


proto.copyItemStates = function (srcItem, dstItems) {
  if (!(dstItems instanceof Array)) {
    dstItems.states = srcItem.states;
    return;
  }

  for (var i = 0; i < dstItems.length; ++i) {
    dstItems[i].states = srcItem.states;
  }
};
/**
 * !#en Returns an item in pipeline.
 * !#zh 根据 id 获取一个 item
 * @method getItem
 * @param {Object} id The id of the item
 * @return {Object}
 */


proto.getItem = function (id) {
  var item = this._cache[id];
  if (!item) return item; // downloader.js downloadUuid

  if (item.alias) item = item.alias;
  return item;
};
/**
 * !#en Removes an completed item in pipeline.
 * It will only remove the cache in the pipeline or loader, its dependencies won't be released.
 * cc.loader provided another method to completely cleanup the resource and its dependencies,
 * please refer to {{#crossLink "loader/release:method"}}cc.loader.release{{/crossLink}}
 * !#zh 移除指定的已完成 item。
 * 这将仅仅从 pipeline 或者 loader 中删除其缓存，并不会释放它所依赖的资源。
 * cc.loader 中提供了另一种删除资源及其依赖的清理方法，请参考 {{#crossLink "loader/release:method"}}cc.loader.release{{/crossLink}}
 * @method removeItem
 * @param {Object} id The id of the item
 * @return {Boolean} succeed or not
 */


proto.removeItem = function (id) {
  var removed = this._cache[id];

  if (removed && removed.complete) {
    delete this._cache[id];
  }

  return removed;
};
/**
 * !#en Clear the current pipeline, this function will clean up the items.
 * !#zh 清空当前 pipeline，该函数将清理 items。
 * @method clear
 */


proto.clear = function () {
  for (var id in this._cache) {
    var item = this._cache[id];
    delete this._cache[id];

    if (!item.complete) {
      item.error = new Error('Canceled manually');
      this.flowOut(item);
    }
  }
};

cc.Pipeline = module.exports = Pipeline;
                    }
                    if (nodeEnv) {
                        __define(__module.exports, __require, __module);
                    }
                    else {
                        __quick_compile_engine__.registerModuleFunc(__filename, function () {
                            __define(__module.exports, __require, __module);
                        });
                    }
                })();
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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