
                (function() {
                    var nodeEnv = typeof require !== 'undefined' && typeof process !== 'undefined';
                    var __module = nodeEnv ? module : {exports:{}};
                    var __filename = 'engine-dev/cocos2d/core/load-pipeline/loading-items.js';
                    var __require = nodeEnv ? function (request) {
                        return require(request);
                    } : function (request) {
                        return __quick_compile_engine__.require(request, __filename);
                    };
                    function __define (exports, require, module) {
                        if (!nodeEnv) {__quick_compile_engine__.registerModule(__filename, module);}"use strict";

/****************************************************************************
 Copyright (c) 2013-2016 Chukong Technologies Inc.
 Copyright (c) 2017-2018 Xiamen Yaji Software Co., Ltd.

 https://www.cocos.com/

 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated engine source code (the "Software"), a limited,
  worldwide, royalty-free, non-assignable, revocable and non-exclusive license
 to use Cocos Creator solely to develop games on your target platforms. You shall
  not use Cocos Creator software for developing other software or tools that's
  used for developing games. You are not granted to publish, distribute,
  sublicense, and/or sell copies of Cocos Creator.

 The software or tools in this License Agreement are licensed, not sold.
 Xiamen Yaji Software Co., Ltd. reserves all rights not expressly granted to you.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 THE SOFTWARE.
 ****************************************************************************/
var CallbacksInvoker = require('../platform/callbacks-invoker');

require('../utils/CCPath');

var js = require('../platform/js');

var _qid = 0 | Math.random() * 998;

var _queues = js.createMap(true);

var _pool = [];
var _POOL_MAX_LENGTH = 10;
var ItemState = {
  WORKING: 1,
  COMPLETE: 2,
  ERROR: 3
};

var _queueDeps = js.createMap(true);

function isIdValid(id) {
  var realId = id.url || id;
  return typeof realId === 'string';
}

function _parseUrlParam(url) {
  if (!url) return undefined;
  var split = url.split('?');

  if (!split || !split[0] || !split[1]) {
    return undefined;
  }

  var urlParam = {};
  var queries = split[1].split('&');
  queries.forEach(function (item) {
    var itemSplit = item.split('=');
    urlParam[itemSplit[0]] = itemSplit[1];
  });
  return urlParam;
}

function createItem(id, queueId) {
  var url = typeof id === 'object' ? id.url : id;
  var result = {
    queueId: queueId,
    id: url,
    url: url,
    // real download url, maybe changed
    rawUrl: undefined,
    // url used in scripts
    urlParam: _parseUrlParam(url),
    type: "",
    error: null,
    content: null,
    complete: false,
    states: {},
    deps: null
  };

  if (typeof id === 'object') {
    js.mixin(result, id);

    if (id.skips) {
      for (var i = 0; i < id.skips.length; i++) {
        var skip = id.skips[i];
        result.states[skip] = ItemState.COMPLETE;
      }
    }
  }

  result.rawUrl = result.url;

  if (url && !result.type) {
    result.type = cc.path.extname(url).toLowerCase().substr(1);
  }

  return result;
}

var checkedIds = [];

function checkCircleReference(owner, item, recursiveCall) {
  if (!owner || !item) {
    return false;
  }

  var result = false;
  checkedIds.push(item.id);

  if (item.deps) {
    var i,
        deps = item.deps,
        subDep;

    for (i = 0; i < deps.length; i++) {
      subDep = deps[i];

      if (subDep.id === owner.id) {
        result = true;
        break;
      } else if (checkedIds.indexOf(subDep.id) >= 0) {
        continue;
      } else if (subDep.deps && checkCircleReference(owner, subDep, true)) {
        result = true;
        break;
      }
    }
  }

  if (!recursiveCall) {
    checkedIds.length = 0;
  }

  return result;
}
/**
 * !#en
 * LoadingItems is the queue of items which can flow them into the loading pipeline.<br/>
 * Please don't construct it directly, use {{#crossLink "LoadingItems.create"}}cc.LoadingItems.create{{/crossLink}} instead, because we use an internal pool to recycle the queues.<br/>
 * It hold a map of items, each entry in the map is a url to object key value pair.<br/>
 * Each item always contains the following property:<br/>
 * - id: The identification of the item, usually it's identical to url<br/>
 * - url: The url <br/>
 * - type: The type, it's the extension name of the url by default, could be specified manually too.<br/>
 * - error: The error happened in pipeline will be stored in this property.<br/>
 * - content: The content processed by the pipeline, the final result will also be stored in this property.<br/>
 * - complete: The flag indicate whether the item is completed by the pipeline.<br/>
 * - states: An object stores the states of each pipe the item go through, the state can be: Pipeline.ItemState.WORKING | Pipeline.ItemState.ERROR | Pipeline.ItemState.COMPLETE<br/>
 * <br/>
 * Item can hold other custom properties.<br/>
 * Each LoadingItems object will be destroyed for recycle after onComplete callback<br/>
 * So please don't hold its reference for later usage, you can copy properties in it though.
 * !#zh
 * LoadingItems 是一个加载对象队列，可以用来输送加载对象到加载管线中。<br/>
 * 请不要直接使用 new 构造这个类的对象，你可以使用 {{#crossLink "LoadingItems.create"}}cc.LoadingItems.create{{/crossLink}} 来创建一个新的加载队列，这样可以允许我们的内部对象池回收并重利用加载队列。
 * 它有一个 map 属性用来存放加载项，在 map 对象中以 url 为 key 值。<br/>
 * 每个对象都会包含下列属性：<br/>
 * - id：该对象的标识，通常与 url 相同。<br/>
 * - url：路径 <br/>
 * - type: 类型，它这是默认的 URL 的扩展名，可以手动指定赋值。<br/>
 * - error：pipeline 中发生的错误将被保存在这个属性中。<br/>
 * - content: pipeline 中处理的临时结果，最终的结果也将被存储在这个属性中。<br/>
 * - complete：该标志表明该对象是否通过 pipeline 完成。<br/>
 * - states：该对象存储每个管道中对象经历的状态，状态可以是 Pipeline.ItemState.WORKING | Pipeline.ItemState.ERROR | Pipeline.ItemState.COMPLETE<br/>
 * <br/>
 * 对象可容纳其他自定义属性。<br/>
 * 每个 LoadingItems 对象都会在 onComplete 回调之后被销毁，所以请不要持有它的引用并在结束回调之后依赖它的内容执行任何逻辑，有这种需求的话你可以提前复制它的内容。
 *
 * @class LoadingItems
 * @extends CallbacksInvoker
 */


var LoadingItems = function LoadingItems(pipeline, urlList, onProgress, onComplete) {
  CallbacksInvoker.call(this);
  this._id = ++_qid;
  _queues[this._id] = this;
  this._pipeline = pipeline;
  this._errorUrls = js.createMap(true);
  this._appending = false;
  this._ownerQueue = null;
  /**
   * !#en This is a callback which will be invoked while an item flow out the pipeline.
   * You can pass the callback function in LoadingItems.create or set it later.
   * !#zh 这个回调函数将在 item 加载结束后被调用。你可以在构造时传递这个回调函数或者是在构造之后直接设置。
   * @method onProgress
   * @param {Number} completedCount The number of the items that are already completed.
   * @param {Number} totalCount The total number of the items.
   * @param {Object} item The latest item which flow out the pipeline.
   * @example
   *  loadingItems.onProgress = function (completedCount, totalCount, item) {
   *      var progress = (100 * completedCount / totalCount).toFixed(2);
   *      cc.log(progress + '%');
   *  }
   */

  this.onProgress = onProgress;
  /**
   * !#en This is a callback which will be invoked while all items is completed,
   * You can pass the callback function in LoadingItems.create or set it later.
   * !#zh 该函数将在加载队列全部完成时被调用。你可以在构造时传递这个回调函数或者是在构造之后直接设置。
   * @method onComplete
   * @param {Array} errors All errored urls will be stored in this array, if no error happened, then it will be null
   * @param {LoadingItems} items All items.
   * @example
   *  loadingItems.onComplete = function (errors, items) {
   *      if (error)
   *          cc.log('Completed with ' + errors.length + ' errors');
   *      else
   *          cc.log('Completed ' + items.totalCount + ' items');
   *  }
   */

  this.onComplete = onComplete;
  /**
   * !#en The map of all items.
   * !#zh 存储所有加载项的对象。
   * @property map
   * @type {Object}
   */

  this.map = js.createMap(true);
  /**
   * !#en The map of completed items.
   * !#zh 存储已经完成的加载项。
   * @property completed
   * @type {Object}
   */

  this.completed = {};
  /**
   * !#en Total count of all items.
   * !#zh 所有加载项的总数。
   * @property totalCount
   * @type {Number}
   */

  this.totalCount = 0;
  /**
   * !#en Total count of completed items.
   * !#zh 所有完成加载项的总数。
   * @property completedCount
   * @type {Number}
   */

  this.completedCount = 0;
  /**
   * !#en Activated or not.
   * !#zh 是否启用。
   * @property active
   * @type {Boolean}
   */

  if (this._pipeline) {
    this.active = true;
  } else {
    this.active = false;
  }

  if (urlList) {
    if (urlList.length > 0) {
      this.append(urlList);
    } else {
      this.allComplete();
    }
  }
};
/**
 * !#en The item states of the LoadingItems, its value could be LoadingItems.ItemState.WORKING | LoadingItems.ItemState.COMPLETET | LoadingItems.ItemState.ERROR
 * !#zh LoadingItems 队列中的加载项状态，状态的值可能是 LoadingItems.ItemState.WORKING | LoadingItems.ItemState.COMPLETET | LoadingItems.ItemState.ERROR
 * @enum LoadingItems.ItemState
 */

/**
 * @property {Number} WORKING
 */

/**
 * @property {Number} COMPLETET
 */

/**
 * @property {Number} ERROR
 */


LoadingItems.ItemState = new cc.Enum(ItemState);
/**
 * @class LoadingItems
 * @extends CallbacksInvoker
*/

/**
 * !#en The constructor function of LoadingItems, this will use recycled LoadingItems in the internal pool if possible.
 * You can pass onProgress and onComplete callbacks to visualize the loading process.
 * !#zh LoadingItems 的构造函数，这种构造方式会重用内部对象缓冲池中的 LoadingItems 队列，以尽量避免对象创建。
 * 你可以传递 onProgress 和 onComplete 回调函数来获知加载进度信息。
 * @method create
 * @static
 * @param {Pipeline} pipeline The pipeline to process the queue.
 * @param {Array} urlList The items array.
 * @param {Function} onProgress The progression callback, refer to {{#crossLink "LoadingItems.onProgress"}}{{/crossLink}}
 * @param {Function} onComplete The completion callback, refer to {{#crossLink "LoadingItems.onComplete"}}{{/crossLink}}
 * @return {LoadingItems} The LoadingItems queue object
 * @example
 *  cc.LoadingItems.create(cc.loader, ['a.png', 'b.plist'], function (completedCount, totalCount, item) {
 *      var progress = (100 * completedCount / totalCount).toFixed(2);
 *      cc.log(progress + '%');
 *  }, function (errors, items) {
 *      if (errors) {
 *          for (var i = 0; i < errors.length; ++i) {
 *              cc.log('Error url: ' + errors[i] + ', error: ' + items.getError(errors[i]));
 *          }
 *      }
 *      else {
 *          var result_a = items.getContent('a.png');
 *          // ...
 *      }
 *  })
 */

LoadingItems.create = function (pipeline, urlList, onProgress, onComplete) {
  if (onProgress === undefined) {
    if (typeof urlList === 'function') {
      onComplete = urlList;
      urlList = onProgress = null;
    }
  } else if (onComplete === undefined) {
    if (typeof urlList === 'function') {
      onComplete = onProgress;
      onProgress = urlList;
      urlList = null;
    } else {
      onComplete = onProgress;
      onProgress = null;
    }
  }

  var queue = _pool.pop();

  if (queue) {
    queue._pipeline = pipeline;
    queue.onProgress = onProgress;
    queue.onComplete = onComplete;
    _queues[queue._id] = queue;

    if (queue._pipeline) {
      queue.active = true;
    }

    if (urlList) {
      queue.append(urlList);
    }
  } else {
    queue = new LoadingItems(pipeline, urlList, onProgress, onComplete);
  }

  return queue;
};
/**
 * !#en Retrieve the LoadingItems queue object for an item.
 * !#zh 通过 item 对象获取它的 LoadingItems 队列。
 * @method getQueue
 * @static
 * @param {Object} item The item to query
 * @return {LoadingItems} The LoadingItems queue object
 */


LoadingItems.getQueue = function (item) {
  return item.queueId ? _queues[item.queueId] : null;
};
/**
 * !#en Complete an item in the LoadingItems queue, please do not call this method unless you know what's happening.
 * !#zh 通知 LoadingItems 队列一个 item 对象已完成，请不要调用这个函数，除非你知道自己在做什么。
 * @method itemComplete
 * @static
 * @param {Object} item The item which has completed
 */


LoadingItems.itemComplete = function (item) {
  var queue = _queues[item.queueId];

  if (queue) {
    // console.log('----- Completed by pipeline ' + item.id + ', rest: ' + (queue.totalCount - queue.completedCount-1));
    queue.itemComplete(item.id);
  }
};

LoadingItems.initQueueDeps = function (queue) {
  var dep = _queueDeps[queue._id];

  if (!dep) {
    dep = _queueDeps[queue._id] = {
      completed: [],
      deps: []
    };
  } else {
    dep.completed.length = 0;
    dep.deps.length = 0;
  }
};

LoadingItems.registerQueueDep = function (owner, depId) {
  var queueId = owner.queueId || owner;

  if (!queueId) {
    return false;
  }

  var queueDepList = _queueDeps[queueId]; // Owner is root queue

  if (queueDepList) {
    if (queueDepList.deps.indexOf(depId) === -1) {
      queueDepList.deps.push(depId);
    }
  } // Owner is an item in the intermediate queue
  else if (owner.id) {
      for (var id in _queueDeps) {
        var queue = _queueDeps[id]; // Found root queue

        if (queue.deps.indexOf(owner.id) !== -1) {
          if (queue.deps.indexOf(depId) === -1) {
            queue.deps.push(depId);
          }
        }
      }
    }
};

LoadingItems.finishDep = function (depId) {
  for (var id in _queueDeps) {
    var queue = _queueDeps[id]; // Found root queue

    if (queue.deps.indexOf(depId) !== -1 && queue.completed.indexOf(depId) === -1) {
      queue.completed.push(depId);
    }
  }
};

var proto = LoadingItems.prototype;
js.mixin(proto, CallbacksInvoker.prototype);
/**
 * !#en Add urls to the LoadingItems queue.
 * !#zh 向一个 LoadingItems 队列添加加载项。
 * @method append
 * @param {Array} urlList The url list to be appended, the url can be object or string
 * @return {Array} The accepted url list, some invalid items could be refused.
 */

proto.append = function (urlList, owner) {
  if (!this.active) {
    return [];
  }

  if (owner && !owner.deps) {
    owner.deps = [];
  }

  this._appending = true;
  var accepted = [],
      i,
      url,
      item;

  for (i = 0; i < urlList.length; ++i) {
    url = urlList[i]; // Already queued in another items queue, url is actually the item

    if (url.queueId && !this.map[url.id]) {
      this.map[url.id] = url; // Register item deps for circle reference check

      owner && owner.deps.push(url); // Queued and completed or Owner circle referenced by dependency

      if (url.complete || checkCircleReference(owner, url)) {
        this.totalCount++; // console.log('----- Completed already or circle referenced ' + url.id + ', rest: ' + (this.totalCount - this.completedCount-1));

        this.itemComplete(url.id);
        continue;
      } // Not completed yet, should wait it
      else {
          var self = this;
          var queue = _queues[url.queueId];

          if (queue) {
            this.totalCount++;
            LoadingItems.registerQueueDep(owner || this._id, url.id); // console.log('+++++ Waited ' + url.id);

            queue.addListener(url.id, function (item) {
              // console.log('----- Completed by waiting ' + item.id + ', rest: ' + (self.totalCount - self.completedCount-1));
              self.itemComplete(item.id);
            });
          }

          continue;
        }
    } // Queue new items


    if (isIdValid(url)) {
      item = createItem(url, this._id);
      var key = item.id; // No duplicated url

      if (!this.map[key]) {
        this.map[key] = item;
        this.totalCount++; // Register item deps for circle reference check

        owner && owner.deps.push(item);
        LoadingItems.registerQueueDep(owner || this._id, key);
        accepted.push(item); // console.log('+++++ Appended ' + item.id);
      }
    }
  }

  this._appending = false; // Manually complete

  if (this.completedCount === this.totalCount) {
    // console.log('===== All Completed ');
    this.allComplete();
  } else {
    this._pipeline.flowIn(accepted);
  }

  return accepted;
};

proto._childOnProgress = function (item) {
  if (this.onProgress) {
    var dep = _queueDeps[this._id];
    this.onProgress(dep ? dep.completed.length : this.completedCount, dep ? dep.deps.length : this.totalCount, item);
  }
};
/**
 * !#en Complete a LoadingItems queue, please do not call this method unless you know what's happening.
 * !#zh 完成一个 LoadingItems 队列，请不要调用这个函数，除非你知道自己在做什么。
 * @method allComplete
 */


proto.allComplete = function () {
  var errors = js.isEmptyObject(this._errorUrls) ? null : this._errorUrls;

  if (this.onComplete) {
    this.onComplete(errors, this);
  }
};
/**
 * !#en Check whether all items are completed.
 * !#zh 检查是否所有加载项都已经完成。
 * @method isCompleted
 * @return {Boolean}
 */


proto.isCompleted = function () {
  return this.completedCount >= this.totalCount;
};
/**
 * !#en Check whether an item is completed.
 * !#zh 通过 id 检查指定加载项是否已经加载完成。
 * @method isItemCompleted
 * @param {String} id The item's id.
 * @return {Boolean}
 */


proto.isItemCompleted = function (id) {
  return !!this.completed[id];
};
/**
 * !#en Check whether an item exists.
 * !#zh 通过 id 检查加载项是否存在。
 * @method exists
 * @param {String} id The item's id.
 * @return {Boolean}
 */


proto.exists = function (id) {
  return !!this.map[id];
};
/**
 * !#en Returns the content of an internal item.
 * !#zh 通过 id 获取指定对象的内容。
 * @method getContent
 * @param {String} id The item's id.
 * @return {Object}
 */


proto.getContent = function (id) {
  var item = this.map[id];
  var ret = null;

  if (item) {
    if (item.content) {
      ret = item.content;
    } else if (item.alias) {
      ret = item.alias.content;
    }
  }

  return ret;
};
/**
 * !#en Returns the error of an internal item.
 * !#zh 通过 id 获取指定对象的错误信息。
 * @method getError
 * @param {String} id The item's id.
 * @return {Object}
 */


proto.getError = function (id) {
  var item = this.map[id];
  var ret = null;

  if (item) {
    if (item.error) {
      ret = item.error;
    } else if (item.alias) {
      ret = item.alias.error;
    }
  }

  return ret;
};
/**
 * !#en Add a listener for an item, the callback will be invoked when the item is completed.
 * !#zh 监听加载项（通过 key 指定）的完成事件。
 * @method addListener
 * @param {String} key
 * @param {Function} callback - can be null
 * @param {Object} target - can be null
 * @return {Boolean} whether the key is new
 */


proto.addListener = CallbacksInvoker.prototype.on;
/**
 * !#en
 * Check if the specified key has any registered callback. 
 * If a callback is also specified, it will only return true if the callback is registered.
 * !#zh
 * 检查指定的加载项是否有完成事件监听器。
 * 如果同时还指定了一个回调方法，并且回调有注册，它只会返回 true。
 * @method hasListener
 * @param {String} key
 * @param {Function} [callback]
 * @param {Object} [target]
 * @return {Boolean}
 */

proto.hasListener = CallbacksInvoker.prototype.hasEventListener;
/**
 * !#en
 * Removes a listener. 
 * It will only remove when key, callback, target all match correctly.
 * !#zh
 * 移除指定加载项已经注册的完成事件监听器。
 * 只会删除 key, callback, target 均匹配的监听器。
 * @method remove
 * @param {String} key
 * @param {Function} callback
 * @param {Object} target
 * @return {Boolean} removed
 */

proto.removeListener = CallbacksInvoker.prototype.off;
/**
 * !#en
 * Removes all callbacks registered in a certain event
 * type or all callbacks registered with a certain target.
 * !#zh 删除指定目标的所有完成事件监听器。
 * @method removeAllListeners
 * @param {String|Object} key - The event key to be removed or the target to be removed
 */

proto.removeAllListeners = CallbacksInvoker.prototype.removeAll;
/**
 * !#en Remove an item, can only remove completed item, ongoing item can not be removed.
 * !#zh 移除加载项，这里只会移除已经完成的加载项，正在进行的加载项将不能被删除。
 * @param {String} url
 */

proto.removeItem = function (url) {
  var item = this.map[url];
  if (!item) return;
  if (!this.completed[item.alias || url]) return;
  delete this.completed[url];
  delete this.map[url];

  if (item.alias) {
    delete this.completed[item.alias.id];
    delete this.map[item.alias.id];
  }

  this.completedCount--;
  this.totalCount--;
};
/**
 * !#en Complete an item in the LoadingItems queue, please do not call this method unless you know what's happening.
 * !#zh 通知 LoadingItems 队列一个 item 对象已完成，请不要调用这个函数，除非你知道自己在做什么。
 * @method itemComplete
 * @param {String} id The item url
 */


proto.itemComplete = function (id) {
  var item = this.map[id];

  if (!item) {
    return;
  } // Register or unregister errors


  var errorListId = id in this._errorUrls;

  if (item.error instanceof Error || js.isString(item.error)) {
    this._errorUrls[id] = item.error;
  } else if (item.error) {
    js.mixin(this._errorUrls, item.error);
  } else if (!item.error && errorListId) {
    delete this._errorUrls[id];
  }

  this.completed[id] = item;
  this.completedCount++;
  LoadingItems.finishDep(item.id);

  if (this.onProgress) {
    var dep = _queueDeps[this._id];
    this.onProgress(dep ? dep.completed.length : this.completedCount, dep ? dep.deps.length : this.totalCount, item);
  }

  this.emit(id, item);
  this.removeAll(id); // All completed

  if (!this._appending && this.completedCount >= this.totalCount) {
    // console.log('===== All Completed ');
    this.allComplete();
  }
};
/**
 * !#en Destroy the LoadingItems queue, the queue object won't be garbage collected, it will be recycled, so every after destroy is not reliable.
 * !#zh 销毁一个 LoadingItems 队列，这个队列对象会被内部缓冲池回收，所以销毁后的所有内部信息都是不可依赖的。
 * @method destroy
 */


proto.destroy = function () {
  this.active = false;
  this._appending = false;
  this._pipeline = null;
  this._ownerQueue = null;
  js.clear(this._errorUrls);
  this.onProgress = null;
  this.onComplete = null;
  this.map = js.createMap(true);
  this.completed = {};
  this.totalCount = 0;
  this.completedCount = 0; // Reinitialize CallbacksInvoker, generate three new objects, could be improved

  CallbacksInvoker.call(this);

  if (_queueDeps[this._id]) {
    _queueDeps[this._id].completed.length = 0;
    _queueDeps[this._id].deps.length = 0;
  }

  delete _queues[this._id];
  delete _queueDeps[this._id];

  if (_pool.indexOf(this) === -1 && _pool.length < _POOL_MAX_LENGTH) {
    _pool.push(this);
  }
};

cc.LoadingItems = module.exports = LoadingItems;
                    }
                    if (nodeEnv) {
                        __define(__module.exports, __require, __module);
                    }
                    else {
                        __quick_compile_engine__.registerModuleFunc(__filename, function () {
                            __define(__module.exports, __require, __module);
                        });
                    }
                })();
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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