
                (function() {
                    var nodeEnv = typeof require !== 'undefined' && typeof process !== 'undefined';
                    var __module = nodeEnv ? module : {exports:{}};
                    var __filename = 'engine-dev/cocos2d/core/load-pipeline/CCLoader.js';
                    var __require = nodeEnv ? function (request) {
                        return require(request);
                    } : function (request) {
                        return __quick_compile_engine__.require(request, __filename);
                    };
                    function __define (exports, require, module) {
                        if (!nodeEnv) {__quick_compile_engine__.registerModule(__filename, module);}"use strict";

/****************************************************************************
 Copyright (c) 2013-2016 Chukong Technologies Inc.
 Copyright (c) 2017-2018 Xiamen Yaji Software Co., Ltd.

 https://www.cocos.com/

 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated engine source code (the "Software"), a limited,
  worldwide, royalty-free, non-assignable, revocable and non-exclusive license
 to use Cocos Creator solely to develop games on your target platforms. You shall
  not use Cocos Creator software for developing other software or tools that's
  used for developing games. You are not granted to publish, distribute,
  sublicense, and/or sell copies of Cocos Creator.

 The software or tools in this License Agreement are licensed, not sold.
 Xiamen Yaji Software Co., Ltd. reserves all rights not expressly granted to you.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 THE SOFTWARE.
 ****************************************************************************/
var js = require('../platform/js');

var Pipeline = require('./pipeline');

var LoadingItems = require('./loading-items');

var AssetLoader = require('./asset-loader');

var Downloader = require('./downloader');

var Loader = require('./loader');

var AssetTable = require('./asset-table');

var callInNextTick = require('../platform/utils').callInNextTick;

var AutoReleaseUtils = require('./auto-release-utils'); // var pushToMap = require('../utils/misc').pushToMap;


var ReleasedAssetChecker = CC_DEBUG && require('./released-asset-checker');

var assetTables = Object.create(null);
assetTables.assets = new AssetTable();
assetTables.internal = new AssetTable();

function getXMLHttpRequest() {
  return window.XMLHttpRequest ? new window.XMLHttpRequest() : new ActiveXObject('MSXML2.XMLHTTP');
}

var _info = {
  url: null,
  raw: false
}; // Convert a resources by finding its real url with uuid, otherwise we will use the uuid or raw url as its url
// So we gurantee there will be url in result

function getResWithUrl(res) {
  var id, result, isUuid;

  if (typeof res === 'object') {
    result = res;

    if (res.url) {
      return result;
    } else {
      id = res.uuid;
    }
  } else {
    result = {};
    id = res;
  }

  isUuid = result.type ? result.type === 'uuid' : cc.AssetLibrary._uuidInSettings(id);

  cc.AssetLibrary._getAssetInfoInRuntime(id, _info);

  result.url = !isUuid ? id : _info.url;

  if (_info.url && result.type === 'uuid' && _info.raw) {
    result.type = null;
    result.isRawAsset = true;
  } else if (!isUuid) {
    result.isRawAsset = true;
  }

  return result;
}

var _sharedResources = [];
var _sharedList = [];
/**
 * Loader for resource loading process. It's a singleton object.
 * @class loader
 * @extends Pipeline
 * @static
 */

function CCLoader() {
  var assetLoader = new AssetLoader();
  var downloader = new Downloader();
  var loader = new Loader();
  Pipeline.call(this, [assetLoader, downloader, loader]);
  /**
   * The asset loader in cc.loader's pipeline, it's by default the first pipe.
   * It's used to identify an asset's type, and determine how to download it.
   * @property assetLoader
   * @type {Object}
   */

  this.assetLoader = assetLoader;
  /**
   * The md5 pipe in cc.loader's pipeline, it could be absent if the project isn't build with md5 option.
   * It's used to modify the url to the real downloadable url with md5 suffix.
   * @property md5Pipe
   * @type {Object}
   */

  this.md5Pipe = null;
  /**
   * The downloader in cc.loader's pipeline, it's by default the second pipe.
   * It's used to download files with several handlers: pure text, image, script, audio, font, uuid.
   * You can add your own download function with addDownloadHandlers
   * @property downloader
   * @type {Object}
   */

  this.downloader = downloader;
  /**
   * The loader in cc.loader's pipeline, it's by default the third pipe.
   * It's used to parse downloaded content with several handlers: JSON, image, plist, fnt, uuid.
   * You can add your own download function with addLoadHandlers
   * @property loader
   * @type {Object}
   */

  this.loader = loader;
  this.onProgress = null; // assets to release automatically

  this._autoReleaseSetting = js.createMap(true);

  if (CC_DEBUG) {
    this._releasedAssetChecker_DEBUG = new ReleasedAssetChecker();
  }
}

js.extend(CCLoader, Pipeline);
var proto = CCLoader.prototype;

proto.init = function (director) {
  if (CC_DEBUG) {
    var self = this;
    director.on(cc.Director.EVENT_AFTER_UPDATE, function () {
      self._releasedAssetChecker_DEBUG.checkCouldRelease(self._cache);
    });
  }
};
/**
 * Gets a new XMLHttpRequest instance.
 * @method getXMLHttpRequest
 * @returns {XMLHttpRequest}
 */


proto.getXMLHttpRequest = getXMLHttpRequest;
/**
 * Add custom supported types handler or modify existing type handler for download process.
 * @example
 *  cc.loader.addDownloadHandlers({
 *      // This will match all url with `.scene` extension or all url with `scene` type
 *      'scene' : function (url, callback) {}
 *  });
 * @method addDownloadHandlers
 * @param {Object} extMap Custom supported types with corresponded handler
 */

proto.addDownloadHandlers = function (extMap) {
  this.downloader.addHandlers(extMap);
};
/**
 * Add custom supported types handler or modify existing type handler for load process.
 * @example
 *  cc.loader.addLoadHandlers({
 *      // This will match all url with `.scene` extension or all url with `scene` type
 *      'scene' : function (url, callback) {}
 *  });
 * @method addLoadHandlers
 * @param {Object} extMap Custom supported types with corresponded handler
 */


proto.addLoadHandlers = function (extMap) {
  this.loader.addHandlers(extMap);
};
/**
 * Load resources with a progression callback and a complete callback.
 * The progression callback is the same as Pipeline's {{#crossLink "LoadingItems/onProgress:method"}}onProgress{{/crossLink}}
 * The complete callback is almost the same as Pipeline's {{#crossLink "LoadingItems/onComplete:method"}}onComplete{{/crossLink}}
 * The only difference is when user pass a single url as resources, the complete callback will set its result directly as the second parameter.
 *
 * @example
 * cc.loader.load('a.png', function (err, tex) {
 *     cc.log('Result should be a texture: ' + (tex instanceof cc.Texture2D));
 * });
 *
 * cc.loader.load('http://example.com/a.png', function (err, tex) {
 *     cc.log('Should load a texture from external url: ' + (tex instanceof cc.Texture2D));
 * });
 *
 * cc.loader.load({url: 'http://example.com/getImageREST?file=a.png', type: 'png'}, function (err, tex) {
 *     cc.log('Should load a texture from RESTful API by specify the type: ' + (tex instanceof cc.Texture2D));
 * });
 *
 * cc.loader.load(['a.png', 'b.json'], function (errors, results) {
 *     if (errors) {
 *         for (var i = 0; i < errors.length; i++) {
 *             cc.log('Error url [' + errors[i] + ']: ' + results.getError(errors[i]));
 *         }
 *     }
 *     var aTex = results.getContent('a.png');
 *     var bJsonObj = results.getContent('b.json');
 * });
 *
 * @method load
 * @param {String|String[]|Object} resources - Url list in an array
 * @param {Function} [progressCallback] - Callback invoked when progression change
 * @param {Number} progressCallback.completedCount - The number of the items that are already completed
 * @param {Number} progressCallback.totalCount - The total number of the items
 * @param {Object} progressCallback.item - The latest item which flow out the pipeline
 * @param {Function} [completeCallback] - Callback invoked when all resources loaded
 * @typescript
 * load(resources: string|string[]|{uuid?: string, url?: string, type?: string}, completeCallback?: Function): void
 * load(resources: string|string[]|{uuid?: string, url?: string, type?: string}, progressCallback: (completedCount: number, totalCount: number, item: any) => void, completeCallback: Function|null): void
 */


proto.load = function (resources, progressCallback, completeCallback) {
  if (CC_DEV && !resources) {
    return cc.error("[cc.loader.load] resources must be non-nil.");
  }

  if (completeCallback === undefined) {
    completeCallback = progressCallback;
    progressCallback = this.onProgress || null;
  }

  var self = this;
  var singleRes = false;
  var res;

  if (!(resources instanceof Array)) {
    if (resources) {
      singleRes = true;
      resources = [resources];
    } else {
      resources = [];
    }
  }

  _sharedResources.length = 0;

  for (var i = 0; i < resources.length; ++i) {
    var resource = resources[i]; // Backward compatibility

    if (resource && resource.id) {
      cc.warnID(4920, resource.id);

      if (!resource.uuid && !resource.url) {
        resource.url = resource.id;
      }
    }

    res = getResWithUrl(resource);
    if (!res.url && !res.uuid) continue;
    var item = this._cache[res.url];

    _sharedResources.push(item || res);
  }

  var queue = LoadingItems.create(this, progressCallback, function (errors, items) {
    callInNextTick(function () {
      if (completeCallback) {
        if (singleRes) {
          var id = res.url;
          completeCallback.call(self, errors, items.getContent(id));
        } else {
          completeCallback.call(self, errors, items);
        }

        completeCallback = null;
      }

      if (CC_EDITOR) {
        for (var _id in self._cache) {
          if (self._cache[_id].complete) {
            self.removeItem(_id);
          }
        }
      }

      items.destroy();
    });
  });
  LoadingItems.initQueueDeps(queue);
  queue.append(_sharedResources);
  _sharedResources.length = 0;
};

proto.flowInDeps = function (owner, urlList, callback) {
  _sharedList.length = 0;

  for (var i = 0; i < urlList.length; ++i) {
    var res = getResWithUrl(urlList[i]);
    if (!res.url && !res.uuid) continue;
    var item = this._cache[res.url];

    if (item) {
      _sharedList.push(item);
    } else {
      _sharedList.push(res);
    }
  }

  var queue = LoadingItems.create(this, owner ? function (completedCount, totalCount, item) {
    if (this._ownerQueue && this._ownerQueue.onProgress) {
      this._ownerQueue._childOnProgress(item);
    }
  } : null, function (errors, items) {
    callback(errors, items); // Clear deps because it's already done
    // Each item will only flowInDeps once, so it's still safe here

    owner && owner.deps && (owner.deps.length = 0);
    items.destroy();
  });

  if (owner) {
    var ownerQueue = LoadingItems.getQueue(owner); // Set the root ownerQueue, if no ownerQueue defined in ownerQueue, it's the root

    queue._ownerQueue = ownerQueue._ownerQueue || ownerQueue;
  }

  var accepted = queue.append(_sharedList, owner);
  _sharedList.length = 0;
  return accepted;
};

proto._assetTables = assetTables;

proto._getResUuid = function (url, type, mount, quiet) {
  mount = mount || 'assets';
  var assetTable = assetTables[mount];

  if (!url || !assetTable) {
    return null;
  } // Ignore parameter


  var index = url.indexOf('?');
  if (index !== -1) url = url.substr(0, index);
  var uuid = assetTable.getUuid(url, type);

  if (!uuid) {
    var extname = cc.path.extname(url);

    if (extname) {
      // strip extname
      url = url.slice(0, -extname.length);
      uuid = assetTable.getUuid(url, type);

      if (uuid && !quiet) {
        cc.warnID(4901, url, extname);
      }
    }
  }

  return uuid;
}; // Find the asset's reference id in loader, asset could be asset object, asset uuid or asset url


proto._getReferenceKey = function (assetOrUrlOrUuid) {
  var key;

  if (typeof assetOrUrlOrUuid === 'object') {
    key = assetOrUrlOrUuid._uuid || null;
  } else if (typeof assetOrUrlOrUuid === 'string') {
    key = this._getResUuid(assetOrUrlOrUuid, null, null, true) || assetOrUrlOrUuid;
  }

  if (!key) {
    cc.warnID(4800, assetOrUrlOrUuid);
    return key;
  }

  cc.AssetLibrary._getAssetInfoInRuntime(key, _info);

  return this._cache[_info.url] ? _info.url : key;
};

proto._urlNotFound = function (url, type, completeCallback) {
  callInNextTick(function () {
    url = cc.url.normalize(url);
    var info = (type ? js.getClassName(type) : 'Asset') + " in \"resources/" + url + "\" does not exist.";

    if (completeCallback) {
      completeCallback(new Error(info), []);
    }
  });
};
/**
 * @param {Function} [type]
 * @param {Function} [onProgress]
 * @param {Function} onComplete
 * @returns {Object} arguments
 * @returns {Function} arguments.type
 * @returns {Function} arguments.onProgress
 * @returns {Function} arguments.onComplete
 */


proto._parseLoadResArgs = function (type, onProgress, onComplete) {
  if (onComplete === undefined) {
    var isValidType = type instanceof Array || js.isChildClassOf(type, cc.RawAsset);

    if (onProgress) {
      onComplete = onProgress;

      if (isValidType) {
        onProgress = this.onProgress || null;
      }
    } else if (onProgress === undefined && !isValidType) {
      onComplete = type;
      onProgress = this.onProgress || null;
      type = null;
    }

    if (onProgress !== undefined && !isValidType) {
      onProgress = type;
      type = null;
    }
  }

  return {
    type: type,
    onProgress: onProgress,
    onComplete: onComplete
  };
};
/**
 * Load resources from the "resources" folder inside the "assets" folder of your project.<br>
 * <br>
 * Note: All asset URLs in Creator use forward slashes, URLs using backslashes will not work.
 *
 * @method loadRes
 * @param {String} url - Url of the target resource.
 *                       The url is relative to the "resources" folder, extensions must be omitted.
 * @param {Function} [type] - Only asset of type will be loaded if this argument is supplied.
 * @param {Function} [progressCallback] - Callback invoked when progression change.
 * @param {Number} progressCallback.completedCount - The number of the items that are already completed.
 * @param {Number} progressCallback.totalCount - The total number of the items.
 * @param {Object} progressCallback.item - The latest item which flow out the pipeline.
 * @param {Function} [completeCallback] - Callback invoked when the resource loaded.
 * @param {Error} completeCallback.error - The error info or null if loaded successfully.
 * @param {Object} completeCallback.resource - The loaded resource if it can be found otherwise returns null.
 *
 * @example
 *
 * // load the prefab (project/assets/resources/misc/character/cocos) from resources folder
 * cc.loader.loadRes('misc/character/cocos', function (err, prefab) {
 *     if (err) {
 *         cc.error(err.message || err);
 *         return;
 *     }
 *     cc.log('Result should be a prefab: ' + (prefab instanceof cc.Prefab));
 * });
 *
 * // load the sprite frame of (project/assets/resources/imgs/cocos.png) from resources folder
 * cc.loader.loadRes('imgs/cocos', cc.SpriteFrame, function (err, spriteFrame) {
 *     if (err) {
 *         cc.error(err.message || err);
 *         return;
 *     }
 *     cc.log('Result should be a sprite frame: ' + (spriteFrame instanceof cc.SpriteFrame));
 * });
 * @typescript
 * loadRes(url: string, type: typeof cc.Asset, progressCallback: (completedCount: number, totalCount: number, item: any) => void, completeCallback: ((error: Error, resource: any) => void)|null): void
 * loadRes(url: string, type: typeof cc.Asset, completeCallback: (error: Error, resource: any) => void): void
 * loadRes(url: string, type: typeof cc.Asset): void
 * loadRes(url: string, progressCallback: (completedCount: number, totalCount: number, item: any) => void, completeCallback: ((error: Error, resource: any) => void)|null): void
 * loadRes(url: string, completeCallback: (error: Error, resource: any) => void): void
 * loadRes(url: string): void
 */


proto.loadRes = function (url, type, mount, progressCallback, completeCallback) {
  if (arguments.length !== 5) {
    completeCallback = progressCallback;
    progressCallback = mount;
    mount = 'assets';
  }

  var args = this._parseLoadResArgs(type, progressCallback, completeCallback);

  type = args.type;
  progressCallback = args.onProgress;
  completeCallback = args.onComplete;
  var self = this;

  var uuid = self._getResUuid(url, type, mount);

  if (uuid) {
    this.load({
      type: 'uuid',
      uuid: uuid
    }, progressCallback, function (err, asset) {
      if (asset) {
        // should not release these assets, even if they are static referenced in the scene.
        self.setAutoReleaseRecursively(uuid, false);
      }

      if (completeCallback) {
        completeCallback(err, asset);
      }
    });
  } else {
    self._urlNotFound(url, type, completeCallback);
  }
};

proto._loadResUuids = function (uuids, progressCallback, completeCallback, urls) {
  if (uuids.length > 0) {
    var self = this;
    var res = uuids.map(function (uuid) {
      return {
        type: 'uuid',
        uuid: uuid
      };
    });
    this.load(res, progressCallback, function (errors, items) {
      if (completeCallback) {
        var assetRes = [];
        var urlRes = urls && [];

        for (var i = 0; i < res.length; ++i) {
          var uuid = res[i].uuid;

          var id = this._getReferenceKey(uuid);

          var item = items.getContent(id);

          if (item) {
            // should not release these assets, even if they are static referenced in the scene.
            self.setAutoReleaseRecursively(uuid, false);
            assetRes.push(item);

            if (urlRes) {
              urlRes.push(urls[i]);
            }
          }
        }

        if (urls) {
          completeCallback(errors, assetRes, urlRes);
        } else {
          completeCallback(errors, assetRes);
        }
      }
    });
  } else {
    if (completeCallback) {
      callInNextTick(function () {
        if (urls) {
          completeCallback(null, [], []);
        } else {
          completeCallback(null, []);
        }
      });
    }
  }
};
/**
 * This method is like {{#crossLink "loader/loadRes:method"}}{{/crossLink}} except that it accepts array of url.
 *
 * @method loadResArray
 * @param {String[]} urls - Array of URLs of the target resource.
 *                          The url is relative to the "resources" folder, extensions must be omitted.
 * @param {Function} [type] - Only asset of type will be loaded if this argument is supplied.
 * @param {Function} [progressCallback] - Callback invoked when progression change.
 * @param {Number} progressCallback.completedCount - The number of the items that are already completed.
 * @param {Number} progressCallback.totalCount - The total number of the items.
 * @param {Object} progressCallback.item - The latest item which flow out the pipeline.
 * @param {Function} [completeCallback] - A callback which is called when all assets have been loaded, or an error occurs.
 * @param {Error} completeCallback.error - If one of the asset failed, the complete callback is immediately called
 *                                         with the error. If all assets are loaded successfully, error will be null.
 * @param {Asset[]|Array} completeCallback.assets - An array of all loaded assets.
 *                                                     If nothing to load, assets will be an empty array.
 * @example
 *
 * // load the SpriteFrames from resources folder
 * var spriteFrames;
 * var urls = ['misc/characters/character_01', 'misc/weapons/weapons_01'];
 * cc.loader.loadResArray(urls, cc.SpriteFrame, function (err, assets) {
 *     if (err) {
 *         cc.error(err);
 *         return;
 *     }
 *     spriteFrames = assets;
 *     // ...
 * });
 * @typescript
 * loadResArray(url: string[], type: typeof cc.Asset, progressCallback: (completedCount: number, totalCount: number, item: any) => void, completeCallback: ((error: Error, resource: any[]) => void)|null): void
 * loadResArray(url: string[], type: typeof cc.Asset, completeCallback: (error: Error, resource: any[]) => void): void
 * loadResArray(url: string[], type: typeof cc.Asset): void
 * loadResArray(url: string[], progressCallback: (completedCount: number, totalCount: number, item: any) => void, completeCallback: ((error: Error, resource: any[]) => void)|null): void
 * loadResArray(url: string[], completeCallback: (error: Error, resource: any[]) => void): void
 * loadResArray(url: string[]): void
 * loadResArray(url: string[], type: typeof cc.Asset[]): void
 */


proto.loadResArray = function (urls, type, mount, progressCallback, completeCallback) {
  if (arguments.length !== 5) {
    completeCallback = progressCallback;
    progressCallback = mount;
    mount = 'assets';
  }

  var args = this._parseLoadResArgs(type, progressCallback, completeCallback);

  type = args.type;
  progressCallback = args.onProgress;
  completeCallback = args.onComplete;
  var uuids = [];
  var isTypesArray = type instanceof Array;

  for (var i = 0; i < urls.length; i++) {
    var url = urls[i];
    var assetType = isTypesArray ? type[i] : type;

    var uuid = this._getResUuid(url, assetType, mount);

    if (uuid) {
      uuids.push(uuid);
    } else {
      this._urlNotFound(url, assetType, completeCallback);

      return;
    }
  }

  this._loadResUuids(uuids, progressCallback, completeCallback);
};
/**
 * Load all assets in a folder inside the "assets/resources" folder of your project.<br>
 * <br>
 * Note: All asset URLs in Creator use forward slashes, URLs using backslashes will not work.
 *
 * @method loadResDir
 * @param {String} url - Url of the target folder.
 *                       The url is relative to the "resources" folder, extensions must be omitted.
 * @param {Function} [type] - Only asset of type will be loaded if this argument is supplied.
 * @param {Function} [progressCallback] - Callback invoked when progression change.
 * @param {Number} progressCallback.completedCount - The number of the items that are already completed.
 * @param {Number} progressCallback.totalCount - The total number of the items.
 * @param {Object} progressCallback.item - The latest item which flow out the pipeline.
 * @param {Function} [completeCallback] - A callback which is called when all assets have been loaded, or an error occurs.
 * @param {Error} completeCallback.error - If one of the asset failed, the complete callback is immediately called
 *                                         with the error. If all assets are loaded successfully, error will be null.
 * @param {Asset[]|Array} completeCallback.assets - An array of all loaded assets.
 *                                             If nothing to load, assets will be an empty array.
 * @param {String[]} completeCallback.urls - An array that lists all the URLs of loaded assets.
 *
 * @example
 *
 * // load the texture (resources/imgs/cocos.png) and the corresponding sprite frame
 * cc.loader.loadResDir('imgs/cocos', function (err, assets) {
 *     if (err) {
 *         cc.error(err);
 *         return;
 *     }
 *     var texture = assets[0];
 *     var spriteFrame = assets[1];
 * });
 *
 * // load all textures in "resources/imgs/"
 * cc.loader.loadResDir('imgs', cc.Texture2D, function (err, textures) {
 *     var texture1 = textures[0];
 *     var texture2 = textures[1];
 * });
 *
 * // load all JSONs in "resources/data/"
 * cc.loader.loadResDir('data', function (err, objects, urls) {
 *     var data = objects[0];
 *     var url = urls[0];
 * });
 * @typescript
 * loadResDir(url: string, type: typeof cc.Asset, progressCallback: (completedCount: number, totalCount: number, item: any) => void, completeCallback: ((error: Error, resource: any[], urls: string[]) => void)|null): void
 * loadResDir(url: string, type: typeof cc.Asset, completeCallback: (error: Error, resource: any[], urls: string[]) => void): void
 * loadResDir(url: string, type: typeof cc.Asset): void
 * loadResDir(url: string, progressCallback: (completedCount: number, totalCount: number, item: any) => void, completeCallback: ((error: Error, resource: any[], urls: string[]) => void)|null): void
 * loadResDir(url: string, completeCallback: (error: Error, resource: any[], urls: string[]) => void): void
 * loadResDir(url: string): void
 */


proto.loadResDir = function (url, type, mount, progressCallback, completeCallback) {
  if (arguments.length !== 5) {
    completeCallback = progressCallback;
    progressCallback = mount;
    mount = 'assets';
  }

  if (!assetTables[mount]) return;

  var args = this._parseLoadResArgs(type, progressCallback, completeCallback);

  type = args.type;
  progressCallback = args.onProgress;
  completeCallback = args.onComplete;
  var urls = [];
  var uuids = assetTables[mount].getUuidArray(url, type, urls);

  this._loadResUuids(uuids, progressCallback, completeCallback, urls);
};
/**
 * Get resource data by id. <br>
 * When you load resources with {{#crossLink "loader/load:method"}}{{/crossLink}} or {{#crossLink "loader/loadRes:method"}}{{/crossLink}},
 * the url will be the unique identity of the resource.
 * After loaded, you can acquire them by passing the url to this API.
 *
 * @method getRes
 * @param {String} url
 * @param {Function} [type] - Only asset of type will be returned if this argument is supplied.
 * @returns {*}
 */


proto.getRes = function (url, type) {
  var item = this._cache[url];

  if (!item) {
    var uuid = this._getResUuid(url, type, null, true);

    if (uuid) {
      var ref = this._getReferenceKey(uuid);

      item = this._cache[ref];
    } else {
      return null;
    }
  }

  if (item && item.alias) {
    item = item.alias;
  }

  return item && item.complete ? item.content : null;
};
/**
 * Get total resources count in loader.
 * @returns {Number}
 */


proto.getResCount = function () {
  return Object.keys(this._cache).length;
};
/**
 * !#en
 * Get all resource dependencies of the loaded asset in an array, including itself.
 * The owner parameter accept the following types: 1. The asset itself; 2. The resource url; 3. The asset's uuid.<br>
 * The returned array stores the dependencies with their uuids, after retrieve dependencies,
 * you can release them, access dependent assets by passing the uuid to {{#crossLink "loader/getRes:method"}}{{/crossLink}}, or other stuffs you want.<br>
 * For release all dependencies of an asset, please refer to {{#crossLink "loader/release:method"}}{{/crossLink}}
 * Here is some examples:
 * !#zh
 * 获取某个已经加载好的资源的所有依赖资源，包含它自身，并保存在数组中返回。owner 参数接收以下几种类型：1. 资源 asset 对象；2. 资源目录下的 url；3. 资源的 uuid。<br>
 * 返回的数组将仅保存依赖资源的 uuid，获取这些 uuid 后，你可以从 loader 释放这些资源；通过 {{#crossLink "loader/getRes:method"}}{{/crossLink}} 获取某个资源或者进行其他你需要的操作。<br>
 * 想要释放一个资源及其依赖资源，可以参考 {{#crossLink "loader/release:method"}}{{/crossLink}}。下面是一些示例代码：
 *
 * @example
 * // Release all dependencies of a loaded prefab
 * var deps = cc.loader.getDependsRecursively(prefab);
 * cc.loader.release(deps);
 * // Retrieve all dependent textures
 * var deps = cc.loader.getDependsRecursively('prefabs/sample');
 * var textures = [];
 * for (var i = 0; i < deps.length; ++i) {
 *     var item = cc.loader.getRes(deps[i]);
 *     if (item instanceof cc.Texture2D) {
 *         textures.push(item);
 *     }
 * }
 *
 * @method getDependsRecursively
 * @param {Asset|RawAsset|String} owner - The owner asset or the resource url or the asset's uuid
 * @returns {Array}
 */


proto.getDependsRecursively = function (owner) {
  if (owner) {
    var key = this._getReferenceKey(owner);

    var assets = AutoReleaseUtils.getDependsRecursively(key);
    assets.push(key);
    return assets;
  } else {
    return [];
  }
};
/**
 * !#en
 * Release the content of an asset or an array of assets by uuid.
 * Start from v1.3, this method will not only remove the cache of the asset in loader, but also clean up its content.
 * For example, if you release a texture, the texture asset and its gl texture data will be freed up.
 * In complexe project, you can use this function with {{#crossLink "loader/getDependsRecursively:method"}}{{/crossLink}} to free up memory in critical circumstances.
 * Notice, this method may cause the texture to be unusable, if there are still other nodes use the same texture, they may turn to black and report gl errors.
 * If you only want to remove the cache of an asset, please use {{#crossLink "pipeline/removeItem:method"}}{{/crossLink}}
 * !#zh
 * 通过 id（通常是资源 url）来释放一个资源或者一个资源数组。
 * 从 v1.3 开始，这个方法不仅会从 loader 中删除资源的缓存引用，还会清理它的资源内容。
 * 比如说，当你释放一个 texture 资源，这个 texture 和它的 gl 贴图数据都会被释放。
 * 在复杂项目中，我们建议你结合 {{#crossLink "loader/getDependsRecursively:method"}}{{/crossLink}} 来使用，便于在设备内存告急的情况下更快地释放不再需要的资源的内存。
 * 注意，这个函数可能会导致资源贴图或资源所依赖的贴图不可用，如果场景中存在节点仍然依赖同样的贴图，它们可能会变黑并报 GL 错误。
 * 如果你只想删除一个资源的缓存引用，请使用 {{#crossLink "pipeline/removeItem:method"}}{{/crossLink}}
 *
 * @example
 * // Release a texture which is no longer need
 * cc.loader.release(texture);
 * // Release all dependencies of a loaded prefab
 * var deps = cc.loader.getDependsRecursively('prefabs/sample');
 * cc.loader.release(deps);
 * // If there is no instance of this prefab in the scene, the prefab and its dependencies like textures, sprite frames, etc, will be freed up.
 * // If you have some other nodes share a texture in this prefab, you can skip it in two ways:
 * // 1. Forbid auto release a texture before release
 * cc.loader.setAutoRelease(texture2d, false);
 * // 2. Remove it from the dependencies array
 * var deps = cc.loader.getDependsRecursively('prefabs/sample');
 * var index = deps.indexOf(texture2d._uuid);
 * if (index !== -1)
 *     deps.splice(index, 1);
 * cc.loader.release(deps);
 *
 * @method release
 * @param {Asset|RawAsset|String|Array} asset
 */


proto.release = function (asset) {
  if (Array.isArray(asset)) {
    for (var i = 0; i < asset.length; i++) {
      var key = asset[i];
      this.release(key);
    }
  } else if (asset) {
    var id = this._getReferenceKey(asset);

    if (!CC_EDITOR && id && id in cc.AssetLibrary.getBuiltinDeps()) return;
    var item = this.getItem(id);

    if (item) {
      var removed = this.removeItem(id);
      asset = item.content;

      if (CC_DEBUG && removed) {
        this._releasedAssetChecker_DEBUG.setReleased(item, id);
      }
    }

    if (asset instanceof cc.Asset) {
      var nativeUrl = asset.nativeUrl;

      if (nativeUrl) {
        this.release(nativeUrl); // uncache loading item of native asset
      }

      asset.destroy();
    }
  }
};
/**
 * !#en Release the asset by its object. Refer to {{#crossLink "loader/release:method"}}{{/crossLink}} for detailed informations.
 * !#zh 通过资源对象自身来释放资源。详细信息请参考 {{#crossLink "loader/release:method"}}{{/crossLink}}
 *
 * @method releaseAsset
 * @param {Asset} asset
 */


proto.releaseAsset = function (asset) {
  var uuid = asset._uuid;

  if (uuid) {
    this.release(uuid);
  }
};
/**
 * !#en Release the asset loaded by {{#crossLink "loader/loadRes:method"}}{{/crossLink}}. Refer to {{#crossLink "loader/release:method"}}{{/crossLink}} for detailed informations.
 * !#zh 释放通过 {{#crossLink "loader/loadRes:method"}}{{/crossLink}} 加载的资源。详细信息请参考 {{#crossLink "loader/release:method"}}{{/crossLink}}
 *
 * @method releaseRes
 * @param {String} url
 * @param {Function} [type] - Only asset of type will be released if this argument is supplied.
 */


proto.releaseRes = function (url, type, mount) {
  var uuid = this._getResUuid(url, type, mount);

  if (uuid) {
    this.release(uuid);
  } else {
    cc.errorID(4914, url);
  }
};
/**
 * !#en Release the all assets loaded by {{#crossLink "loader/loadResDir:method"}}{{/crossLink}}. Refer to {{#crossLink "loader/release:method"}}{{/crossLink}} for detailed informations.
 * !#zh 释放通过 {{#crossLink "loader/loadResDir:method"}}{{/crossLink}} 加载的资源。详细信息请参考 {{#crossLink "loader/release:method"}}{{/crossLink}}
 *
 * @method releaseResDir
 * @param {String} url
 * @param {Function} [type] - Only asset of type will be released if this argument is supplied.
 */


proto.releaseResDir = function (url, type, mount) {
  mount = mount || 'assets';
  if (!assetTables[mount]) return;
  var uuids = assetTables[mount].getUuidArray(url, type);

  for (var i = 0; i < uuids.length; i++) {
    var uuid = uuids[i];
    this.release(uuid);
  }
};
/**
 * !#en Resource all assets. Refer to {{#crossLink "loader/release:method"}}{{/crossLink}} for detailed informations.
 * !#zh 释放所有资源。详细信息请参考 {{#crossLink "loader/release:method"}}{{/crossLink}}
 *
 * @method releaseAll
 */


proto.releaseAll = function () {
  for (var id in this._cache) {
    this.release(id);
  }
}; // AUTO RELEASE
// override


proto.removeItem = function (key) {
  var removed = Pipeline.prototype.removeItem.call(this, key);
  delete this._autoReleaseSetting[key];
  return removed;
};
/**
 * !#en
 * Indicates whether to release the asset when loading a new scene.<br>
 * By default, when loading a new scene, all assets in the previous scene will be released or preserved
 * according to whether the previous scene checked the "Auto Release Assets" option.
 * On the other hand, assets dynamically loaded by using `cc.loader.loadRes` or `cc.loader.loadResDir`
 * will not be affected by that option, remain not released by default.<br>
 * Use this API to change the default behavior on a single asset, to force preserve or release specified asset when scene switching.<br>
 * <br>
 * See: {{#crossLink "loader/setAutoReleaseRecursively:method"}}cc.loader.setAutoReleaseRecursively{{/crossLink}}, {{#crossLink "loader/isAutoRelease:method"}}cc.loader.isAutoRelease{{/crossLink}}
 * !#zh
 * 设置当场景切换时是否自动释放资源。<br>
 * 默认情况下，当加载新场景时，旧场景的资源根据旧场景是否勾选“Auto Release Assets”，将会被释放或者保留。
 * 而使用 `cc.loader.loadRes` 或 `cc.loader.loadResDir` 动态加载的资源，则不受场景设置的影响，默认不自动释放。<br>
 * 使用这个 API 可以在单个资源上改变这个默认行为，强制在切换场景时保留或者释放指定资源。<br>
 * <br>
 * 参考：{{#crossLink "loader/setAutoReleaseRecursively:method"}}cc.loader.setAutoReleaseRecursively{{/crossLink}}，{{#crossLink "loader/isAutoRelease:method"}}cc.loader.isAutoRelease{{/crossLink}}
 *
 * @example
 * // auto release the texture event if "Auto Release Assets" disabled in current scene
 * cc.loader.setAutoRelease(texture2d, true);
 * // don't release the texture even if "Auto Release Assets" enabled in current scene
 * cc.loader.setAutoRelease(texture2d, false);
 * // first parameter can be url
 * cc.loader.setAutoRelease(audioUrl, false);
 *
 * @method setAutoRelease
 * @param {Asset|String} assetOrUrlOrUuid - asset object or the raw asset's url or uuid
 * @param {Boolean} autoRelease - indicates whether should release automatically
 */


proto.setAutoRelease = function (assetOrUrlOrUuid, autoRelease) {
  var key = this._getReferenceKey(assetOrUrlOrUuid);

  if (key) {
    this._autoReleaseSetting[key] = !!autoRelease;
  } else if (CC_DEV) {
    cc.warnID(4902);
  }
};
/**
 * !#en
 * Indicates whether to release the asset and its referenced other assets when loading a new scene.<br>
 * By default, when loading a new scene, all assets in the previous scene will be released or preserved
 * according to whether the previous scene checked the "Auto Release Assets" option.
 * On the other hand, assets dynamically loaded by using `cc.loader.loadRes` or `cc.loader.loadResDir`
 * will not be affected by that option, remain not released by default.<br>
 * Use this API to change the default behavior on the specified asset and its recursively referenced assets, to force preserve or release specified asset when scene switching.<br>
 * <br>
 * See: {{#crossLink "loader/setAutoRelease:method"}}cc.loader.setAutoRelease{{/crossLink}}, {{#crossLink "loader/isAutoRelease:method"}}cc.loader.isAutoRelease{{/crossLink}}
 * !#zh
 * 设置当场景切换时是否自动释放资源及资源引用的其它资源。<br>
 * 默认情况下，当加载新场景时，旧场景的资源根据旧场景是否勾选“Auto Release Assets”，将会被释放或者保留。
 * 而使用 `cc.loader.loadRes` 或 `cc.loader.loadResDir` 动态加载的资源，则不受场景设置的影响，默认不自动释放。<br>
 * 使用这个 API 可以在指定资源及资源递归引用到的所有资源上改变这个默认行为，强制在切换场景时保留或者释放指定资源。<br>
 * <br>
 * 参考：{{#crossLink "loader/setAutoRelease:method"}}cc.loader.setAutoRelease{{/crossLink}}，{{#crossLink "loader/isAutoRelease:method"}}cc.loader.isAutoRelease{{/crossLink}}
 *
 * @example
 * // auto release the SpriteFrame and its Texture event if "Auto Release Assets" disabled in current scene
 * cc.loader.setAutoReleaseRecursively(spriteFrame, true);
 * // don't release the SpriteFrame and its Texture even if "Auto Release Assets" enabled in current scene
 * cc.loader.setAutoReleaseRecursively(spriteFrame, false);
 * // don't release the Prefab and all the referenced assets
 * cc.loader.setAutoReleaseRecursively(prefab, false);
 *
 * @method setAutoReleaseRecursively
 * @param {Asset|String} assetOrUrlOrUuid - asset object or the raw asset's url or uuid
 * @param {Boolean} autoRelease - indicates whether should release automatically
 */


proto.setAutoReleaseRecursively = function (assetOrUrlOrUuid, autoRelease) {
  autoRelease = !!autoRelease;

  var key = this._getReferenceKey(assetOrUrlOrUuid);

  if (key) {
    this._autoReleaseSetting[key] = autoRelease;
    var depends = AutoReleaseUtils.getDependsRecursively(key);

    for (var i = 0; i < depends.length; i++) {
      var depend = depends[i];
      this._autoReleaseSetting[depend] = autoRelease;
    }
  } else if (CC_DEV) {
    cc.warnID(4902);
  }
};
/**
 * !#en
 * Returns whether the asset is configured as auto released, despite how "Auto Release Assets" property is set on scene asset.<br>
 * <br>
 * See: {{#crossLink "loader/setAutoRelease:method"}}cc.loader.setAutoRelease{{/crossLink}}, {{#crossLink "loader/setAutoReleaseRecursively:method"}}cc.loader.setAutoReleaseRecursively{{/crossLink}}
 *
 * !#zh
 * 返回指定的资源是否有被设置为自动释放，不论场景的“Auto Release Assets”如何设置。<br>
 * <br>
 * 参考：{{#crossLink "loader/setAutoRelease:method"}}cc.loader.setAutoRelease{{/crossLink}}，{{#crossLink "loader/setAutoReleaseRecursively:method"}}cc.loader.setAutoReleaseRecursively{{/crossLink}}
 * @method isAutoRelease
 * @param {Asset|String} assetOrUrl - asset object or the raw asset's url
 * @returns {Boolean}
 */


proto.isAutoRelease = function (assetOrUrl) {
  var key = this._getReferenceKey(assetOrUrl);

  if (key) {
    return !!this._autoReleaseSetting[key];
  }

  return false;
};

cc.loader = new CCLoader();

if (CC_EDITOR) {
  cc.loader.refreshUrl = function (uuid, oldUrl, newUrl) {
    var item = this._cache[uuid];

    if (item) {
      item.url = newUrl;
    }

    item = this._cache[oldUrl];

    if (item) {
      item.id = newUrl;
      item.url = newUrl;
      this._cache[newUrl] = item;
      delete this._cache[oldUrl];
    }
  };
}

module.exports = cc.loader;
                    }
                    if (nodeEnv) {
                        __define(__module.exports, __require, __module);
                    }
                    else {
                        __quick_compile_engine__.registerModuleFunc(__filename, function () {
                            __define(__module.exports, __require, __module);
                        });
                    }
                })();
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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