
                (function() {
                    var nodeEnv = typeof require !== 'undefined' && typeof process !== 'undefined';
                    var __module = nodeEnv ? module : {exports:{}};
                    var __filename = 'engine-dev/cocos2d/core/components/CCRichText.js';
                    var __require = nodeEnv ? function (request) {
                        return require(request);
                    } : function (request) {
                        return __quick_compile_engine__.require(request, __filename);
                    };
                    function __define (exports, require, module) {
                        if (!nodeEnv) {__quick_compile_engine__.registerModule(__filename, module);}"use strict";

/****************************************************************************
 Copyright (c) 2013-2016 Chukong Technologies Inc.
 Copyright (c) 2017-2018 Xiamen Yaji Software Co., Ltd.

 https://www.cocos.com/

 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated engine source code (the "Software"), a limited,
  worldwide, royalty-free, non-assignable, revocable and non-exclusive license
 to use Cocos Creator solely to develop games on your target platforms. You shall
  not use Cocos Creator software for developing other software or tools that's
  used for developing games. You are not granted to publish, distribute,
  sublicense, and/or sell copies of Cocos Creator.

 The software or tools in this License Agreement are licensed, not sold.
 Xiamen Yaji Software Co., Ltd. reserves all rights not expressly granted to you.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 THE SOFTWARE.
 ****************************************************************************/
var js = require('../platform/js');

var macro = require('../platform/CCMacro');

var textUtils = require('../utils/text-utils');

var HtmlTextParser = require('../utils/html-text-parser');

var _htmlTextParser = new HtmlTextParser();

var HorizontalAlign = macro.TextAlignment;
var VerticalAlign = macro.VerticalTextAlignment;
var RichTextChildName = "RICHTEXT_CHILD";
var RichTextChildImageName = "RICHTEXT_Image_CHILD";
var CacheMode = cc.Label.CacheMode; // Returns a function, that, as long as it continues to be invoked, will not
// be triggered. The function will be called after it stops being called for
// N milliseconds. If `immediate` is passed, trigger the function on the
// leading edge, instead of the trailing.

function debounce(func, wait, immediate) {
  var timeout;
  return function () {
    var context = this;

    var later = function later() {
      timeout = null;
      if (!immediate) func.apply(context, arguments);
    };

    var callNow = immediate && !timeout;
    clearTimeout(timeout);
    timeout = setTimeout(later, wait);
    if (callNow) func.apply(context, arguments);
  };
}
/**
 * RichText pool
 */


var pool = new js.Pool(function (node) {
  if (CC_EDITOR) {
    cc.isValid(node) && node.destroy();
    return false;
  }

  if (CC_DEV) {
    cc.assert(!node._parent, 'Recycling node\'s parent should be null!');
  }

  if (!cc.isValid(node)) {
    return false;
  } else {
    var outline = node.getComponent(cc.LabelOutline);

    if (outline) {
      outline.width = 0;
    }
  }

  return true;
}, 20);

pool.get = function (string, richtext) {
  var labelNode = this._get();

  if (!labelNode) {
    labelNode = new cc.PrivateNode(RichTextChildName);
  }

  labelNode.setPosition(0, 0);
  labelNode.setAnchorPoint(0.5, 0.5);
  labelNode.skewX = 0;
  var labelComponent = labelNode.getComponent(cc.Label);

  if (!labelComponent) {
    labelComponent = labelNode.addComponent(cc.Label);
  }

  labelComponent.string = "";
  labelComponent.horizontalAlign = HorizontalAlign.LEFT;
  labelComponent.verticalAlign = VerticalAlign.CENTER;
  return labelNode;
};
/**
 * !#en The RichText Component.
 * !#zh 富文本组件
 * @class RichText
 * @extends Component
 */


var RichText = cc.Class({
  name: 'cc.RichText',
  "extends": cc.Component,
  ctor: function ctor() {
    this._textArray = null;
    this._labelSegments = [];
    this._labelSegmentsCache = [];
    this._linesWidth = [];

    if (CC_EDITOR) {
      this._userDefinedFont = null;
      this._updateRichTextStatus = debounce(this._updateRichText, 200);
    } else {
      this._updateRichTextStatus = this._updateRichText;
    }
  },
  editor: CC_EDITOR && {
    menu: 'i18n:MAIN_MENU.component.renderers/RichText',
    help: 'i18n:COMPONENT.help_url.richtext',
    inspector: 'packages://inspector/inspectors/comps/richtext.js',
    executeInEditMode: true
  },
  properties: {
    /**
     * !#en Content string of RichText.
     * !#zh 富文本显示的文本内容。
     * @property {String} string
     */
    string: {
      "default": '<color=#00ff00>Rich</c><color=#0fffff>Text</color>',
      multiline: true,
      tooltip: CC_DEV && 'i18n:COMPONENT.richtext.string',
      notify: function notify() {
        this._updateRichTextStatus();
      }
    },

    /**
     * !#en Horizontal Alignment of each line in RichText.
     * !#zh 文本内容的水平对齐方式。
     * @property {macro.TextAlignment} horizontalAlign
     */
    horizontalAlign: {
      "default": HorizontalAlign.LEFT,
      type: HorizontalAlign,
      tooltip: CC_DEV && 'i18n:COMPONENT.richtext.horizontal_align',
      animatable: false,
      notify: function notify(oldValue) {
        if (this.horizontalAlign === oldValue) return;
        this._layoutDirty = true;

        this._updateRichTextStatus();
      }
    },

    /**
     * !#en Font size of RichText.
     * !#zh 富文本字体大小。
     * @property {Number} fontSize
     */
    fontSize: {
      "default": 40,
      tooltip: CC_DEV && 'i18n:COMPONENT.richtext.font_size',
      notify: function notify(oldValue) {
        if (this.fontSize === oldValue) return;
        this._layoutDirty = true;

        this._updateRichTextStatus();
      }
    },

    /**
     * !#en Custom System font of RichText
     * !#zh 富文本定制系统字体
     * @property {String} fontFamily
     */
    _fontFamily: "Arial",
    fontFamily: {
      tooltip: CC_DEV && 'i18n:COMPONENT.richtext.font_family',
      get: function get() {
        return this._fontFamily;
      },
      set: function set(value) {
        if (this._fontFamily === value) return;
        this._fontFamily = value;
        this._layoutDirty = true;

        this._updateRichTextStatus();
      },
      animatable: false
    },

    /**
     * !#en Custom TTF font of RichText
     * !#zh  富文本定制字体
     * @property {cc.TTFFont} font
     */
    font: {
      "default": null,
      type: cc.TTFFont,
      tooltip: CC_DEV && 'i18n:COMPONENT.richtext.font',
      notify: function notify(oldValue) {
        if (this.font === oldValue) return;
        this._layoutDirty = true;

        if (this.font) {
          if (CC_EDITOR) {
            this._userDefinedFont = this.font;
          }

          this.useSystemFont = false;

          this._onTTFLoaded();
        } else {
          this.useSystemFont = true;
        }

        this._updateRichTextStatus();
      }
    },

    /**
     * !#en Whether use system font name or not.
     * !#zh 是否使用系统字体。
     * @property {Boolean} useSystemFont
     */
    _isSystemFontUsed: true,
    useSystemFont: {
      get: function get() {
        return this._isSystemFontUsed;
      },
      set: function set(value) {
        if (this._isSystemFontUsed === value) {
          return;
        }

        this._isSystemFontUsed = value;

        if (CC_EDITOR) {
          if (value) {
            this.font = null;
          } else if (this._userDefinedFont) {
            this.font = this._userDefinedFont;
            return;
          }
        }

        this._layoutDirty = true;

        this._updateRichTextStatus();
      },
      animatable: false,
      tooltip: CC_DEV && 'i18n:COMPONENT.richtext.system_font'
    },

    /**
     * !#en The cache mode of label. This mode only supports system fonts.
     * !#zh 文本缓存模式, 该模式只支持系统字体。
     * @property {Label.CacheMode} cacheMode
     */
    cacheMode: {
      "default": CacheMode.NONE,
      type: CacheMode,
      tooltip: CC_DEV && 'i18n:COMPONENT.label.cacheMode',
      notify: function notify(oldValue) {
        if (this.cacheMode === oldValue) return;

        this._updateRichTextStatus();
      },
      animatable: false
    },

    /**
     * !#en The maximize width of the RichText
     * !#zh 富文本的最大宽度
     * @property {Number} maxWidth
     */
    maxWidth: {
      "default": 0,
      tooltip: CC_DEV && 'i18n:COMPONENT.richtext.max_width',
      notify: function notify(oldValue) {
        if (this.maxWidth === oldValue) return;
        this._layoutDirty = true;

        this._updateRichTextStatus();
      }
    },

    /**
     * !#en Line Height of RichText.
     * !#zh 富文本行高。
     * @property {Number} lineHeight
     */
    lineHeight: {
      "default": 40,
      tooltip: CC_DEV && 'i18n:COMPONENT.richtext.line_height',
      notify: function notify(oldValue) {
        if (this.lineHeight === oldValue) return;
        this._layoutDirty = true;

        this._updateRichTextStatus();
      }
    },

    /**
     * !#en The image atlas for the img tag. For each src value in the img tag, there should be a valid spriteFrame in the image atlas.
     * !#zh 对于 img 标签里面的 src 属性名称，都需要在 imageAtlas 里面找到一个有效的 spriteFrame，否则 img tag 会判定为无效。
     * @property {SpriteAtlas} imageAtlas
     */
    imageAtlas: {
      "default": null,
      type: cc.SpriteAtlas,
      tooltip: CC_DEV && 'i18n:COMPONENT.richtext.image_atlas',
      notify: function notify(oldValue) {
        if (this.imageAtlas === oldValue) return;
        this._layoutDirty = true;

        this._updateRichTextStatus();
      }
    },

    /**
     * !#en
     * Once checked, the RichText will block all input events (mouse and touch) within
     * the bounding box of the node, preventing the input from penetrating into the underlying node.
     * !#zh
     * 选中此选项后，RichText 将阻止节点边界框中的所有输入事件（鼠标和触摸），从而防止输入事件穿透到底层节点。
     * @property {Boolean} handleTouchEvent
     * @default true
     */
    handleTouchEvent: {
      "default": true,
      tooltip: CC_DEV && 'i18n:COMPONENT.richtext.handleTouchEvent',
      notify: function notify(oldValue) {
        if (this.handleTouchEvent === oldValue) return;

        if (this.enabledInHierarchy) {
          this.handleTouchEvent ? this._addEventListeners() : this._removeEventListeners();
        }
      }
    }
  },
  statics: {
    HorizontalAlign: HorizontalAlign,
    VerticalAlign: VerticalAlign
  },
  onEnable: function onEnable() {
    if (this.handleTouchEvent) {
      this._addEventListeners();
    }

    this._updateRichText();

    this._activateChildren(true);
  },
  onDisable: function onDisable() {
    if (this.handleTouchEvent) {
      this._removeEventListeners();
    }

    this._activateChildren(false);
  },
  start: function start() {
    this._onTTFLoaded();
  },
  _onColorChanged: function _onColorChanged(parentColor) {
    var children = this.node.children;
    children.forEach(function (childNode) {
      childNode.color = parentColor;
    });
  },
  _addEventListeners: function _addEventListeners() {
    this.node.on(cc.Node.EventType.TOUCH_END, this._onTouchEnded, this);
    this.node.on(cc.Node.EventType.COLOR_CHANGED, this._onColorChanged, this);
  },
  _removeEventListeners: function _removeEventListeners() {
    this.node.off(cc.Node.EventType.TOUCH_END, this._onTouchEnded, this);
    this.node.off(cc.Node.EventType.COLOR_CHANGED, this._onColorChanged, this);
  },
  _updateLabelSegmentTextAttributes: function _updateLabelSegmentTextAttributes() {
    this._labelSegments.forEach(function (item) {
      this._applyTextAttribute(item, null, true);
    }.bind(this));
  },
  _createFontLabel: function _createFontLabel(string) {
    return pool.get(string, this);
  },
  _onTTFLoaded: function _onTTFLoaded() {
    if (this.font instanceof cc.TTFFont) {
      if (this.font._nativeAsset) {
        this._layoutDirty = true;

        this._updateRichText();
      } else {
        var self = this;
        cc.loader.load(this.font.nativeUrl, function (err, fontFamily) {
          self._layoutDirty = true;

          self._updateRichText();
        });
      }
    } else {
      this._layoutDirty = true;

      this._updateRichText();
    }
  },
  _measureText: function _measureText(styleIndex, string) {
    var self = this;

    var func = function func(string) {
      var label;

      if (self._labelSegmentsCache.length === 0) {
        label = self._createFontLabel(string);

        self._labelSegmentsCache.push(label);
      } else {
        label = self._labelSegmentsCache[0];
      }

      label._styleIndex = styleIndex;

      self._applyTextAttribute(label, string, true);

      var labelSize = label.getContentSize();
      return labelSize.width;
    };

    if (string) {
      return func(string);
    } else {
      return func;
    }
  },
  _onTouchEnded: function _onTouchEnded(event) {
    var _this = this;

    var components = this.node.getComponents(cc.Component);

    var _loop = function _loop(i) {
      var labelSegment = _this._labelSegments[i];
      var clickHandler = labelSegment._clickHandler;
      var clickParam = labelSegment._clickParam;

      if (clickHandler && _this._containsTouchLocation(labelSegment, event.touch.getLocation())) {
        components.forEach(function (component) {
          if (component.enabledInHierarchy && component[clickHandler]) {
            component[clickHandler](event, clickParam);
          }
        });
        event.stopPropagation();
      }
    };

    for (var i = 0; i < this._labelSegments.length; ++i) {
      _loop(i);
    }
  },
  _containsTouchLocation: function _containsTouchLocation(label, point) {
    var myRect = label.getBoundingBoxToWorld();
    return myRect.contains(point);
  },
  _resetState: function _resetState() {
    var children = this.node.children;

    for (var i = children.length - 1; i >= 0; i--) {
      var child = children[i];

      if (child.name === RichTextChildName || child.name === RichTextChildImageName) {
        if (child.parent === this.node) {
          child.parent = null;
        } else {
          // In case child.parent !== this.node, child cannot be removed from children
          children.splice(i, 1);
        }

        if (child.name === RichTextChildName) {
          pool.put(child);
        }
      }
    }

    this._labelSegments.length = 0;
    this._labelSegmentsCache.length = 0;
    this._linesWidth.length = 0;
    this._lineOffsetX = 0;
    this._lineCount = 1;
    this._labelWidth = 0;
    this._labelHeight = 0;
    this._layoutDirty = true;
  },
  onRestore: CC_EDITOR && function () {
    // TODO: refine undo/redo system
    // Because undo/redo will not call onEnable/onDisable,
    // we need call onEnable/onDisable manually to active/disactive children nodes.
    if (this.enabledInHierarchy) {
      this.onEnable();
    } else {
      this.onDisable();
    }
  },
  _activateChildren: function _activateChildren(active) {
    for (var i = this.node.children.length - 1; i >= 0; i--) {
      var child = this.node.children[i];

      if (child.name === RichTextChildName || child.name === RichTextChildImageName) {
        child.active = active;
      }
    }
  },
  _addLabelSegment: function _addLabelSegment(stringToken, styleIndex) {
    var labelSegment;

    if (this._labelSegmentsCache.length === 0) {
      labelSegment = this._createFontLabel(stringToken);
    } else {
      labelSegment = this._labelSegmentsCache.pop();
    }

    labelSegment._styleIndex = styleIndex;
    labelSegment._lineCount = this._lineCount;
    labelSegment.active = this.node.active;
    labelSegment.setAnchorPoint(0, 0);

    this._applyTextAttribute(labelSegment, stringToken);

    this.node.addChild(labelSegment);

    this._labelSegments.push(labelSegment);

    return labelSegment;
  },
  _updateRichTextWithMaxWidth: function _updateRichTextWithMaxWidth(labelString, labelWidth, styleIndex) {
    var fragmentWidth = labelWidth;
    var labelSegment;

    if (this._lineOffsetX > 0 && fragmentWidth + this._lineOffsetX > this.maxWidth) {
      //concat previous line
      var checkStartIndex = 0;

      while (this._lineOffsetX <= this.maxWidth) {
        var checkEndIndex = this._getFirstWordLen(labelString, checkStartIndex, labelString.length);

        var checkString = labelString.substr(checkStartIndex, checkEndIndex);

        var checkStringWidth = this._measureText(styleIndex, checkString);

        if (this._lineOffsetX + checkStringWidth <= this.maxWidth) {
          this._lineOffsetX += checkStringWidth;
          checkStartIndex += checkEndIndex;
        } else {
          if (checkStartIndex > 0) {
            var remainingString = labelString.substr(0, checkStartIndex);

            this._addLabelSegment(remainingString, styleIndex);

            labelString = labelString.substr(checkStartIndex, labelString.length);
            fragmentWidth = this._measureText(styleIndex, labelString);
          }

          this._updateLineInfo();

          break;
        }
      }
    }

    if (fragmentWidth > this.maxWidth) {
      var fragments = textUtils.fragmentText(labelString, fragmentWidth, this.maxWidth, this._measureText(styleIndex));

      for (var k = 0; k < fragments.length; ++k) {
        var splitString = fragments[k];
        labelSegment = this._addLabelSegment(splitString, styleIndex);
        var labelSize = labelSegment.getContentSize();
        this._lineOffsetX += labelSize.width;

        if (fragments.length > 1 && k < fragments.length - 1) {
          this._updateLineInfo();
        }
      }
    } else {
      this._lineOffsetX += fragmentWidth;

      this._addLabelSegment(labelString, styleIndex);
    }
  },
  _isLastComponentCR: function _isLastComponentCR(stringToken) {
    return stringToken.length - 1 === stringToken.lastIndexOf("\n");
  },
  _updateLineInfo: function _updateLineInfo() {
    this._linesWidth.push(this._lineOffsetX);

    this._lineOffsetX = 0;
    this._lineCount++;
  },
  _needsUpdateTextLayout: function _needsUpdateTextLayout(newTextArray) {
    if (this._layoutDirty || !this._textArray || !newTextArray) {
      return true;
    }

    if (this._textArray.length !== newTextArray.length) {
      return true;
    }

    for (var i = 0; i < this._textArray.length; ++i) {
      var oldItem = this._textArray[i];
      var newItem = newTextArray[i];

      if (oldItem.text !== newItem.text) {
        return true;
      } else {
        var oldStyle = oldItem.style,
            newStyle = newItem.style;

        if (oldStyle) {
          if (newStyle) {
            if (!oldStyle.outline !== !newStyle.outline) {
              return true;
            }

            if (oldStyle.size !== newStyle.size || !oldStyle.italic !== !newStyle.italic || oldStyle.isImage !== newStyle.isImage) {
              return true;
            }

            if (oldStyle.src !== newStyle.src || oldStyle.imageAlign !== newStyle.imageAlign || oldStyle.imageHeight !== newStyle.imageHeight || oldStyle.imageWidth !== newStyle.imageWidth || oldStyle.imageOffset !== newStyle.imageOffset) {
              return true;
            }
          } else {
            if (oldStyle.size || oldStyle.italic || oldStyle.isImage || oldStyle.outline) {
              return true;
            }
          }
        } else {
          if (newStyle) {
            if (newStyle.size || newStyle.italic || newStyle.isImage || newStyle.outline) {
              return true;
            }
          }
        }
      }
    }

    return false;
  },
  _addRichTextImageElement: function _addRichTextImageElement(richTextElement) {
    var spriteFrameName = richTextElement.style.src;
    var spriteFrame = this.imageAtlas.getSpriteFrame(spriteFrameName);

    if (spriteFrame) {
      var spriteNode = new cc.PrivateNode(RichTextChildImageName);
      var spriteComponent = spriteNode.addComponent(cc.Sprite);

      switch (richTextElement.style.imageAlign) {
        case 'top':
          spriteNode.setAnchorPoint(0, 1);
          break;

        case 'center':
          spriteNode.setAnchorPoint(0, 0.5);
          break;

        default:
          spriteNode.setAnchorPoint(0, 0);
          break;
      }

      if (richTextElement.style.imageOffset) spriteNode._imageOffset = richTextElement.style.imageOffset;
      spriteComponent.type = cc.Sprite.Type.SLICED;
      spriteComponent.sizeMode = cc.Sprite.SizeMode.CUSTOM;
      this.node.addChild(spriteNode);

      this._labelSegments.push(spriteNode);

      var spriteRect = spriteFrame.getRect();
      var scaleFactor = 1;
      var spriteWidth = spriteRect.width;
      var spriteHeight = spriteRect.height;
      var expectWidth = richTextElement.style.imageWidth;
      var expectHeight = richTextElement.style.imageHeight;

      if (expectHeight > 0) {
        scaleFactor = expectHeight / spriteHeight;
        spriteWidth = spriteWidth * scaleFactor;
        spriteHeight = spriteHeight * scaleFactor;
      } else {
        scaleFactor = this.lineHeight / spriteHeight;
        spriteWidth = spriteWidth * scaleFactor;
        spriteHeight = spriteHeight * scaleFactor;
      }

      if (expectWidth > 0) spriteWidth = expectWidth;

      if (this.maxWidth > 0) {
        if (this._lineOffsetX + spriteWidth > this.maxWidth) {
          this._updateLineInfo();
        }

        this._lineOffsetX += spriteWidth;
      } else {
        this._lineOffsetX += spriteWidth;

        if (this._lineOffsetX > this._labelWidth) {
          this._labelWidth = this._lineOffsetX;
        }
      }

      spriteComponent.spriteFrame = spriteFrame;
      spriteNode.setContentSize(spriteWidth, spriteHeight);
      spriteNode._lineCount = this._lineCount;

      if (richTextElement.style.event) {
        if (richTextElement.style.event.click) {
          spriteNode._clickHandler = richTextElement.style.event.click;
        }

        if (richTextElement.style.event.param) {
          spriteNode._clickParam = richTextElement.style.event.param;
        } else {
          spriteNode._clickParam = '';
        }
      } else {
        spriteNode._clickHandler = null;
      }
    } else {
      cc.warnID(4400);
    }
  },
  _updateRichText: function _updateRichText() {
    if (!this.enabledInHierarchy) return;

    var newTextArray = _htmlTextParser.parse(this.string);

    if (!this._needsUpdateTextLayout(newTextArray)) {
      this._textArray = newTextArray;

      this._updateLabelSegmentTextAttributes();

      return;
    }

    this._textArray = newTextArray;

    this._resetState();

    var lastEmptyLine = false;
    var label;
    var labelSize;

    for (var i = 0; i < this._textArray.length; ++i) {
      var richTextElement = this._textArray[i];
      var text = richTextElement.text; //handle <br/> <img /> tag

      if (text === "") {
        if (richTextElement.style && richTextElement.style.newline) {
          this._updateLineInfo();

          continue;
        }

        if (richTextElement.style && richTextElement.style.isImage && this.imageAtlas) {
          this._addRichTextImageElement(richTextElement);

          continue;
        }
      }

      var multilineTexts = text.split("\n");

      for (var j = 0; j < multilineTexts.length; ++j) {
        var labelString = multilineTexts[j];

        if (labelString === "") {
          //for continues \n
          if (this._isLastComponentCR(text) && j === multilineTexts.length - 1) {
            continue;
          }

          this._updateLineInfo();

          lastEmptyLine = true;
          continue;
        }

        lastEmptyLine = false;

        if (this.maxWidth > 0) {
          var labelWidth = this._measureText(i, labelString);

          this._updateRichTextWithMaxWidth(labelString, labelWidth, i);

          if (multilineTexts.length > 1 && j < multilineTexts.length - 1) {
            this._updateLineInfo();
          }
        } else {
          label = this._addLabelSegment(labelString, i);
          labelSize = label.getContentSize();
          this._lineOffsetX += labelSize.width;

          if (this._lineOffsetX > this._labelWidth) {
            this._labelWidth = this._lineOffsetX;
          }

          if (multilineTexts.length > 1 && j < multilineTexts.length - 1) {
            this._updateLineInfo();
          }
        }
      }
    }

    if (!lastEmptyLine) {
      this._linesWidth.push(this._lineOffsetX);
    }

    if (this.maxWidth > 0) {
      this._labelWidth = this.maxWidth;
    }

    this._labelHeight = (this._lineCount + textUtils.BASELINE_RATIO) * this.lineHeight; // trigger "size-changed" event

    this.node.setContentSize(this._labelWidth, this._labelHeight);

    this._updateRichTextPosition();

    this._layoutDirty = false;
  },
  _getFirstWordLen: function _getFirstWordLen(text, startIndex, textLen) {
    var character = text.charAt(startIndex);

    if (textUtils.isUnicodeCJK(character) || textUtils.isUnicodeSpace(character)) {
      return 1;
    }

    var len = 1;

    for (var index = startIndex + 1; index < textLen; ++index) {
      character = text.charAt(index);

      if (textUtils.isUnicodeSpace(character) || textUtils.isUnicodeCJK(character)) {
        break;
      }

      len++;
    }

    return len;
  },
  _updateRichTextPosition: function _updateRichTextPosition() {
    var nextTokenX = 0;
    var nextLineIndex = 1;
    var totalLineCount = this._lineCount;

    for (var i = 0; i < this._labelSegments.length; ++i) {
      var label = this._labelSegments[i];
      var lineCount = label._lineCount;

      if (lineCount > nextLineIndex) {
        nextTokenX = 0;
        nextLineIndex = lineCount;
      }

      var lineOffsetX = 0; // let nodeAnchorXOffset = (0.5 - this.node.anchorX) * this._labelWidth;

      switch (this.horizontalAlign) {
        case HorizontalAlign.LEFT:
          lineOffsetX = -this._labelWidth / 2;
          break;

        case HorizontalAlign.CENTER:
          lineOffsetX = -this._linesWidth[lineCount - 1] / 2;
          break;

        case HorizontalAlign.RIGHT:
          lineOffsetX = this._labelWidth / 2 - this._linesWidth[lineCount - 1];
          break;

        default:
          break;
      }

      label.x = nextTokenX + lineOffsetX;
      var labelSize = label.getContentSize();
      label.y = this.lineHeight * (totalLineCount - lineCount) - this._labelHeight / 2;

      if (lineCount === nextLineIndex) {
        nextTokenX += labelSize.width;
      }

      var sprite = label.getComponent(cc.Sprite);

      if (sprite) {
        // adjust img align (from <img align=top|center|bottom>)
        var lineHeightSet = this.lineHeight;
        var lineHeightReal = this.lineHeight * (1 + textUtils.BASELINE_RATIO); //single line node height

        switch (label.anchorY) {
          case 1:
            label.y += lineHeightSet + (lineHeightReal - lineHeightSet) / 2;
            break;

          case 0.5:
            label.y += lineHeightReal / 2;
            break;

          default:
            label.y += (lineHeightReal - lineHeightSet) / 2;
            break;
        } // adjust img offset (from <img offset=12|12,34>)


        if (label._imageOffset) {
          var offsets = label._imageOffset.split(',');

          if (offsets.length === 1 && offsets[0]) {
            var offsetY = parseFloat(offsets[0]);
            if (Number.isInteger(offsetY)) label.y += offsetY;
          } else if (offsets.length === 2) {
            var offsetX = parseFloat(offsets[0]);

            var _offsetY = parseFloat(offsets[1]);

            if (Number.isInteger(offsetX)) label.x += offsetX;
            if (Number.isInteger(_offsetY)) label.y += _offsetY;
          }
        }
      } //adjust y for label with outline


      var outline = label.getComponent(cc.LabelOutline);
      if (outline && outline.width) label.y = label.y - outline.width;
    }
  },
  _convertLiteralColorValue: function _convertLiteralColorValue(color) {
    var colorValue = color.toUpperCase();

    if (cc.Color[colorValue]) {
      return cc.Color[colorValue];
    } else {
      var out = cc.color();
      return out.fromHEX(color);
    }
  },
  // When string is null, it means that the text does not need to be updated.
  _applyTextAttribute: function _applyTextAttribute(labelNode, string, force) {
    var labelComponent = labelNode.getComponent(cc.Label);

    if (!labelComponent) {
      return;
    }

    var index = labelNode._styleIndex;
    var textStyle = null;

    if (this._textArray[index]) {
      textStyle = this._textArray[index].style;
    }

    if (textStyle && textStyle.color) {
      labelNode.color = this._convertLiteralColorValue(textStyle.color);
    } else {
      labelNode.color = this.node.color;
    }

    labelComponent.cacheMode = this.cacheMode;
    var isAsset = this.font instanceof cc.Font;

    if (isAsset && !this._isSystemFontUsed) {
      labelComponent.font = this.font;
    } else {
      labelComponent.fontFamily = this.fontFamily;
    }

    labelComponent.useSystemFont = this._isSystemFontUsed;
    labelComponent.lineHeight = this.lineHeight;
    labelComponent.enableBold = textStyle && textStyle.bold;
    labelComponent.enableItalics = textStyle && textStyle.italic; //TODO: temporary implementation, the italic effect should be implemented in the internal of label-assembler.

    if (textStyle && textStyle.italic) {
      labelNode.skewX = 12;
    }

    labelComponent.enableUnderline = textStyle && textStyle.underline;

    if (textStyle && textStyle.outline) {
      var labelOutlineComponent = labelNode.getComponent(cc.LabelOutline);

      if (!labelOutlineComponent) {
        labelOutlineComponent = labelNode.addComponent(cc.LabelOutline);
      }

      labelOutlineComponent.color = this._convertLiteralColorValue(textStyle.outline.color);
      labelOutlineComponent.width = textStyle.outline.width;
    }

    if (textStyle && textStyle.size) {
      labelComponent.fontSize = textStyle.size;
    } else {
      labelComponent.fontSize = this.fontSize;
    }

    if (string !== null) {
      if (typeof string !== 'string') {
        string = '' + string;
      }

      labelComponent.string = string;
    }

    force && labelComponent._forceUpdateRenderData();

    if (textStyle && textStyle.event) {
      if (textStyle.event.click) {
        labelNode._clickHandler = textStyle.event.click;
      }

      if (textStyle.event.param) {
        labelNode._clickParam = textStyle.event.param;
      } else {
        labelNode._clickParam = '';
      }
    } else {
      labelNode._clickHandler = null;
    }
  },
  onDestroy: function onDestroy() {
    for (var i = 0; i < this._labelSegments.length; ++i) {
      this._labelSegments[i].removeFromParent();

      pool.put(this._labelSegments[i]);
    }
  }
});
cc.RichText = module.exports = RichText;
                    }
                    if (nodeEnv) {
                        __define(__module.exports, __require, __module);
                    }
                    else {
                        __quick_compile_engine__.registerModuleFunc(__filename, function () {
                            __define(__module.exports, __require, __module);
                        });
                    }
                })();
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImVuZ2luZS1kZXYvY29jb3MyZC9jb3JlL2NvbXBvbmVudHMvQ0NSaWNoVGV4dC5qcyJdLCJuYW1lcyI6WyJqcyIsInJlcXVpcmUiLCJtYWNybyIsInRleHRVdGlscyIsIkh0bWxUZXh0UGFyc2VyIiwiX2h0bWxUZXh0UGFyc2VyIiwiSG9yaXpvbnRhbEFsaWduIiwiVGV4dEFsaWdubWVudCIsIlZlcnRpY2FsQWxpZ24iLCJWZXJ0aWNhbFRleHRBbGlnbm1lbnQiLCJSaWNoVGV4dENoaWxkTmFtZSIsIlJpY2hUZXh0Q2hpbGRJbWFnZU5hbWUiLCJDYWNoZU1vZGUiLCJjYyIsIkxhYmVsIiwiZGVib3VuY2UiLCJmdW5jIiwid2FpdCIsImltbWVkaWF0ZSIsInRpbWVvdXQiLCJjb250ZXh0IiwibGF0ZXIiLCJhcHBseSIsImFyZ3VtZW50cyIsImNhbGxOb3ciLCJjbGVhclRpbWVvdXQiLCJzZXRUaW1lb3V0IiwicG9vbCIsIlBvb2wiLCJub2RlIiwiQ0NfRURJVE9SIiwiaXNWYWxpZCIsImRlc3Ryb3kiLCJDQ19ERVYiLCJhc3NlcnQiLCJfcGFyZW50Iiwib3V0bGluZSIsImdldENvbXBvbmVudCIsIkxhYmVsT3V0bGluZSIsIndpZHRoIiwiZ2V0Iiwic3RyaW5nIiwicmljaHRleHQiLCJsYWJlbE5vZGUiLCJfZ2V0IiwiUHJpdmF0ZU5vZGUiLCJzZXRQb3NpdGlvbiIsInNldEFuY2hvclBvaW50Iiwic2tld1giLCJsYWJlbENvbXBvbmVudCIsImFkZENvbXBvbmVudCIsImhvcml6b250YWxBbGlnbiIsIkxFRlQiLCJ2ZXJ0aWNhbEFsaWduIiwiQ0VOVEVSIiwiUmljaFRleHQiLCJDbGFzcyIsIm5hbWUiLCJDb21wb25lbnQiLCJjdG9yIiwiX3RleHRBcnJheSIsIl9sYWJlbFNlZ21lbnRzIiwiX2xhYmVsU2VnbWVudHNDYWNoZSIsIl9saW5lc1dpZHRoIiwiX3VzZXJEZWZpbmVkRm9udCIsIl91cGRhdGVSaWNoVGV4dFN0YXR1cyIsIl91cGRhdGVSaWNoVGV4dCIsImVkaXRvciIsIm1lbnUiLCJoZWxwIiwiaW5zcGVjdG9yIiwiZXhlY3V0ZUluRWRpdE1vZGUiLCJwcm9wZXJ0aWVzIiwibXVsdGlsaW5lIiwidG9vbHRpcCIsIm5vdGlmeSIsInR5cGUiLCJhbmltYXRhYmxlIiwib2xkVmFsdWUiLCJfbGF5b3V0RGlydHkiLCJmb250U2l6ZSIsIl9mb250RmFtaWx5IiwiZm9udEZhbWlseSIsInNldCIsInZhbHVlIiwiZm9udCIsIlRURkZvbnQiLCJ1c2VTeXN0ZW1Gb250IiwiX29uVFRGTG9hZGVkIiwiX2lzU3lzdGVtRm9udFVzZWQiLCJjYWNoZU1vZGUiLCJOT05FIiwibWF4V2lkdGgiLCJsaW5lSGVpZ2h0IiwiaW1hZ2VBdGxhcyIsIlNwcml0ZUF0bGFzIiwiaGFuZGxlVG91Y2hFdmVudCIsImVuYWJsZWRJbkhpZXJhcmNoeSIsIl9hZGRFdmVudExpc3RlbmVycyIsIl9yZW1vdmVFdmVudExpc3RlbmVycyIsInN0YXRpY3MiLCJvbkVuYWJsZSIsIl9hY3RpdmF0ZUNoaWxkcmVuIiwib25EaXNhYmxlIiwic3RhcnQiLCJfb25Db2xvckNoYW5nZWQiLCJwYXJlbnRDb2xvciIsImNoaWxkcmVuIiwiZm9yRWFjaCIsImNoaWxkTm9kZSIsImNvbG9yIiwib24iLCJOb2RlIiwiRXZlbnRUeXBlIiwiVE9VQ0hfRU5EIiwiX29uVG91Y2hFbmRlZCIsIkNPTE9SX0NIQU5HRUQiLCJvZmYiLCJfdXBkYXRlTGFiZWxTZWdtZW50VGV4dEF0dHJpYnV0ZXMiLCJpdGVtIiwiX2FwcGx5VGV4dEF0dHJpYnV0ZSIsImJpbmQiLCJfY3JlYXRlRm9udExhYmVsIiwiX25hdGl2ZUFzc2V0Iiwic2VsZiIsImxvYWRlciIsImxvYWQiLCJuYXRpdmVVcmwiLCJlcnIiLCJfbWVhc3VyZVRleHQiLCJzdHlsZUluZGV4IiwibGFiZWwiLCJsZW5ndGgiLCJwdXNoIiwiX3N0eWxlSW5kZXgiLCJsYWJlbFNpemUiLCJnZXRDb250ZW50U2l6ZSIsImV2ZW50IiwiY29tcG9uZW50cyIsImdldENvbXBvbmVudHMiLCJpIiwibGFiZWxTZWdtZW50IiwiY2xpY2tIYW5kbGVyIiwiX2NsaWNrSGFuZGxlciIsImNsaWNrUGFyYW0iLCJfY2xpY2tQYXJhbSIsIl9jb250YWluc1RvdWNoTG9jYXRpb24iLCJ0b3VjaCIsImdldExvY2F0aW9uIiwiY29tcG9uZW50Iiwic3RvcFByb3BhZ2F0aW9uIiwicG9pbnQiLCJteVJlY3QiLCJnZXRCb3VuZGluZ0JveFRvV29ybGQiLCJjb250YWlucyIsIl9yZXNldFN0YXRlIiwiY2hpbGQiLCJwYXJlbnQiLCJzcGxpY2UiLCJwdXQiLCJfbGluZU9mZnNldFgiLCJfbGluZUNvdW50IiwiX2xhYmVsV2lkdGgiLCJfbGFiZWxIZWlnaHQiLCJvblJlc3RvcmUiLCJhY3RpdmUiLCJfYWRkTGFiZWxTZWdtZW50Iiwic3RyaW5nVG9rZW4iLCJwb3AiLCJhZGRDaGlsZCIsIl91cGRhdGVSaWNoVGV4dFdpdGhNYXhXaWR0aCIsImxhYmVsU3RyaW5nIiwibGFiZWxXaWR0aCIsImZyYWdtZW50V2lkdGgiLCJjaGVja1N0YXJ0SW5kZXgiLCJjaGVja0VuZEluZGV4IiwiX2dldEZpcnN0V29yZExlbiIsImNoZWNrU3RyaW5nIiwic3Vic3RyIiwiY2hlY2tTdHJpbmdXaWR0aCIsInJlbWFpbmluZ1N0cmluZyIsIl91cGRhdGVMaW5lSW5mbyIsImZyYWdtZW50cyIsImZyYWdtZW50VGV4dCIsImsiLCJzcGxpdFN0cmluZyIsIl9pc0xhc3RDb21wb25lbnRDUiIsImxhc3RJbmRleE9mIiwiX25lZWRzVXBkYXRlVGV4dExheW91dCIsIm5ld1RleHRBcnJheSIsIm9sZEl0ZW0iLCJuZXdJdGVtIiwidGV4dCIsIm9sZFN0eWxlIiwic3R5bGUiLCJuZXdTdHlsZSIsInNpemUiLCJpdGFsaWMiLCJpc0ltYWdlIiwic3JjIiwiaW1hZ2VBbGlnbiIsImltYWdlSGVpZ2h0IiwiaW1hZ2VXaWR0aCIsImltYWdlT2Zmc2V0IiwiX2FkZFJpY2hUZXh0SW1hZ2VFbGVtZW50IiwicmljaFRleHRFbGVtZW50Iiwic3ByaXRlRnJhbWVOYW1lIiwic3ByaXRlRnJhbWUiLCJnZXRTcHJpdGVGcmFtZSIsInNwcml0ZU5vZGUiLCJzcHJpdGVDb21wb25lbnQiLCJTcHJpdGUiLCJfaW1hZ2VPZmZzZXQiLCJUeXBlIiwiU0xJQ0VEIiwic2l6ZU1vZGUiLCJTaXplTW9kZSIsIkNVU1RPTSIsInNwcml0ZVJlY3QiLCJnZXRSZWN0Iiwic2NhbGVGYWN0b3IiLCJzcHJpdGVXaWR0aCIsInNwcml0ZUhlaWdodCIsImhlaWdodCIsImV4cGVjdFdpZHRoIiwiZXhwZWN0SGVpZ2h0Iiwic2V0Q29udGVudFNpemUiLCJjbGljayIsInBhcmFtIiwid2FybklEIiwicGFyc2UiLCJsYXN0RW1wdHlMaW5lIiwibmV3bGluZSIsIm11bHRpbGluZVRleHRzIiwic3BsaXQiLCJqIiwiQkFTRUxJTkVfUkFUSU8iLCJfdXBkYXRlUmljaFRleHRQb3NpdGlvbiIsInN0YXJ0SW5kZXgiLCJ0ZXh0TGVuIiwiY2hhcmFjdGVyIiwiY2hhckF0IiwiaXNVbmljb2RlQ0pLIiwiaXNVbmljb2RlU3BhY2UiLCJsZW4iLCJpbmRleCIsIm5leHRUb2tlblgiLCJuZXh0TGluZUluZGV4IiwidG90YWxMaW5lQ291bnQiLCJsaW5lQ291bnQiLCJsaW5lT2Zmc2V0WCIsIlJJR0hUIiwieCIsInkiLCJzcHJpdGUiLCJsaW5lSGVpZ2h0U2V0IiwibGluZUhlaWdodFJlYWwiLCJhbmNob3JZIiwib2Zmc2V0cyIsIm9mZnNldFkiLCJwYXJzZUZsb2F0IiwiTnVtYmVyIiwiaXNJbnRlZ2VyIiwib2Zmc2V0WCIsIl9jb252ZXJ0TGl0ZXJhbENvbG9yVmFsdWUiLCJjb2xvclZhbHVlIiwidG9VcHBlckNhc2UiLCJDb2xvciIsIm91dCIsImZyb21IRVgiLCJmb3JjZSIsInRleHRTdHlsZSIsImlzQXNzZXQiLCJGb250IiwiZW5hYmxlQm9sZCIsImJvbGQiLCJlbmFibGVJdGFsaWNzIiwiZW5hYmxlVW5kZXJsaW5lIiwidW5kZXJsaW5lIiwibGFiZWxPdXRsaW5lQ29tcG9uZW50IiwiX2ZvcmNlVXBkYXRlUmVuZGVyRGF0YSIsIm9uRGVzdHJveSIsInJlbW92ZUZyb21QYXJlbnQiLCJtb2R1bGUiLCJleHBvcnRzIl0sIm1hcHBpbmdzIjoiOzs7Ozs7Ozs7Ozs7O0FBQUE7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7QUEwQkEsSUFBTUEsRUFBRSxHQUFHQyxPQUFPLENBQUMsZ0JBQUQsQ0FBbEI7O0FBQ0EsSUFBTUMsS0FBSyxHQUFHRCxPQUFPLENBQUMscUJBQUQsQ0FBckI7O0FBQ0EsSUFBTUUsU0FBUyxHQUFHRixPQUFPLENBQUMscUJBQUQsQ0FBekI7O0FBQ0EsSUFBTUcsY0FBYyxHQUFHSCxPQUFPLENBQUMsMkJBQUQsQ0FBOUI7O0FBQ0EsSUFBTUksZUFBZSxHQUFHLElBQUlELGNBQUosRUFBeEI7O0FBRUEsSUFBTUUsZUFBZSxHQUFHSixLQUFLLENBQUNLLGFBQTlCO0FBQ0EsSUFBTUMsYUFBYSxHQUFHTixLQUFLLENBQUNPLHFCQUE1QjtBQUNBLElBQU1DLGlCQUFpQixHQUFHLGdCQUExQjtBQUNBLElBQU1DLHNCQUFzQixHQUFHLHNCQUEvQjtBQUNBLElBQU1DLFNBQVMsR0FBR0MsRUFBRSxDQUFDQyxLQUFILENBQVNGLFNBQTNCLEVBRUE7QUFDQTtBQUNBO0FBQ0E7O0FBQ0EsU0FBU0csUUFBVCxDQUFrQkMsSUFBbEIsRUFBd0JDLElBQXhCLEVBQThCQyxTQUE5QixFQUF5QztBQUNyQyxNQUFJQyxPQUFKO0FBQ0EsU0FBTyxZQUFZO0FBQ2YsUUFBSUMsT0FBTyxHQUFHLElBQWQ7O0FBQ0EsUUFBSUMsS0FBSyxHQUFHLFNBQVJBLEtBQVEsR0FBWTtBQUNwQkYsTUFBQUEsT0FBTyxHQUFHLElBQVY7QUFDQSxVQUFJLENBQUNELFNBQUwsRUFBZ0JGLElBQUksQ0FBQ00sS0FBTCxDQUFXRixPQUFYLEVBQW9CRyxTQUFwQjtBQUNuQixLQUhEOztBQUlBLFFBQUlDLE9BQU8sR0FBR04sU0FBUyxJQUFJLENBQUNDLE9BQTVCO0FBQ0FNLElBQUFBLFlBQVksQ0FBQ04sT0FBRCxDQUFaO0FBQ0FBLElBQUFBLE9BQU8sR0FBR08sVUFBVSxDQUFDTCxLQUFELEVBQVFKLElBQVIsQ0FBcEI7QUFDQSxRQUFJTyxPQUFKLEVBQWFSLElBQUksQ0FBQ00sS0FBTCxDQUFXRixPQUFYLEVBQW9CRyxTQUFwQjtBQUNoQixHQVZEO0FBV0g7QUFFRDs7Ozs7QUFHQSxJQUFJSSxJQUFJLEdBQUcsSUFBSTNCLEVBQUUsQ0FBQzRCLElBQVAsQ0FBWSxVQUFVQyxJQUFWLEVBQWdCO0FBQ25DLE1BQUlDLFNBQUosRUFBZTtBQUNYakIsSUFBQUEsRUFBRSxDQUFDa0IsT0FBSCxDQUFXRixJQUFYLEtBQW9CQSxJQUFJLENBQUNHLE9BQUwsRUFBcEI7QUFDQSxXQUFPLEtBQVA7QUFDSDs7QUFDRCxNQUFJQyxNQUFKLEVBQVk7QUFDUnBCLElBQUFBLEVBQUUsQ0FBQ3FCLE1BQUgsQ0FBVSxDQUFDTCxJQUFJLENBQUNNLE9BQWhCLEVBQXlCLDBDQUF6QjtBQUNIOztBQUNELE1BQUksQ0FBQ3RCLEVBQUUsQ0FBQ2tCLE9BQUgsQ0FBV0YsSUFBWCxDQUFMLEVBQXVCO0FBQ25CLFdBQU8sS0FBUDtBQUNILEdBRkQsTUFFTztBQUNILFFBQUlPLE9BQU8sR0FBR1AsSUFBSSxDQUFDUSxZQUFMLENBQWtCeEIsRUFBRSxDQUFDeUIsWUFBckIsQ0FBZDs7QUFDQSxRQUFJRixPQUFKLEVBQWE7QUFDVEEsTUFBQUEsT0FBTyxDQUFDRyxLQUFSLEdBQWdCLENBQWhCO0FBQ0g7QUFDSjs7QUFFRCxTQUFPLElBQVA7QUFDSCxDQWxCVSxFQWtCUixFQWxCUSxDQUFYOztBQW9CQVosSUFBSSxDQUFDYSxHQUFMLEdBQVcsVUFBVUMsTUFBVixFQUFrQkMsUUFBbEIsRUFBNEI7QUFDbkMsTUFBSUMsU0FBUyxHQUFHLEtBQUtDLElBQUwsRUFBaEI7O0FBQ0EsTUFBSSxDQUFDRCxTQUFMLEVBQWdCO0FBQ1pBLElBQUFBLFNBQVMsR0FBRyxJQUFJOUIsRUFBRSxDQUFDZ0MsV0FBUCxDQUFtQm5DLGlCQUFuQixDQUFaO0FBQ0g7O0FBRURpQyxFQUFBQSxTQUFTLENBQUNHLFdBQVYsQ0FBc0IsQ0FBdEIsRUFBeUIsQ0FBekI7QUFDQUgsRUFBQUEsU0FBUyxDQUFDSSxjQUFWLENBQXlCLEdBQXpCLEVBQThCLEdBQTlCO0FBQ0FKLEVBQUFBLFNBQVMsQ0FBQ0ssS0FBVixHQUFrQixDQUFsQjtBQUVBLE1BQUlDLGNBQWMsR0FBR04sU0FBUyxDQUFDTixZQUFWLENBQXVCeEIsRUFBRSxDQUFDQyxLQUExQixDQUFyQjs7QUFDQSxNQUFJLENBQUNtQyxjQUFMLEVBQXFCO0FBQ2pCQSxJQUFBQSxjQUFjLEdBQUdOLFNBQVMsQ0FBQ08sWUFBVixDQUF1QnJDLEVBQUUsQ0FBQ0MsS0FBMUIsQ0FBakI7QUFDSDs7QUFFRG1DLEVBQUFBLGNBQWMsQ0FBQ1IsTUFBZixHQUF3QixFQUF4QjtBQUNBUSxFQUFBQSxjQUFjLENBQUNFLGVBQWYsR0FBaUM3QyxlQUFlLENBQUM4QyxJQUFqRDtBQUNBSCxFQUFBQSxjQUFjLENBQUNJLGFBQWYsR0FBK0I3QyxhQUFhLENBQUM4QyxNQUE3QztBQUVBLFNBQU9YLFNBQVA7QUFDSCxDQXBCRDtBQXNCQTs7Ozs7Ozs7QUFNQSxJQUFJWSxRQUFRLEdBQUcxQyxFQUFFLENBQUMyQyxLQUFILENBQVM7QUFDcEJDLEVBQUFBLElBQUksRUFBRSxhQURjO0FBRXBCLGFBQVM1QyxFQUFFLENBQUM2QyxTQUZRO0FBSXBCQyxFQUFBQSxJQUFJLEVBQUUsZ0JBQVk7QUFDZCxTQUFLQyxVQUFMLEdBQWtCLElBQWxCO0FBQ0EsU0FBS0MsY0FBTCxHQUFzQixFQUF0QjtBQUNBLFNBQUtDLG1CQUFMLEdBQTJCLEVBQTNCO0FBQ0EsU0FBS0MsV0FBTCxHQUFtQixFQUFuQjs7QUFFQSxRQUFJakMsU0FBSixFQUFlO0FBQ1gsV0FBS2tDLGdCQUFMLEdBQXdCLElBQXhCO0FBQ0EsV0FBS0MscUJBQUwsR0FBNkJsRCxRQUFRLENBQUMsS0FBS21ELGVBQU4sRUFBdUIsR0FBdkIsQ0FBckM7QUFDSCxLQUhELE1BSUs7QUFDRCxXQUFLRCxxQkFBTCxHQUE2QixLQUFLQyxlQUFsQztBQUNIO0FBQ0osR0FqQm1CO0FBbUJwQkMsRUFBQUEsTUFBTSxFQUFFckMsU0FBUyxJQUFJO0FBQ2pCc0MsSUFBQUEsSUFBSSxFQUFFLDZDQURXO0FBRWpCQyxJQUFBQSxJQUFJLEVBQUUsa0NBRlc7QUFHakJDLElBQUFBLFNBQVMsRUFBRSxtREFITTtBQUlqQkMsSUFBQUEsaUJBQWlCLEVBQUU7QUFKRixHQW5CRDtBQTBCcEJDLEVBQUFBLFVBQVUsRUFBRTtBQUNSOzs7OztBQUtBL0IsSUFBQUEsTUFBTSxFQUFFO0FBQ0osaUJBQVMsb0RBREw7QUFFSmdDLE1BQUFBLFNBQVMsRUFBRSxJQUZQO0FBR0pDLE1BQUFBLE9BQU8sRUFBRXpDLE1BQU0sSUFBSSxnQ0FIZjtBQUlKMEMsTUFBQUEsTUFBTSxFQUFFLGtCQUFZO0FBQ2hCLGFBQUtWLHFCQUFMO0FBQ0g7QUFORyxLQU5BOztBQWVSOzs7OztBQUtBZCxJQUFBQSxlQUFlLEVBQUU7QUFDYixpQkFBUzdDLGVBQWUsQ0FBQzhDLElBRFo7QUFFYndCLE1BQUFBLElBQUksRUFBRXRFLGVBRk87QUFHYm9FLE1BQUFBLE9BQU8sRUFBRXpDLE1BQU0sSUFBSSwwQ0FITjtBQUliNEMsTUFBQUEsVUFBVSxFQUFFLEtBSkM7QUFLYkYsTUFBQUEsTUFBTSxFQUFFLGdCQUFVRyxRQUFWLEVBQW9CO0FBQ3hCLFlBQUksS0FBSzNCLGVBQUwsS0FBeUIyQixRQUE3QixFQUF1QztBQUV2QyxhQUFLQyxZQUFMLEdBQW9CLElBQXBCOztBQUNBLGFBQUtkLHFCQUFMO0FBQ0g7QUFWWSxLQXBCVDs7QUFpQ1I7Ozs7O0FBS0FlLElBQUFBLFFBQVEsRUFBRTtBQUNOLGlCQUFTLEVBREg7QUFFTk4sTUFBQUEsT0FBTyxFQUFFekMsTUFBTSxJQUFJLG1DQUZiO0FBR04wQyxNQUFBQSxNQUFNLEVBQUUsZ0JBQVVHLFFBQVYsRUFBb0I7QUFDeEIsWUFBSSxLQUFLRSxRQUFMLEtBQWtCRixRQUF0QixFQUFnQztBQUVoQyxhQUFLQyxZQUFMLEdBQW9CLElBQXBCOztBQUNBLGFBQUtkLHFCQUFMO0FBQ0g7QUFSSyxLQXRDRjs7QUFpRFI7Ozs7O0FBS0FnQixJQUFBQSxXQUFXLEVBQUUsT0F0REw7QUF1RFJDLElBQUFBLFVBQVUsRUFBRTtBQUNSUixNQUFBQSxPQUFPLEVBQUV6QyxNQUFNLElBQUkscUNBRFg7QUFFUk8sTUFBQUEsR0FGUSxpQkFFRDtBQUNILGVBQU8sS0FBS3lDLFdBQVo7QUFDSCxPQUpPO0FBS1JFLE1BQUFBLEdBTFEsZUFLSEMsS0FMRyxFQUtJO0FBQ1IsWUFBSSxLQUFLSCxXQUFMLEtBQXFCRyxLQUF6QixFQUFnQztBQUNoQyxhQUFLSCxXQUFMLEdBQW1CRyxLQUFuQjtBQUNBLGFBQUtMLFlBQUwsR0FBb0IsSUFBcEI7O0FBQ0EsYUFBS2QscUJBQUw7QUFDSCxPQVZPO0FBV1JZLE1BQUFBLFVBQVUsRUFBRTtBQVhKLEtBdkRKOztBQXFFUjs7Ozs7QUFLQVEsSUFBQUEsSUFBSSxFQUFFO0FBQ0YsaUJBQVMsSUFEUDtBQUVGVCxNQUFBQSxJQUFJLEVBQUUvRCxFQUFFLENBQUN5RSxPQUZQO0FBR0ZaLE1BQUFBLE9BQU8sRUFBRXpDLE1BQU0sSUFBSSw4QkFIakI7QUFJRjBDLE1BQUFBLE1BQU0sRUFBRSxnQkFBVUcsUUFBVixFQUFvQjtBQUN4QixZQUFJLEtBQUtPLElBQUwsS0FBY1AsUUFBbEIsRUFBNEI7QUFFNUIsYUFBS0MsWUFBTCxHQUFvQixJQUFwQjs7QUFDQSxZQUFJLEtBQUtNLElBQVQsRUFBZTtBQUNYLGNBQUl2RCxTQUFKLEVBQWU7QUFDWCxpQkFBS2tDLGdCQUFMLEdBQXdCLEtBQUtxQixJQUE3QjtBQUNIOztBQUNELGVBQUtFLGFBQUwsR0FBcUIsS0FBckI7O0FBQ0EsZUFBS0MsWUFBTDtBQUNILFNBTkQsTUFPSztBQUNELGVBQUtELGFBQUwsR0FBcUIsSUFBckI7QUFDSDs7QUFDRCxhQUFLdEIscUJBQUw7QUFDSDtBQW5CQyxLQTFFRTs7QUFnR1I7Ozs7O0FBS0F3QixJQUFBQSxpQkFBaUIsRUFBRSxJQXJHWDtBQXNHUkYsSUFBQUEsYUFBYSxFQUFFO0FBQ1gvQyxNQUFBQSxHQURXLGlCQUNKO0FBQ0gsZUFBTyxLQUFLaUQsaUJBQVo7QUFDSCxPQUhVO0FBSVhOLE1BQUFBLEdBSlcsZUFJTkMsS0FKTSxFQUlDO0FBQ1IsWUFBSSxLQUFLSyxpQkFBTCxLQUEyQkwsS0FBL0IsRUFBc0M7QUFDbEM7QUFDSDs7QUFDRCxhQUFLSyxpQkFBTCxHQUF5QkwsS0FBekI7O0FBRUEsWUFBSXRELFNBQUosRUFBZTtBQUNYLGNBQUlzRCxLQUFKLEVBQVc7QUFDUCxpQkFBS0MsSUFBTCxHQUFZLElBQVo7QUFDSCxXQUZELE1BR0ssSUFBSSxLQUFLckIsZ0JBQVQsRUFBMkI7QUFDNUIsaUJBQUtxQixJQUFMLEdBQVksS0FBS3JCLGdCQUFqQjtBQUNBO0FBQ0g7QUFDSjs7QUFFRCxhQUFLZSxZQUFMLEdBQW9CLElBQXBCOztBQUNBLGFBQUtkLHFCQUFMO0FBQ0gsT0F0QlU7QUF1QlhZLE1BQUFBLFVBQVUsRUFBRSxLQXZCRDtBQXdCWEgsTUFBQUEsT0FBTyxFQUFFekMsTUFBTSxJQUFJO0FBeEJSLEtBdEdQOztBQWlJUjs7Ozs7QUFLQXlELElBQUFBLFNBQVMsRUFBRTtBQUNQLGlCQUFTOUUsU0FBUyxDQUFDK0UsSUFEWjtBQUVQZixNQUFBQSxJQUFJLEVBQUVoRSxTQUZDO0FBR1A4RCxNQUFBQSxPQUFPLEVBQUV6QyxNQUFNLElBQUksZ0NBSFo7QUFJUDBDLE1BQUFBLE1BSk8sa0JBSUNHLFFBSkQsRUFJVztBQUNkLFlBQUksS0FBS1ksU0FBTCxLQUFtQlosUUFBdkIsRUFBaUM7O0FBRWpDLGFBQUtiLHFCQUFMO0FBQ0gsT0FSTTtBQVNQWSxNQUFBQSxVQUFVLEVBQUU7QUFUTCxLQXRJSDs7QUFrSlI7Ozs7O0FBS0FlLElBQUFBLFFBQVEsRUFBRTtBQUNOLGlCQUFTLENBREg7QUFFTmxCLE1BQUFBLE9BQU8sRUFBRXpDLE1BQU0sSUFBSSxtQ0FGYjtBQUdOMEMsTUFBQUEsTUFBTSxFQUFFLGdCQUFVRyxRQUFWLEVBQW9CO0FBQ3hCLFlBQUksS0FBS2MsUUFBTCxLQUFrQmQsUUFBdEIsRUFBZ0M7QUFFaEMsYUFBS0MsWUFBTCxHQUFvQixJQUFwQjs7QUFDQSxhQUFLZCxxQkFBTDtBQUNIO0FBUkssS0F2SkY7O0FBa0tSOzs7OztBQUtBNEIsSUFBQUEsVUFBVSxFQUFFO0FBQ1IsaUJBQVMsRUFERDtBQUVSbkIsTUFBQUEsT0FBTyxFQUFFekMsTUFBTSxJQUFJLHFDQUZYO0FBR1IwQyxNQUFBQSxNQUFNLEVBQUUsZ0JBQVVHLFFBQVYsRUFBb0I7QUFDeEIsWUFBSSxLQUFLZSxVQUFMLEtBQW9CZixRQUF4QixFQUFrQztBQUVsQyxhQUFLQyxZQUFMLEdBQW9CLElBQXBCOztBQUNBLGFBQUtkLHFCQUFMO0FBQ0g7QUFSTyxLQXZLSjs7QUFrTFI7Ozs7O0FBS0E2QixJQUFBQSxVQUFVLEVBQUU7QUFDUixpQkFBUyxJQUREO0FBRVJsQixNQUFBQSxJQUFJLEVBQUUvRCxFQUFFLENBQUNrRixXQUZEO0FBR1JyQixNQUFBQSxPQUFPLEVBQUV6QyxNQUFNLElBQUkscUNBSFg7QUFJUjBDLE1BQUFBLE1BQU0sRUFBRSxnQkFBVUcsUUFBVixFQUFvQjtBQUN4QixZQUFJLEtBQUtnQixVQUFMLEtBQW9CaEIsUUFBeEIsRUFBa0M7QUFFbEMsYUFBS0MsWUFBTCxHQUFvQixJQUFwQjs7QUFDQSxhQUFLZCxxQkFBTDtBQUNIO0FBVE8sS0F2TEo7O0FBbU1SOzs7Ozs7Ozs7QUFTQStCLElBQUFBLGdCQUFnQixFQUFFO0FBQ2QsaUJBQVMsSUFESztBQUVkdEIsTUFBQUEsT0FBTyxFQUFFekMsTUFBTSxJQUFJLDBDQUZMO0FBR2QwQyxNQUFBQSxNQUFNLEVBQUUsZ0JBQVVHLFFBQVYsRUFBb0I7QUFDeEIsWUFBSSxLQUFLa0IsZ0JBQUwsS0FBMEJsQixRQUE5QixFQUF3Qzs7QUFDeEMsWUFBSSxLQUFLbUIsa0JBQVQsRUFBNkI7QUFDekIsZUFBS0QsZ0JBQUwsR0FBd0IsS0FBS0Usa0JBQUwsRUFBeEIsR0FBb0QsS0FBS0MscUJBQUwsRUFBcEQ7QUFDSDtBQUNKO0FBUmE7QUE1TVYsR0ExQlE7QUFrUHBCQyxFQUFBQSxPQUFPLEVBQUU7QUFDTDlGLElBQUFBLGVBQWUsRUFBRUEsZUFEWjtBQUVMRSxJQUFBQSxhQUFhLEVBQUVBO0FBRlYsR0FsUFc7QUF1UHBCNkYsRUFBQUEsUUF2UG9CLHNCQXVQUjtBQUNSLFFBQUksS0FBS0wsZ0JBQVQsRUFBMkI7QUFDdkIsV0FBS0Usa0JBQUw7QUFDSDs7QUFDRCxTQUFLaEMsZUFBTDs7QUFDQSxTQUFLb0MsaUJBQUwsQ0FBdUIsSUFBdkI7QUFDSCxHQTdQbUI7QUErUHBCQyxFQUFBQSxTQS9Qb0IsdUJBK1BQO0FBQ1QsUUFBSSxLQUFLUCxnQkFBVCxFQUEyQjtBQUN2QixXQUFLRyxxQkFBTDtBQUNIOztBQUNELFNBQUtHLGlCQUFMLENBQXVCLEtBQXZCO0FBQ0gsR0FwUW1CO0FBc1FwQkUsRUFBQUEsS0F0UW9CLG1CQXNRWDtBQUNMLFNBQUtoQixZQUFMO0FBQ0gsR0F4UW1CO0FBMFFwQmlCLEVBQUFBLGVBMVFvQiwyQkEwUUhDLFdBMVFHLEVBMFFVO0FBQzFCLFFBQUlDLFFBQVEsR0FBRyxLQUFLOUUsSUFBTCxDQUFVOEUsUUFBekI7QUFDQUEsSUFBQUEsUUFBUSxDQUFDQyxPQUFULENBQWlCLFVBQVVDLFNBQVYsRUFBcUI7QUFDbENBLE1BQUFBLFNBQVMsQ0FBQ0MsS0FBVixHQUFrQkosV0FBbEI7QUFDSCxLQUZEO0FBR0gsR0EvUW1CO0FBaVJwQlIsRUFBQUEsa0JBalJvQixnQ0FpUkU7QUFDbEIsU0FBS3JFLElBQUwsQ0FBVWtGLEVBQVYsQ0FBYWxHLEVBQUUsQ0FBQ21HLElBQUgsQ0FBUUMsU0FBUixDQUFrQkMsU0FBL0IsRUFBMEMsS0FBS0MsYUFBL0MsRUFBOEQsSUFBOUQ7QUFDQSxTQUFLdEYsSUFBTCxDQUFVa0YsRUFBVixDQUFhbEcsRUFBRSxDQUFDbUcsSUFBSCxDQUFRQyxTQUFSLENBQWtCRyxhQUEvQixFQUE4QyxLQUFLWCxlQUFuRCxFQUFvRSxJQUFwRTtBQUNILEdBcFJtQjtBQXNScEJOLEVBQUFBLHFCQXRSb0IsbUNBc1JLO0FBQ3JCLFNBQUt0RSxJQUFMLENBQVV3RixHQUFWLENBQWN4RyxFQUFFLENBQUNtRyxJQUFILENBQVFDLFNBQVIsQ0FBa0JDLFNBQWhDLEVBQTJDLEtBQUtDLGFBQWhELEVBQStELElBQS9EO0FBQ0EsU0FBS3RGLElBQUwsQ0FBVXdGLEdBQVYsQ0FBY3hHLEVBQUUsQ0FBQ21HLElBQUgsQ0FBUUMsU0FBUixDQUFrQkcsYUFBaEMsRUFBK0MsS0FBS1gsZUFBcEQsRUFBcUUsSUFBckU7QUFDSCxHQXpSbUI7QUEyUnBCYSxFQUFBQSxpQ0EzUm9CLCtDQTJSaUI7QUFDakMsU0FBS3pELGNBQUwsQ0FBb0IrQyxPQUFwQixDQUE0QixVQUFVVyxJQUFWLEVBQWdCO0FBQ3hDLFdBQUtDLG1CQUFMLENBQXlCRCxJQUF6QixFQUErQixJQUEvQixFQUFxQyxJQUFyQztBQUNILEtBRjJCLENBRTFCRSxJQUYwQixDQUVyQixJQUZxQixDQUE1QjtBQUdILEdBL1JtQjtBQWlTcEJDLEVBQUFBLGdCQWpTb0IsNEJBaVNGakYsTUFqU0UsRUFpU007QUFDdEIsV0FBT2QsSUFBSSxDQUFDYSxHQUFMLENBQVNDLE1BQVQsRUFBaUIsSUFBakIsQ0FBUDtBQUNILEdBblNtQjtBQXFTcEIrQyxFQUFBQSxZQXJTb0IsMEJBcVNKO0FBQ1osUUFBSSxLQUFLSCxJQUFMLFlBQXFCeEUsRUFBRSxDQUFDeUUsT0FBNUIsRUFBcUM7QUFDakMsVUFBSSxLQUFLRCxJQUFMLENBQVVzQyxZQUFkLEVBQTRCO0FBQ3hCLGFBQUs1QyxZQUFMLEdBQW9CLElBQXBCOztBQUNBLGFBQUtiLGVBQUw7QUFDSCxPQUhELE1BSUs7QUFDRCxZQUFJMEQsSUFBSSxHQUFHLElBQVg7QUFDQS9HLFFBQUFBLEVBQUUsQ0FBQ2dILE1BQUgsQ0FBVUMsSUFBVixDQUFlLEtBQUt6QyxJQUFMLENBQVUwQyxTQUF6QixFQUFvQyxVQUFVQyxHQUFWLEVBQWU5QyxVQUFmLEVBQTJCO0FBQzNEMEMsVUFBQUEsSUFBSSxDQUFDN0MsWUFBTCxHQUFvQixJQUFwQjs7QUFDQTZDLFVBQUFBLElBQUksQ0FBQzFELGVBQUw7QUFDSCxTQUhEO0FBSUg7QUFDSixLQVpELE1BYUs7QUFDRCxXQUFLYSxZQUFMLEdBQW9CLElBQXBCOztBQUNBLFdBQUtiLGVBQUw7QUFDSDtBQUNKLEdBdlRtQjtBQXlUcEIrRCxFQUFBQSxZQXpUb0Isd0JBeVROQyxVQXpUTSxFQXlUTXpGLE1BelROLEVBeVRjO0FBQzlCLFFBQUltRixJQUFJLEdBQUcsSUFBWDs7QUFDQSxRQUFJNUcsSUFBSSxHQUFHLFNBQVBBLElBQU8sQ0FBVXlCLE1BQVYsRUFBa0I7QUFDekIsVUFBSTBGLEtBQUo7O0FBQ0EsVUFBSVAsSUFBSSxDQUFDOUQsbUJBQUwsQ0FBeUJzRSxNQUF6QixLQUFvQyxDQUF4QyxFQUEyQztBQUN2Q0QsUUFBQUEsS0FBSyxHQUFHUCxJQUFJLENBQUNGLGdCQUFMLENBQXNCakYsTUFBdEIsQ0FBUjs7QUFDQW1GLFFBQUFBLElBQUksQ0FBQzlELG1CQUFMLENBQXlCdUUsSUFBekIsQ0FBOEJGLEtBQTlCO0FBQ0gsT0FIRCxNQUdPO0FBQ0hBLFFBQUFBLEtBQUssR0FBR1AsSUFBSSxDQUFDOUQsbUJBQUwsQ0FBeUIsQ0FBekIsQ0FBUjtBQUNIOztBQUNEcUUsTUFBQUEsS0FBSyxDQUFDRyxXQUFOLEdBQW9CSixVQUFwQjs7QUFDQU4sTUFBQUEsSUFBSSxDQUFDSixtQkFBTCxDQUF5QlcsS0FBekIsRUFBZ0MxRixNQUFoQyxFQUF3QyxJQUF4Qzs7QUFDQSxVQUFJOEYsU0FBUyxHQUFHSixLQUFLLENBQUNLLGNBQU4sRUFBaEI7QUFDQSxhQUFPRCxTQUFTLENBQUNoRyxLQUFqQjtBQUNILEtBWkQ7O0FBYUEsUUFBSUUsTUFBSixFQUFZO0FBQ1IsYUFBT3pCLElBQUksQ0FBQ3lCLE1BQUQsQ0FBWDtBQUNILEtBRkQsTUFHSztBQUNELGFBQU96QixJQUFQO0FBQ0g7QUFDSixHQTlVbUI7QUFnVnBCbUcsRUFBQUEsYUFoVm9CLHlCQWdWTHNCLEtBaFZLLEVBZ1ZFO0FBQUE7O0FBQ2xCLFFBQUlDLFVBQVUsR0FBRyxLQUFLN0csSUFBTCxDQUFVOEcsYUFBVixDQUF3QjlILEVBQUUsQ0FBQzZDLFNBQTNCLENBQWpCOztBQURrQiwrQkFHVGtGLENBSFM7QUFJZCxVQUFJQyxZQUFZLEdBQUcsS0FBSSxDQUFDaEYsY0FBTCxDQUFvQitFLENBQXBCLENBQW5CO0FBQ0EsVUFBSUUsWUFBWSxHQUFHRCxZQUFZLENBQUNFLGFBQWhDO0FBQ0EsVUFBSUMsVUFBVSxHQUFHSCxZQUFZLENBQUNJLFdBQTlCOztBQUNBLFVBQUlILFlBQVksSUFBSSxLQUFJLENBQUNJLHNCQUFMLENBQTRCTCxZQUE1QixFQUEwQ0osS0FBSyxDQUFDVSxLQUFOLENBQVlDLFdBQVosRUFBMUMsQ0FBcEIsRUFBMEY7QUFDdEZWLFFBQUFBLFVBQVUsQ0FBQzlCLE9BQVgsQ0FBbUIsVUFBVXlDLFNBQVYsRUFBcUI7QUFDcEMsY0FBSUEsU0FBUyxDQUFDcEQsa0JBQVYsSUFBZ0NvRCxTQUFTLENBQUNQLFlBQUQsQ0FBN0MsRUFBNkQ7QUFDekRPLFlBQUFBLFNBQVMsQ0FBQ1AsWUFBRCxDQUFULENBQXdCTCxLQUF4QixFQUErQk8sVUFBL0I7QUFDSDtBQUNKLFNBSkQ7QUFLQVAsUUFBQUEsS0FBSyxDQUFDYSxlQUFOO0FBQ0g7QUFkYTs7QUFHbEIsU0FBSyxJQUFJVixDQUFDLEdBQUcsQ0FBYixFQUFnQkEsQ0FBQyxHQUFHLEtBQUsvRSxjQUFMLENBQW9CdUUsTUFBeEMsRUFBZ0QsRUFBRVEsQ0FBbEQsRUFBcUQ7QUFBQSxZQUE1Q0EsQ0FBNEM7QUFZcEQ7QUFDSixHQWhXbUI7QUFrV3BCTSxFQUFBQSxzQkFsV29CLGtDQWtXSWYsS0FsV0osRUFrV1dvQixLQWxXWCxFQWtXa0I7QUFDbEMsUUFBSUMsTUFBTSxHQUFHckIsS0FBSyxDQUFDc0IscUJBQU4sRUFBYjtBQUNBLFdBQU9ELE1BQU0sQ0FBQ0UsUUFBUCxDQUFnQkgsS0FBaEIsQ0FBUDtBQUNILEdBcldtQjtBQXVXcEJJLEVBQUFBLFdBdldvQix5QkF1V0w7QUFDWCxRQUFJaEQsUUFBUSxHQUFHLEtBQUs5RSxJQUFMLENBQVU4RSxRQUF6Qjs7QUFDQSxTQUFLLElBQUlpQyxDQUFDLEdBQUdqQyxRQUFRLENBQUN5QixNQUFULEdBQWtCLENBQS9CLEVBQWtDUSxDQUFDLElBQUksQ0FBdkMsRUFBMENBLENBQUMsRUFBM0MsRUFBK0M7QUFDM0MsVUFBSWdCLEtBQUssR0FBR2pELFFBQVEsQ0FBQ2lDLENBQUQsQ0FBcEI7O0FBQ0EsVUFBSWdCLEtBQUssQ0FBQ25HLElBQU4sS0FBZS9DLGlCQUFmLElBQW9Da0osS0FBSyxDQUFDbkcsSUFBTixLQUFlOUMsc0JBQXZELEVBQStFO0FBQzNFLFlBQUlpSixLQUFLLENBQUNDLE1BQU4sS0FBaUIsS0FBS2hJLElBQTFCLEVBQWdDO0FBQzVCK0gsVUFBQUEsS0FBSyxDQUFDQyxNQUFOLEdBQWUsSUFBZjtBQUNILFNBRkQsTUFHSztBQUNEO0FBQ0FsRCxVQUFBQSxRQUFRLENBQUNtRCxNQUFULENBQWdCbEIsQ0FBaEIsRUFBbUIsQ0FBbkI7QUFDSDs7QUFDRCxZQUFJZ0IsS0FBSyxDQUFDbkcsSUFBTixLQUFlL0MsaUJBQW5CLEVBQXNDO0FBQ2xDaUIsVUFBQUEsSUFBSSxDQUFDb0ksR0FBTCxDQUFTSCxLQUFUO0FBQ0g7QUFDSjtBQUNKOztBQUVELFNBQUsvRixjQUFMLENBQW9CdUUsTUFBcEIsR0FBNkIsQ0FBN0I7QUFDQSxTQUFLdEUsbUJBQUwsQ0FBeUJzRSxNQUF6QixHQUFrQyxDQUFsQztBQUNBLFNBQUtyRSxXQUFMLENBQWlCcUUsTUFBakIsR0FBMEIsQ0FBMUI7QUFDQSxTQUFLNEIsWUFBTCxHQUFvQixDQUFwQjtBQUNBLFNBQUtDLFVBQUwsR0FBa0IsQ0FBbEI7QUFDQSxTQUFLQyxXQUFMLEdBQW1CLENBQW5CO0FBQ0EsU0FBS0MsWUFBTCxHQUFvQixDQUFwQjtBQUNBLFNBQUtwRixZQUFMLEdBQW9CLElBQXBCO0FBQ0gsR0FqWW1CO0FBbVlwQnFGLEVBQUFBLFNBQVMsRUFBRXRJLFNBQVMsSUFBSSxZQUFZO0FBQ2hDO0FBQ0E7QUFDQTtBQUNBLFFBQUksS0FBS21FLGtCQUFULEVBQTZCO0FBQ3pCLFdBQUtJLFFBQUw7QUFDSCxLQUZELE1BR0s7QUFDRCxXQUFLRSxTQUFMO0FBQ0g7QUFDSixHQTdZbUI7QUErWXBCRCxFQUFBQSxpQkEvWW9CLDZCQStZRCtELE1BL1lDLEVBK1lPO0FBQ3ZCLFNBQUssSUFBSXpCLENBQUMsR0FBRyxLQUFLL0csSUFBTCxDQUFVOEUsUUFBVixDQUFtQnlCLE1BQW5CLEdBQTRCLENBQXpDLEVBQTRDUSxDQUFDLElBQUksQ0FBakQsRUFBb0RBLENBQUMsRUFBckQsRUFBeUQ7QUFDckQsVUFBSWdCLEtBQUssR0FBRyxLQUFLL0gsSUFBTCxDQUFVOEUsUUFBVixDQUFtQmlDLENBQW5CLENBQVo7O0FBQ0EsVUFBSWdCLEtBQUssQ0FBQ25HLElBQU4sS0FBZS9DLGlCQUFmLElBQW9Da0osS0FBSyxDQUFDbkcsSUFBTixLQUFlOUMsc0JBQXZELEVBQStFO0FBQzNFaUosUUFBQUEsS0FBSyxDQUFDUyxNQUFOLEdBQWVBLE1BQWY7QUFDSDtBQUNKO0FBQ0osR0F0Wm1CO0FBd1pwQkMsRUFBQUEsZ0JBeFpvQiw0QkF3WkZDLFdBeFpFLEVBd1pXckMsVUF4WlgsRUF3WnVCO0FBQ3ZDLFFBQUlXLFlBQUo7O0FBQ0EsUUFBSSxLQUFLL0UsbUJBQUwsQ0FBeUJzRSxNQUF6QixLQUFvQyxDQUF4QyxFQUEyQztBQUN2Q1MsTUFBQUEsWUFBWSxHQUFHLEtBQUtuQixnQkFBTCxDQUFzQjZDLFdBQXRCLENBQWY7QUFDSCxLQUZELE1BRU87QUFDSDFCLE1BQUFBLFlBQVksR0FBRyxLQUFLL0UsbUJBQUwsQ0FBeUIwRyxHQUF6QixFQUFmO0FBQ0g7O0FBQ0QzQixJQUFBQSxZQUFZLENBQUNQLFdBQWIsR0FBMkJKLFVBQTNCO0FBQ0FXLElBQUFBLFlBQVksQ0FBQ29CLFVBQWIsR0FBMEIsS0FBS0EsVUFBL0I7QUFDQXBCLElBQUFBLFlBQVksQ0FBQ3dCLE1BQWIsR0FBc0IsS0FBS3hJLElBQUwsQ0FBVXdJLE1BQWhDO0FBRUF4QixJQUFBQSxZQUFZLENBQUM5RixjQUFiLENBQTRCLENBQTVCLEVBQStCLENBQS9COztBQUNBLFNBQUt5RSxtQkFBTCxDQUF5QnFCLFlBQXpCLEVBQXVDMEIsV0FBdkM7O0FBRUEsU0FBSzFJLElBQUwsQ0FBVTRJLFFBQVYsQ0FBbUI1QixZQUFuQjs7QUFDQSxTQUFLaEYsY0FBTCxDQUFvQndFLElBQXBCLENBQXlCUSxZQUF6Qjs7QUFFQSxXQUFPQSxZQUFQO0FBQ0gsR0ExYW1CO0FBNGFwQjZCLEVBQUFBLDJCQTVhb0IsdUNBNGFTQyxXQTVhVCxFQTRhc0JDLFVBNWF0QixFQTRha0MxQyxVQTVhbEMsRUE0YThDO0FBQzlELFFBQUkyQyxhQUFhLEdBQUdELFVBQXBCO0FBQ0EsUUFBSS9CLFlBQUo7O0FBRUEsUUFBSSxLQUFLbUIsWUFBTCxHQUFvQixDQUFwQixJQUF5QmEsYUFBYSxHQUFHLEtBQUtiLFlBQXJCLEdBQW9DLEtBQUtwRSxRQUF0RSxFQUFnRjtBQUM1RTtBQUNBLFVBQUlrRixlQUFlLEdBQUcsQ0FBdEI7O0FBQ0EsYUFBTyxLQUFLZCxZQUFMLElBQXFCLEtBQUtwRSxRQUFqQyxFQUEyQztBQUN2QyxZQUFJbUYsYUFBYSxHQUFHLEtBQUtDLGdCQUFMLENBQXNCTCxXQUF0QixFQUNoQkcsZUFEZ0IsRUFFaEJILFdBQVcsQ0FBQ3ZDLE1BRkksQ0FBcEI7O0FBR0EsWUFBSTZDLFdBQVcsR0FBR04sV0FBVyxDQUFDTyxNQUFaLENBQW1CSixlQUFuQixFQUFvQ0MsYUFBcEMsQ0FBbEI7O0FBQ0EsWUFBSUksZ0JBQWdCLEdBQUcsS0FBS2xELFlBQUwsQ0FBa0JDLFVBQWxCLEVBQThCK0MsV0FBOUIsQ0FBdkI7O0FBRUEsWUFBSSxLQUFLakIsWUFBTCxHQUFvQm1CLGdCQUFwQixJQUF3QyxLQUFLdkYsUUFBakQsRUFBMkQ7QUFDdkQsZUFBS29FLFlBQUwsSUFBcUJtQixnQkFBckI7QUFDQUwsVUFBQUEsZUFBZSxJQUFJQyxhQUFuQjtBQUNILFNBSEQsTUFJSztBQUVELGNBQUlELGVBQWUsR0FBRyxDQUF0QixFQUF5QjtBQUNyQixnQkFBSU0sZUFBZSxHQUFHVCxXQUFXLENBQUNPLE1BQVosQ0FBbUIsQ0FBbkIsRUFBc0JKLGVBQXRCLENBQXRCOztBQUNBLGlCQUFLUixnQkFBTCxDQUFzQmMsZUFBdEIsRUFBdUNsRCxVQUF2Qzs7QUFDQXlDLFlBQUFBLFdBQVcsR0FBR0EsV0FBVyxDQUFDTyxNQUFaLENBQW1CSixlQUFuQixFQUFvQ0gsV0FBVyxDQUFDdkMsTUFBaEQsQ0FBZDtBQUNBeUMsWUFBQUEsYUFBYSxHQUFHLEtBQUs1QyxZQUFMLENBQWtCQyxVQUFsQixFQUE4QnlDLFdBQTlCLENBQWhCO0FBQ0g7O0FBQ0QsZUFBS1UsZUFBTDs7QUFDQTtBQUNIO0FBQ0o7QUFDSjs7QUFDRCxRQUFJUixhQUFhLEdBQUcsS0FBS2pGLFFBQXpCLEVBQW1DO0FBQy9CLFVBQUkwRixTQUFTLEdBQUduTCxTQUFTLENBQUNvTCxZQUFWLENBQXVCWixXQUF2QixFQUNaRSxhQURZLEVBRVosS0FBS2pGLFFBRk8sRUFHWixLQUFLcUMsWUFBTCxDQUFrQkMsVUFBbEIsQ0FIWSxDQUFoQjs7QUFJQSxXQUFLLElBQUlzRCxDQUFDLEdBQUcsQ0FBYixFQUFnQkEsQ0FBQyxHQUFHRixTQUFTLENBQUNsRCxNQUE5QixFQUFzQyxFQUFFb0QsQ0FBeEMsRUFBMkM7QUFDdkMsWUFBSUMsV0FBVyxHQUFHSCxTQUFTLENBQUNFLENBQUQsQ0FBM0I7QUFDQTNDLFFBQUFBLFlBQVksR0FBRyxLQUFLeUIsZ0JBQUwsQ0FBc0JtQixXQUF0QixFQUFtQ3ZELFVBQW5DLENBQWY7QUFDQSxZQUFJSyxTQUFTLEdBQUdNLFlBQVksQ0FBQ0wsY0FBYixFQUFoQjtBQUNBLGFBQUt3QixZQUFMLElBQXFCekIsU0FBUyxDQUFDaEcsS0FBL0I7O0FBQ0EsWUFBSStJLFNBQVMsQ0FBQ2xELE1BQVYsR0FBbUIsQ0FBbkIsSUFBd0JvRCxDQUFDLEdBQUdGLFNBQVMsQ0FBQ2xELE1BQVYsR0FBbUIsQ0FBbkQsRUFBc0Q7QUFDbEQsZUFBS2lELGVBQUw7QUFDSDtBQUNKO0FBQ0osS0FkRCxNQWVLO0FBQ0QsV0FBS3JCLFlBQUwsSUFBcUJhLGFBQXJCOztBQUNBLFdBQUtQLGdCQUFMLENBQXNCSyxXQUF0QixFQUFtQ3pDLFVBQW5DO0FBQ0g7QUFDSixHQTlkbUI7QUFnZXBCd0QsRUFBQUEsa0JBaGVvQiw4QkFnZUFuQixXQWhlQSxFQWdlYTtBQUM3QixXQUFPQSxXQUFXLENBQUNuQyxNQUFaLEdBQXFCLENBQXJCLEtBQTJCbUMsV0FBVyxDQUFDb0IsV0FBWixDQUF3QixJQUF4QixDQUFsQztBQUNILEdBbGVtQjtBQW9lcEJOLEVBQUFBLGVBcGVvQiw2QkFvZUQ7QUFDZixTQUFLdEgsV0FBTCxDQUFpQnNFLElBQWpCLENBQXNCLEtBQUsyQixZQUEzQjs7QUFDQSxTQUFLQSxZQUFMLEdBQW9CLENBQXBCO0FBQ0EsU0FBS0MsVUFBTDtBQUNILEdBeGVtQjtBQTBlcEIyQixFQUFBQSxzQkExZW9CLGtDQTBlSUMsWUExZUosRUEwZWtCO0FBQ2xDLFFBQUksS0FBSzlHLFlBQUwsSUFBcUIsQ0FBQyxLQUFLbkIsVUFBM0IsSUFBeUMsQ0FBQ2lJLFlBQTlDLEVBQTREO0FBQ3hELGFBQU8sSUFBUDtBQUNIOztBQUVELFFBQUksS0FBS2pJLFVBQUwsQ0FBZ0J3RSxNQUFoQixLQUEyQnlELFlBQVksQ0FBQ3pELE1BQTVDLEVBQW9EO0FBQ2hELGFBQU8sSUFBUDtBQUNIOztBQUVELFNBQUssSUFBSVEsQ0FBQyxHQUFHLENBQWIsRUFBZ0JBLENBQUMsR0FBRyxLQUFLaEYsVUFBTCxDQUFnQndFLE1BQXBDLEVBQTRDLEVBQUVRLENBQTlDLEVBQWlEO0FBQzdDLFVBQUlrRCxPQUFPLEdBQUcsS0FBS2xJLFVBQUwsQ0FBZ0JnRixDQUFoQixDQUFkO0FBQ0EsVUFBSW1ELE9BQU8sR0FBR0YsWUFBWSxDQUFDakQsQ0FBRCxDQUExQjs7QUFDQSxVQUFJa0QsT0FBTyxDQUFDRSxJQUFSLEtBQWlCRCxPQUFPLENBQUNDLElBQTdCLEVBQW1DO0FBQy9CLGVBQU8sSUFBUDtBQUNILE9BRkQsTUFHSztBQUNELFlBQUlDLFFBQVEsR0FBR0gsT0FBTyxDQUFDSSxLQUF2QjtBQUFBLFlBQThCQyxRQUFRLEdBQUdKLE9BQU8sQ0FBQ0csS0FBakQ7O0FBQ0EsWUFBSUQsUUFBSixFQUFjO0FBQ1YsY0FBSUUsUUFBSixFQUFjO0FBQ1YsZ0JBQUksQ0FBQ0YsUUFBUSxDQUFDN0osT0FBVixLQUFzQixDQUFDK0osUUFBUSxDQUFDL0osT0FBcEMsRUFBNkM7QUFDekMscUJBQU8sSUFBUDtBQUNIOztBQUNELGdCQUFJNkosUUFBUSxDQUFDRyxJQUFULEtBQWtCRCxRQUFRLENBQUNDLElBQTNCLElBQ0csQ0FBQ0gsUUFBUSxDQUFDSSxNQUFWLEtBQXFCLENBQUNGLFFBQVEsQ0FBQ0UsTUFEbEMsSUFFR0osUUFBUSxDQUFDSyxPQUFULEtBQXFCSCxRQUFRLENBQUNHLE9BRnJDLEVBRThDO0FBQzFDLHFCQUFPLElBQVA7QUFDSDs7QUFDRCxnQkFBSUwsUUFBUSxDQUFDTSxHQUFULEtBQWlCSixRQUFRLENBQUNJLEdBQTFCLElBQ0FOLFFBQVEsQ0FBQ08sVUFBVCxLQUF3QkwsUUFBUSxDQUFDSyxVQURqQyxJQUVBUCxRQUFRLENBQUNRLFdBQVQsS0FBeUJOLFFBQVEsQ0FBQ00sV0FGbEMsSUFHQVIsUUFBUSxDQUFDUyxVQUFULEtBQXdCUCxRQUFRLENBQUNPLFVBSGpDLElBSUFULFFBQVEsQ0FBQ1UsV0FBVCxLQUF5QlIsUUFBUSxDQUFDUSxXQUp0QyxFQUltRDtBQUMvQyxxQkFBTyxJQUFQO0FBQ0g7QUFDSixXQWhCRCxNQWlCSztBQUNELGdCQUFJVixRQUFRLENBQUNHLElBQVQsSUFBaUJILFFBQVEsQ0FBQ0ksTUFBMUIsSUFBb0NKLFFBQVEsQ0FBQ0ssT0FBN0MsSUFBd0RMLFFBQVEsQ0FBQzdKLE9BQXJFLEVBQThFO0FBQzFFLHFCQUFPLElBQVA7QUFDSDtBQUNKO0FBQ0osU0F2QkQsTUF3Qks7QUFDRCxjQUFJK0osUUFBSixFQUFjO0FBQ1YsZ0JBQUlBLFFBQVEsQ0FBQ0MsSUFBVCxJQUFpQkQsUUFBUSxDQUFDRSxNQUExQixJQUFvQ0YsUUFBUSxDQUFDRyxPQUE3QyxJQUF3REgsUUFBUSxDQUFDL0osT0FBckUsRUFBOEU7QUFDMUUscUJBQU8sSUFBUDtBQUNIO0FBQ0o7QUFDSjtBQUNKO0FBQ0o7O0FBQ0QsV0FBTyxLQUFQO0FBQ0gsR0E3aEJtQjtBQStoQnBCd0ssRUFBQUEsd0JBL2hCb0Isb0NBK2hCTUMsZUEvaEJOLEVBK2hCdUI7QUFDdkMsUUFBSUMsZUFBZSxHQUFHRCxlQUFlLENBQUNYLEtBQWhCLENBQXNCSyxHQUE1QztBQUNBLFFBQUlRLFdBQVcsR0FBRyxLQUFLakgsVUFBTCxDQUFnQmtILGNBQWhCLENBQStCRixlQUEvQixDQUFsQjs7QUFDQSxRQUFJQyxXQUFKLEVBQWlCO0FBQ2IsVUFBSUUsVUFBVSxHQUFHLElBQUlwTSxFQUFFLENBQUNnQyxXQUFQLENBQW1CbEMsc0JBQW5CLENBQWpCO0FBQ0EsVUFBSXVNLGVBQWUsR0FBR0QsVUFBVSxDQUFDL0osWUFBWCxDQUF3QnJDLEVBQUUsQ0FBQ3NNLE1BQTNCLENBQXRCOztBQUNBLGNBQVFOLGVBQWUsQ0FBQ1gsS0FBaEIsQ0FBc0JNLFVBQTlCO0FBRUksYUFBSyxLQUFMO0FBQ0lTLFVBQUFBLFVBQVUsQ0FBQ2xLLGNBQVgsQ0FBMEIsQ0FBMUIsRUFBNkIsQ0FBN0I7QUFDQTs7QUFDSixhQUFLLFFBQUw7QUFDSWtLLFVBQUFBLFVBQVUsQ0FBQ2xLLGNBQVgsQ0FBMEIsQ0FBMUIsRUFBNkIsR0FBN0I7QUFDQTs7QUFDSjtBQUNJa0ssVUFBQUEsVUFBVSxDQUFDbEssY0FBWCxDQUEwQixDQUExQixFQUE2QixDQUE3QjtBQUNBO0FBVlI7O0FBWUEsVUFBSThKLGVBQWUsQ0FBQ1gsS0FBaEIsQ0FBc0JTLFdBQTFCLEVBQXVDTSxVQUFVLENBQUNHLFlBQVgsR0FBMEJQLGVBQWUsQ0FBQ1gsS0FBaEIsQ0FBc0JTLFdBQWhEO0FBQ3ZDTyxNQUFBQSxlQUFlLENBQUN0SSxJQUFoQixHQUF1Qi9ELEVBQUUsQ0FBQ3NNLE1BQUgsQ0FBVUUsSUFBVixDQUFlQyxNQUF0QztBQUNBSixNQUFBQSxlQUFlLENBQUNLLFFBQWhCLEdBQTJCMU0sRUFBRSxDQUFDc00sTUFBSCxDQUFVSyxRQUFWLENBQW1CQyxNQUE5QztBQUNBLFdBQUs1TCxJQUFMLENBQVU0SSxRQUFWLENBQW1Cd0MsVUFBbkI7O0FBQ0EsV0FBS3BKLGNBQUwsQ0FBb0J3RSxJQUFwQixDQUF5QjRFLFVBQXpCOztBQUVBLFVBQUlTLFVBQVUsR0FBR1gsV0FBVyxDQUFDWSxPQUFaLEVBQWpCO0FBQ0EsVUFBSUMsV0FBVyxHQUFHLENBQWxCO0FBQ0EsVUFBSUMsV0FBVyxHQUFHSCxVQUFVLENBQUNuTCxLQUE3QjtBQUNBLFVBQUl1TCxZQUFZLEdBQUdKLFVBQVUsQ0FBQ0ssTUFBOUI7QUFDQSxVQUFJQyxXQUFXLEdBQUduQixlQUFlLENBQUNYLEtBQWhCLENBQXNCUSxVQUF4QztBQUNBLFVBQUl1QixZQUFZLEdBQUdwQixlQUFlLENBQUNYLEtBQWhCLENBQXNCTyxXQUF6Qzs7QUFFQSxVQUFJd0IsWUFBWSxHQUFHLENBQW5CLEVBQXNCO0FBQ2xCTCxRQUFBQSxXQUFXLEdBQUdLLFlBQVksR0FBR0gsWUFBN0I7QUFDQUQsUUFBQUEsV0FBVyxHQUFHQSxXQUFXLEdBQUdELFdBQTVCO0FBQ0FFLFFBQUFBLFlBQVksR0FBR0EsWUFBWSxHQUFHRixXQUE5QjtBQUNILE9BSkQsTUFLSztBQUNEQSxRQUFBQSxXQUFXLEdBQUcsS0FBSy9ILFVBQUwsR0FBa0JpSSxZQUFoQztBQUNBRCxRQUFBQSxXQUFXLEdBQUdBLFdBQVcsR0FBR0QsV0FBNUI7QUFDQUUsUUFBQUEsWUFBWSxHQUFHQSxZQUFZLEdBQUdGLFdBQTlCO0FBQ0g7O0FBRUQsVUFBSUksV0FBVyxHQUFHLENBQWxCLEVBQXFCSCxXQUFXLEdBQUdHLFdBQWQ7O0FBRXJCLFVBQUksS0FBS3BJLFFBQUwsR0FBZ0IsQ0FBcEIsRUFBdUI7QUFDbkIsWUFBSSxLQUFLb0UsWUFBTCxHQUFvQjZELFdBQXBCLEdBQWtDLEtBQUtqSSxRQUEzQyxFQUFxRDtBQUNqRCxlQUFLeUYsZUFBTDtBQUNIOztBQUNELGFBQUtyQixZQUFMLElBQXFCNkQsV0FBckI7QUFFSCxPQU5ELE1BT0s7QUFDRCxhQUFLN0QsWUFBTCxJQUFxQjZELFdBQXJCOztBQUNBLFlBQUksS0FBSzdELFlBQUwsR0FBb0IsS0FBS0UsV0FBN0IsRUFBMEM7QUFDdEMsZUFBS0EsV0FBTCxHQUFtQixLQUFLRixZQUF4QjtBQUNIO0FBQ0o7O0FBQ0RrRCxNQUFBQSxlQUFlLENBQUNILFdBQWhCLEdBQThCQSxXQUE5QjtBQUNBRSxNQUFBQSxVQUFVLENBQUNpQixjQUFYLENBQTBCTCxXQUExQixFQUF1Q0MsWUFBdkM7QUFDQWIsTUFBQUEsVUFBVSxDQUFDaEQsVUFBWCxHQUF3QixLQUFLQSxVQUE3Qjs7QUFFQSxVQUFJNEMsZUFBZSxDQUFDWCxLQUFoQixDQUFzQnpELEtBQTFCLEVBQWlDO0FBQzdCLFlBQUlvRSxlQUFlLENBQUNYLEtBQWhCLENBQXNCekQsS0FBdEIsQ0FBNEIwRixLQUFoQyxFQUF1QztBQUNuQ2xCLFVBQUFBLFVBQVUsQ0FBQ2xFLGFBQVgsR0FBMkI4RCxlQUFlLENBQUNYLEtBQWhCLENBQXNCekQsS0FBdEIsQ0FBNEIwRixLQUF2RDtBQUNIOztBQUNELFlBQUl0QixlQUFlLENBQUNYLEtBQWhCLENBQXNCekQsS0FBdEIsQ0FBNEIyRixLQUFoQyxFQUF1QztBQUNuQ25CLFVBQUFBLFVBQVUsQ0FBQ2hFLFdBQVgsR0FBeUI0RCxlQUFlLENBQUNYLEtBQWhCLENBQXNCekQsS0FBdEIsQ0FBNEIyRixLQUFyRDtBQUNILFNBRkQsTUFHSztBQUNEbkIsVUFBQUEsVUFBVSxDQUFDaEUsV0FBWCxHQUF5QixFQUF6QjtBQUNIO0FBQ0osT0FWRCxNQVdLO0FBQ0RnRSxRQUFBQSxVQUFVLENBQUNsRSxhQUFYLEdBQTJCLElBQTNCO0FBQ0g7QUFDSixLQXhFRCxNQXlFSztBQUNEbEksTUFBQUEsRUFBRSxDQUFDd04sTUFBSCxDQUFVLElBQVY7QUFDSDtBQUNKLEdBOW1CbUI7QUFnbkJwQm5LLEVBQUFBLGVBaG5Cb0IsNkJBZ25CRDtBQUNmLFFBQUksQ0FBQyxLQUFLK0Isa0JBQVYsRUFBOEI7O0FBRTlCLFFBQUk0RixZQUFZLEdBQUd4TCxlQUFlLENBQUNpTyxLQUFoQixDQUFzQixLQUFLN0wsTUFBM0IsQ0FBbkI7O0FBQ0EsUUFBSSxDQUFDLEtBQUttSixzQkFBTCxDQUE0QkMsWUFBNUIsQ0FBTCxFQUFnRDtBQUM1QyxXQUFLakksVUFBTCxHQUFrQmlJLFlBQWxCOztBQUNBLFdBQUt2RSxpQ0FBTDs7QUFDQTtBQUNIOztBQUVELFNBQUsxRCxVQUFMLEdBQWtCaUksWUFBbEI7O0FBQ0EsU0FBS2xDLFdBQUw7O0FBRUEsUUFBSTRFLGFBQWEsR0FBRyxLQUFwQjtBQUNBLFFBQUlwRyxLQUFKO0FBQ0EsUUFBSUksU0FBSjs7QUFFQSxTQUFLLElBQUlLLENBQUMsR0FBRyxDQUFiLEVBQWdCQSxDQUFDLEdBQUcsS0FBS2hGLFVBQUwsQ0FBZ0J3RSxNQUFwQyxFQUE0QyxFQUFFUSxDQUE5QyxFQUFpRDtBQUM3QyxVQUFJaUUsZUFBZSxHQUFHLEtBQUtqSixVQUFMLENBQWdCZ0YsQ0FBaEIsQ0FBdEI7QUFDQSxVQUFJb0QsSUFBSSxHQUFHYSxlQUFlLENBQUNiLElBQTNCLENBRjZDLENBRzdDOztBQUNBLFVBQUlBLElBQUksS0FBSyxFQUFiLEVBQWlCO0FBQ2IsWUFBSWEsZUFBZSxDQUFDWCxLQUFoQixJQUF5QlcsZUFBZSxDQUFDWCxLQUFoQixDQUFzQnNDLE9BQW5ELEVBQTREO0FBQ3hELGVBQUtuRCxlQUFMOztBQUNBO0FBQ0g7O0FBQ0QsWUFBSXdCLGVBQWUsQ0FBQ1gsS0FBaEIsSUFBeUJXLGVBQWUsQ0FBQ1gsS0FBaEIsQ0FBc0JJLE9BQS9DLElBQTBELEtBQUt4RyxVQUFuRSxFQUErRTtBQUMzRSxlQUFLOEcsd0JBQUwsQ0FBOEJDLGVBQTlCOztBQUNBO0FBQ0g7QUFDSjs7QUFDRCxVQUFJNEIsY0FBYyxHQUFHekMsSUFBSSxDQUFDMEMsS0FBTCxDQUFXLElBQVgsQ0FBckI7O0FBRUEsV0FBSyxJQUFJQyxDQUFDLEdBQUcsQ0FBYixFQUFnQkEsQ0FBQyxHQUFHRixjQUFjLENBQUNyRyxNQUFuQyxFQUEyQyxFQUFFdUcsQ0FBN0MsRUFBZ0Q7QUFDNUMsWUFBSWhFLFdBQVcsR0FBRzhELGNBQWMsQ0FBQ0UsQ0FBRCxDQUFoQzs7QUFDQSxZQUFJaEUsV0FBVyxLQUFLLEVBQXBCLEVBQXdCO0FBQ3BCO0FBQ0EsY0FBSSxLQUFLZSxrQkFBTCxDQUF3Qk0sSUFBeEIsS0FDRzJDLENBQUMsS0FBS0YsY0FBYyxDQUFDckcsTUFBZixHQUF3QixDQURyQyxFQUN3QztBQUNwQztBQUNIOztBQUNELGVBQUtpRCxlQUFMOztBQUNBa0QsVUFBQUEsYUFBYSxHQUFHLElBQWhCO0FBQ0E7QUFDSDs7QUFDREEsUUFBQUEsYUFBYSxHQUFHLEtBQWhCOztBQUVBLFlBQUksS0FBSzNJLFFBQUwsR0FBZ0IsQ0FBcEIsRUFBdUI7QUFDbkIsY0FBSWdGLFVBQVUsR0FBRyxLQUFLM0MsWUFBTCxDQUFrQlcsQ0FBbEIsRUFBcUIrQixXQUFyQixDQUFqQjs7QUFDQSxlQUFLRCwyQkFBTCxDQUFpQ0MsV0FBakMsRUFBOENDLFVBQTlDLEVBQTBEaEMsQ0FBMUQ7O0FBRUEsY0FBSTZGLGNBQWMsQ0FBQ3JHLE1BQWYsR0FBd0IsQ0FBeEIsSUFBNkJ1RyxDQUFDLEdBQUdGLGNBQWMsQ0FBQ3JHLE1BQWYsR0FBd0IsQ0FBN0QsRUFBZ0U7QUFDNUQsaUJBQUtpRCxlQUFMO0FBQ0g7QUFDSixTQVBELE1BUUs7QUFDRGxELFVBQUFBLEtBQUssR0FBRyxLQUFLbUMsZ0JBQUwsQ0FBc0JLLFdBQXRCLEVBQW1DL0IsQ0FBbkMsQ0FBUjtBQUNBTCxVQUFBQSxTQUFTLEdBQUdKLEtBQUssQ0FBQ0ssY0FBTixFQUFaO0FBRUEsZUFBS3dCLFlBQUwsSUFBcUJ6QixTQUFTLENBQUNoRyxLQUEvQjs7QUFDQSxjQUFJLEtBQUt5SCxZQUFMLEdBQW9CLEtBQUtFLFdBQTdCLEVBQTBDO0FBQ3RDLGlCQUFLQSxXQUFMLEdBQW1CLEtBQUtGLFlBQXhCO0FBQ0g7O0FBRUQsY0FBSXlFLGNBQWMsQ0FBQ3JHLE1BQWYsR0FBd0IsQ0FBeEIsSUFBNkJ1RyxDQUFDLEdBQUdGLGNBQWMsQ0FBQ3JHLE1BQWYsR0FBd0IsQ0FBN0QsRUFBZ0U7QUFDNUQsaUJBQUtpRCxlQUFMO0FBQ0g7QUFDSjtBQUNKO0FBQ0o7O0FBQ0QsUUFBSSxDQUFDa0QsYUFBTCxFQUFvQjtBQUNoQixXQUFLeEssV0FBTCxDQUFpQnNFLElBQWpCLENBQXNCLEtBQUsyQixZQUEzQjtBQUNIOztBQUVELFFBQUksS0FBS3BFLFFBQUwsR0FBZ0IsQ0FBcEIsRUFBdUI7QUFDbkIsV0FBS3NFLFdBQUwsR0FBbUIsS0FBS3RFLFFBQXhCO0FBQ0g7O0FBQ0QsU0FBS3VFLFlBQUwsR0FBb0IsQ0FBQyxLQUFLRixVQUFMLEdBQWtCOUosU0FBUyxDQUFDeU8sY0FBN0IsSUFBK0MsS0FBSy9JLFVBQXhFLENBN0VlLENBK0VmOztBQUNBLFNBQUtoRSxJQUFMLENBQVVxTSxjQUFWLENBQXlCLEtBQUtoRSxXQUE5QixFQUEyQyxLQUFLQyxZQUFoRDs7QUFFQSxTQUFLMEUsdUJBQUw7O0FBQ0EsU0FBSzlKLFlBQUwsR0FBb0IsS0FBcEI7QUFDSCxHQXBzQm1CO0FBc3NCcEJpRyxFQUFBQSxnQkF0c0JvQiw0QkFzc0JGZ0IsSUF0c0JFLEVBc3NCSThDLFVBdHNCSixFQXNzQmdCQyxPQXRzQmhCLEVBc3NCeUI7QUFDekMsUUFBSUMsU0FBUyxHQUFHaEQsSUFBSSxDQUFDaUQsTUFBTCxDQUFZSCxVQUFaLENBQWhCOztBQUNBLFFBQUkzTyxTQUFTLENBQUMrTyxZQUFWLENBQXVCRixTQUF2QixLQUNHN08sU0FBUyxDQUFDZ1AsY0FBVixDQUF5QkgsU0FBekIsQ0FEUCxFQUM0QztBQUN4QyxhQUFPLENBQVA7QUFDSDs7QUFFRCxRQUFJSSxHQUFHLEdBQUcsQ0FBVjs7QUFDQSxTQUFLLElBQUlDLEtBQUssR0FBR1AsVUFBVSxHQUFHLENBQTlCLEVBQWlDTyxLQUFLLEdBQUdOLE9BQXpDLEVBQWtELEVBQUVNLEtBQXBELEVBQTJEO0FBQ3ZETCxNQUFBQSxTQUFTLEdBQUdoRCxJQUFJLENBQUNpRCxNQUFMLENBQVlJLEtBQVosQ0FBWjs7QUFDQSxVQUFJbFAsU0FBUyxDQUFDZ1AsY0FBVixDQUF5QkgsU0FBekIsS0FDRzdPLFNBQVMsQ0FBQytPLFlBQVYsQ0FBdUJGLFNBQXZCLENBRFAsRUFDMEM7QUFDdEM7QUFDSDs7QUFDREksTUFBQUEsR0FBRztBQUNOOztBQUVELFdBQU9BLEdBQVA7QUFDSCxHQXh0Qm1CO0FBMHRCcEJQLEVBQUFBLHVCQTF0Qm9CLHFDQTB0Qk87QUFDdkIsUUFBSVMsVUFBVSxHQUFHLENBQWpCO0FBQ0EsUUFBSUMsYUFBYSxHQUFHLENBQXBCO0FBQ0EsUUFBSUMsY0FBYyxHQUFHLEtBQUt2RixVQUExQjs7QUFDQSxTQUFLLElBQUlyQixDQUFDLEdBQUcsQ0FBYixFQUFnQkEsQ0FBQyxHQUFHLEtBQUsvRSxjQUFMLENBQW9CdUUsTUFBeEMsRUFBZ0QsRUFBRVEsQ0FBbEQsRUFBcUQ7QUFDakQsVUFBSVQsS0FBSyxHQUFHLEtBQUt0RSxjQUFMLENBQW9CK0UsQ0FBcEIsQ0FBWjtBQUNBLFVBQUk2RyxTQUFTLEdBQUd0SCxLQUFLLENBQUM4QixVQUF0Qjs7QUFDQSxVQUFJd0YsU0FBUyxHQUFHRixhQUFoQixFQUErQjtBQUMzQkQsUUFBQUEsVUFBVSxHQUFHLENBQWI7QUFDQUMsUUFBQUEsYUFBYSxHQUFHRSxTQUFoQjtBQUNIOztBQUNELFVBQUlDLFdBQVcsR0FBRyxDQUFsQixDQVBpRCxDQVFqRDs7QUFDQSxjQUFRLEtBQUt2TSxlQUFiO0FBQ0ksYUFBSzdDLGVBQWUsQ0FBQzhDLElBQXJCO0FBQ0lzTSxVQUFBQSxXQUFXLEdBQUcsQ0FBRSxLQUFLeEYsV0FBUCxHQUFxQixDQUFuQztBQUNBOztBQUNKLGFBQUs1SixlQUFlLENBQUNnRCxNQUFyQjtBQUNJb00sVUFBQUEsV0FBVyxHQUFHLENBQUUsS0FBSzNMLFdBQUwsQ0FBaUIwTCxTQUFTLEdBQUcsQ0FBN0IsQ0FBRixHQUFvQyxDQUFsRDtBQUNBOztBQUNKLGFBQUtuUCxlQUFlLENBQUNxUCxLQUFyQjtBQUNJRCxVQUFBQSxXQUFXLEdBQUcsS0FBS3hGLFdBQUwsR0FBbUIsQ0FBbkIsR0FBdUIsS0FBS25HLFdBQUwsQ0FBaUIwTCxTQUFTLEdBQUcsQ0FBN0IsQ0FBckM7QUFDQTs7QUFDSjtBQUNJO0FBWFI7O0FBYUF0SCxNQUFBQSxLQUFLLENBQUN5SCxDQUFOLEdBQVVOLFVBQVUsR0FBR0ksV0FBdkI7QUFFQSxVQUFJbkgsU0FBUyxHQUFHSixLQUFLLENBQUNLLGNBQU4sRUFBaEI7QUFFQUwsTUFBQUEsS0FBSyxDQUFDMEgsQ0FBTixHQUFVLEtBQUtoSyxVQUFMLElBQW1CMkosY0FBYyxHQUFHQyxTQUFwQyxJQUFpRCxLQUFLdEYsWUFBTCxHQUFvQixDQUEvRTs7QUFFQSxVQUFJc0YsU0FBUyxLQUFLRixhQUFsQixFQUFpQztBQUM3QkQsUUFBQUEsVUFBVSxJQUFJL0csU0FBUyxDQUFDaEcsS0FBeEI7QUFDSDs7QUFFRCxVQUFJdU4sTUFBTSxHQUFHM0gsS0FBSyxDQUFDOUYsWUFBTixDQUFtQnhCLEVBQUUsQ0FBQ3NNLE1BQXRCLENBQWI7O0FBQ0EsVUFBSTJDLE1BQUosRUFBWTtBQUNSO0FBQ0EsWUFBSUMsYUFBYSxHQUFHLEtBQUtsSyxVQUF6QjtBQUNBLFlBQUltSyxjQUFjLEdBQUcsS0FBS25LLFVBQUwsSUFBbUIsSUFBSTFGLFNBQVMsQ0FBQ3lPLGNBQWpDLENBQXJCLENBSFEsQ0FHK0Q7O0FBQ3ZFLGdCQUFRekcsS0FBSyxDQUFDOEgsT0FBZDtBQUVJLGVBQUssQ0FBTDtBQUNJOUgsWUFBQUEsS0FBSyxDQUFDMEgsQ0FBTixJQUFhRSxhQUFhLEdBQUssQ0FBRUMsY0FBYyxHQUFHRCxhQUFuQixJQUFvQyxDQUFuRTtBQUNBOztBQUNKLGVBQUssR0FBTDtBQUNJNUgsWUFBQUEsS0FBSyxDQUFDMEgsQ0FBTixJQUFhRyxjQUFjLEdBQUcsQ0FBOUI7QUFDQTs7QUFDSjtBQUNJN0gsWUFBQUEsS0FBSyxDQUFDMEgsQ0FBTixJQUFhLENBQUNHLGNBQWMsR0FBR0QsYUFBbEIsSUFBbUMsQ0FBaEQ7QUFDQTtBQVZSLFNBSlEsQ0FnQlI7OztBQUNBLFlBQUk1SCxLQUFLLENBQUNpRixZQUFWLEVBQ0E7QUFDSSxjQUFJOEMsT0FBTyxHQUFHL0gsS0FBSyxDQUFDaUYsWUFBTixDQUFtQnNCLEtBQW5CLENBQXlCLEdBQXpCLENBQWQ7O0FBQ0EsY0FBSXdCLE9BQU8sQ0FBQzlILE1BQVIsS0FBbUIsQ0FBbkIsSUFBd0I4SCxPQUFPLENBQUMsQ0FBRCxDQUFuQyxFQUNBO0FBQ0ksZ0JBQUlDLE9BQU8sR0FBR0MsVUFBVSxDQUFDRixPQUFPLENBQUMsQ0FBRCxDQUFSLENBQXhCO0FBQ0EsZ0JBQUlHLE1BQU0sQ0FBQ0MsU0FBUCxDQUFpQkgsT0FBakIsQ0FBSixFQUErQmhJLEtBQUssQ0FBQzBILENBQU4sSUFBV00sT0FBWDtBQUNsQyxXQUpELE1BS0ssSUFBR0QsT0FBTyxDQUFDOUgsTUFBUixLQUFtQixDQUF0QixFQUNMO0FBQ0ksZ0JBQUltSSxPQUFPLEdBQUdILFVBQVUsQ0FBQ0YsT0FBTyxDQUFDLENBQUQsQ0FBUixDQUF4Qjs7QUFDQSxnQkFBSUMsUUFBTyxHQUFHQyxVQUFVLENBQUNGLE9BQU8sQ0FBQyxDQUFELENBQVIsQ0FBeEI7O0FBQ0EsZ0JBQUlHLE1BQU0sQ0FBQ0MsU0FBUCxDQUFpQkMsT0FBakIsQ0FBSixFQUErQnBJLEtBQUssQ0FBQ3lILENBQU4sSUFBV1csT0FBWDtBQUMvQixnQkFBSUYsTUFBTSxDQUFDQyxTQUFQLENBQWlCSCxRQUFqQixDQUFKLEVBQStCaEksS0FBSyxDQUFDMEgsQ0FBTixJQUFXTSxRQUFYO0FBQ2xDO0FBQ0o7QUFDSixPQWxFZ0QsQ0FvRWpEOzs7QUFDQSxVQUFJL04sT0FBTyxHQUFHK0YsS0FBSyxDQUFDOUYsWUFBTixDQUFtQnhCLEVBQUUsQ0FBQ3lCLFlBQXRCLENBQWQ7QUFDQSxVQUFJRixPQUFPLElBQUlBLE9BQU8sQ0FBQ0csS0FBdkIsRUFBOEI0RixLQUFLLENBQUMwSCxDQUFOLEdBQVUxSCxLQUFLLENBQUMwSCxDQUFOLEdBQVV6TixPQUFPLENBQUNHLEtBQTVCO0FBQ2pDO0FBQ0osR0F0eUJtQjtBQXd5QnBCaU8sRUFBQUEseUJBeHlCb0IscUNBd3lCTzFKLEtBeHlCUCxFQXd5QmM7QUFDOUIsUUFBSTJKLFVBQVUsR0FBRzNKLEtBQUssQ0FBQzRKLFdBQU4sRUFBakI7O0FBQ0EsUUFBSTdQLEVBQUUsQ0FBQzhQLEtBQUgsQ0FBU0YsVUFBVCxDQUFKLEVBQTBCO0FBQ3RCLGFBQU81UCxFQUFFLENBQUM4UCxLQUFILENBQVNGLFVBQVQsQ0FBUDtBQUNILEtBRkQsTUFHSztBQUNELFVBQUlHLEdBQUcsR0FBRy9QLEVBQUUsQ0FBQ2lHLEtBQUgsRUFBVjtBQUNBLGFBQU84SixHQUFHLENBQUNDLE9BQUosQ0FBWS9KLEtBQVosQ0FBUDtBQUNIO0FBQ0osR0FqekJtQjtBQW16QnBCO0FBQ0FVLEVBQUFBLG1CQXB6Qm9CLCtCQW96QkM3RSxTQXB6QkQsRUFvekJZRixNQXB6QlosRUFvekJvQnFPLEtBcHpCcEIsRUFvekIyQjtBQUMzQyxRQUFJN04sY0FBYyxHQUFHTixTQUFTLENBQUNOLFlBQVYsQ0FBdUJ4QixFQUFFLENBQUNDLEtBQTFCLENBQXJCOztBQUNBLFFBQUksQ0FBQ21DLGNBQUwsRUFBcUI7QUFDakI7QUFDSDs7QUFFRCxRQUFJb00sS0FBSyxHQUFHMU0sU0FBUyxDQUFDMkYsV0FBdEI7QUFFQSxRQUFJeUksU0FBUyxHQUFHLElBQWhCOztBQUNBLFFBQUksS0FBS25OLFVBQUwsQ0FBZ0J5TCxLQUFoQixDQUFKLEVBQTRCO0FBQ3hCMEIsTUFBQUEsU0FBUyxHQUFHLEtBQUtuTixVQUFMLENBQWdCeUwsS0FBaEIsRUFBdUJuRCxLQUFuQztBQUNIOztBQUVELFFBQUk2RSxTQUFTLElBQUlBLFNBQVMsQ0FBQ2pLLEtBQTNCLEVBQWtDO0FBQzlCbkUsTUFBQUEsU0FBUyxDQUFDbUUsS0FBVixHQUFrQixLQUFLMEoseUJBQUwsQ0FBK0JPLFNBQVMsQ0FBQ2pLLEtBQXpDLENBQWxCO0FBQ0gsS0FGRCxNQUVNO0FBQ0ZuRSxNQUFBQSxTQUFTLENBQUNtRSxLQUFWLEdBQWtCLEtBQUtqRixJQUFMLENBQVVpRixLQUE1QjtBQUNIOztBQUVEN0QsSUFBQUEsY0FBYyxDQUFDeUMsU0FBZixHQUEyQixLQUFLQSxTQUFoQztBQUVBLFFBQUlzTCxPQUFPLEdBQUcsS0FBSzNMLElBQUwsWUFBcUJ4RSxFQUFFLENBQUNvUSxJQUF0Qzs7QUFDQSxRQUFJRCxPQUFPLElBQUksQ0FBQyxLQUFLdkwsaUJBQXJCLEVBQXdDO0FBQ3BDeEMsTUFBQUEsY0FBYyxDQUFDb0MsSUFBZixHQUFzQixLQUFLQSxJQUEzQjtBQUNILEtBRkQsTUFFTztBQUNIcEMsTUFBQUEsY0FBYyxDQUFDaUMsVUFBZixHQUE0QixLQUFLQSxVQUFqQztBQUNIOztBQUVEakMsSUFBQUEsY0FBYyxDQUFDc0MsYUFBZixHQUErQixLQUFLRSxpQkFBcEM7QUFDQXhDLElBQUFBLGNBQWMsQ0FBQzRDLFVBQWYsR0FBNEIsS0FBS0EsVUFBakM7QUFDQTVDLElBQUFBLGNBQWMsQ0FBQ2lPLFVBQWYsR0FBNEJILFNBQVMsSUFBSUEsU0FBUyxDQUFDSSxJQUFuRDtBQUNBbE8sSUFBQUEsY0FBYyxDQUFDbU8sYUFBZixHQUErQkwsU0FBUyxJQUFJQSxTQUFTLENBQUMxRSxNQUF0RCxDQS9CMkMsQ0FnQzNDOztBQUNBLFFBQUkwRSxTQUFTLElBQUlBLFNBQVMsQ0FBQzFFLE1BQTNCLEVBQW1DO0FBQy9CMUosTUFBQUEsU0FBUyxDQUFDSyxLQUFWLEdBQWtCLEVBQWxCO0FBQ0g7O0FBRURDLElBQUFBLGNBQWMsQ0FBQ29PLGVBQWYsR0FBaUNOLFNBQVMsSUFBSUEsU0FBUyxDQUFDTyxTQUF4RDs7QUFFQSxRQUFJUCxTQUFTLElBQUlBLFNBQVMsQ0FBQzNPLE9BQTNCLEVBQW9DO0FBQ2hDLFVBQUltUCxxQkFBcUIsR0FBRzVPLFNBQVMsQ0FBQ04sWUFBVixDQUF1QnhCLEVBQUUsQ0FBQ3lCLFlBQTFCLENBQTVCOztBQUNBLFVBQUksQ0FBQ2lQLHFCQUFMLEVBQTRCO0FBQ3hCQSxRQUFBQSxxQkFBcUIsR0FBRzVPLFNBQVMsQ0FBQ08sWUFBVixDQUF1QnJDLEVBQUUsQ0FBQ3lCLFlBQTFCLENBQXhCO0FBQ0g7O0FBQ0RpUCxNQUFBQSxxQkFBcUIsQ0FBQ3pLLEtBQXRCLEdBQThCLEtBQUswSix5QkFBTCxDQUErQk8sU0FBUyxDQUFDM08sT0FBVixDQUFrQjBFLEtBQWpELENBQTlCO0FBQ0F5SyxNQUFBQSxxQkFBcUIsQ0FBQ2hQLEtBQXRCLEdBQThCd08sU0FBUyxDQUFDM08sT0FBVixDQUFrQkcsS0FBaEQ7QUFDSDs7QUFFRCxRQUFJd08sU0FBUyxJQUFJQSxTQUFTLENBQUMzRSxJQUEzQixFQUFpQztBQUM3Qm5KLE1BQUFBLGNBQWMsQ0FBQytCLFFBQWYsR0FBMEIrTCxTQUFTLENBQUMzRSxJQUFwQztBQUNILEtBRkQsTUFHSztBQUNEbkosTUFBQUEsY0FBYyxDQUFDK0IsUUFBZixHQUEwQixLQUFLQSxRQUEvQjtBQUNIOztBQUVELFFBQUl2QyxNQUFNLEtBQUssSUFBZixFQUFxQjtBQUNqQixVQUFJLE9BQU9BLE1BQVAsS0FBa0IsUUFBdEIsRUFBZ0M7QUFDNUJBLFFBQUFBLE1BQU0sR0FBRyxLQUFLQSxNQUFkO0FBQ0g7O0FBQ0RRLE1BQUFBLGNBQWMsQ0FBQ1IsTUFBZixHQUF3QkEsTUFBeEI7QUFDSDs7QUFFRHFPLElBQUFBLEtBQUssSUFBSTdOLGNBQWMsQ0FBQ3VPLHNCQUFmLEVBQVQ7O0FBRUEsUUFBSVQsU0FBUyxJQUFJQSxTQUFTLENBQUN0SSxLQUEzQixFQUFrQztBQUM5QixVQUFJc0ksU0FBUyxDQUFDdEksS0FBVixDQUFnQjBGLEtBQXBCLEVBQTJCO0FBQ3ZCeEwsUUFBQUEsU0FBUyxDQUFDb0csYUFBVixHQUEwQmdJLFNBQVMsQ0FBQ3RJLEtBQVYsQ0FBZ0IwRixLQUExQztBQUNIOztBQUNELFVBQUk0QyxTQUFTLENBQUN0SSxLQUFWLENBQWdCMkYsS0FBcEIsRUFBMkI7QUFDdkJ6TCxRQUFBQSxTQUFTLENBQUNzRyxXQUFWLEdBQXdCOEgsU0FBUyxDQUFDdEksS0FBVixDQUFnQjJGLEtBQXhDO0FBQ0gsT0FGRCxNQUdLO0FBQ0R6TCxRQUFBQSxTQUFTLENBQUNzRyxXQUFWLEdBQXdCLEVBQXhCO0FBQ0g7QUFDSixLQVZELE1BV0s7QUFDRHRHLE1BQUFBLFNBQVMsQ0FBQ29HLGFBQVYsR0FBMEIsSUFBMUI7QUFDSDtBQUNKLEdBbDRCbUI7QUFvNEJwQjBJLEVBQUFBLFNBcDRCb0IsdUJBbzRCUDtBQUNULFNBQUssSUFBSTdJLENBQUMsR0FBRyxDQUFiLEVBQWdCQSxDQUFDLEdBQUcsS0FBSy9FLGNBQUwsQ0FBb0J1RSxNQUF4QyxFQUFnRCxFQUFFUSxDQUFsRCxFQUFxRDtBQUNqRCxXQUFLL0UsY0FBTCxDQUFvQitFLENBQXBCLEVBQXVCOEksZ0JBQXZCOztBQUNBL1AsTUFBQUEsSUFBSSxDQUFDb0ksR0FBTCxDQUFTLEtBQUtsRyxjQUFMLENBQW9CK0UsQ0FBcEIsQ0FBVDtBQUNIO0FBQ0o7QUF6NEJtQixDQUFULENBQWY7QUE0NEJBL0gsRUFBRSxDQUFDMEMsUUFBSCxHQUFjb08sTUFBTSxDQUFDQyxPQUFQLEdBQWlCck8sUUFBL0IiLCJzb3VyY2VzQ29udGVudCI6WyIvKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKlxuIENvcHlyaWdodCAoYykgMjAxMy0yMDE2IENodWtvbmcgVGVjaG5vbG9naWVzIEluYy5cbiBDb3B5cmlnaHQgKGMpIDIwMTctMjAxOCBYaWFtZW4gWWFqaSBTb2Z0d2FyZSBDby4sIEx0ZC5cblxuIGh0dHBzOi8vd3d3LmNvY29zLmNvbS9cblxuIFBlcm1pc3Npb24gaXMgaGVyZWJ5IGdyYW50ZWQsIGZyZWUgb2YgY2hhcmdlLCB0byBhbnkgcGVyc29uIG9idGFpbmluZyBhIGNvcHlcbiBvZiB0aGlzIHNvZnR3YXJlIGFuZCBhc3NvY2lhdGVkIGVuZ2luZSBzb3VyY2UgY29kZSAodGhlIFwiU29mdHdhcmVcIiksIGEgbGltaXRlZCxcbiAgd29ybGR3aWRlLCByb3lhbHR5LWZyZWUsIG5vbi1hc3NpZ25hYmxlLCByZXZvY2FibGUgYW5kIG5vbi1leGNsdXNpdmUgbGljZW5zZVxuIHRvIHVzZSBDb2NvcyBDcmVhdG9yIHNvbGVseSB0byBkZXZlbG9wIGdhbWVzIG9uIHlvdXIgdGFyZ2V0IHBsYXRmb3Jtcy4gWW91IHNoYWxsXG4gIG5vdCB1c2UgQ29jb3MgQ3JlYXRvciBzb2Z0d2FyZSBmb3IgZGV2ZWxvcGluZyBvdGhlciBzb2Z0d2FyZSBvciB0b29scyB0aGF0J3NcbiAgdXNlZCBmb3IgZGV2ZWxvcGluZyBnYW1lcy4gWW91IGFyZSBub3QgZ3JhbnRlZCB0byBwdWJsaXNoLCBkaXN0cmlidXRlLFxuICBzdWJsaWNlbnNlLCBhbmQvb3Igc2VsbCBjb3BpZXMgb2YgQ29jb3MgQ3JlYXRvci5cblxuIFRoZSBzb2Z0d2FyZSBvciB0b29scyBpbiB0aGlzIExpY2Vuc2UgQWdyZWVtZW50IGFyZSBsaWNlbnNlZCwgbm90IHNvbGQuXG4gWGlhbWVuIFlhamkgU29mdHdhcmUgQ28uLCBMdGQuIHJlc2VydmVzIGFsbCByaWdodHMgbm90IGV4cHJlc3NseSBncmFudGVkIHRvIHlvdS5cblxuIFRIRSBTT0ZUV0FSRSBJUyBQUk9WSURFRCBcIkFTIElTXCIsIFdJVEhPVVQgV0FSUkFOVFkgT0YgQU5ZIEtJTkQsIEVYUFJFU1MgT1JcbiBJTVBMSUVELCBJTkNMVURJTkcgQlVUIE5PVCBMSU1JVEVEIFRPIFRIRSBXQVJSQU5USUVTIE9GIE1FUkNIQU5UQUJJTElUWSxcbiBGSVRORVNTIEZPUiBBIFBBUlRJQ1VMQVIgUFVSUE9TRSBBTkQgTk9OSU5GUklOR0VNRU5ULiBJTiBOTyBFVkVOVCBTSEFMTCBUSEVcbiBBVVRIT1JTIE9SIENPUFlSSUdIVCBIT0xERVJTIEJFIExJQUJMRSBGT1IgQU5ZIENMQUlNLCBEQU1BR0VTIE9SIE9USEVSXG4gTElBQklMSVRZLCBXSEVUSEVSIElOIEFOIEFDVElPTiBPRiBDT05UUkFDVCwgVE9SVCBPUiBPVEhFUldJU0UsIEFSSVNJTkcgRlJPTSxcbiBPVVQgT0YgT1IgSU4gQ09OTkVDVElPTiBXSVRIIFRIRSBTT0ZUV0FSRSBPUiBUSEUgVVNFIE9SIE9USEVSIERFQUxJTkdTIElOXG4gVEhFIFNPRlRXQVJFLlxuICoqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKiovXG5cbmNvbnN0IGpzID0gcmVxdWlyZSgnLi4vcGxhdGZvcm0vanMnKTtcbmNvbnN0IG1hY3JvID0gcmVxdWlyZSgnLi4vcGxhdGZvcm0vQ0NNYWNybycpO1xuY29uc3QgdGV4dFV0aWxzID0gcmVxdWlyZSgnLi4vdXRpbHMvdGV4dC11dGlscycpO1xuY29uc3QgSHRtbFRleHRQYXJzZXIgPSByZXF1aXJlKCcuLi91dGlscy9odG1sLXRleHQtcGFyc2VyJyk7XG5jb25zdCBfaHRtbFRleHRQYXJzZXIgPSBuZXcgSHRtbFRleHRQYXJzZXIoKTtcblxuY29uc3QgSG9yaXpvbnRhbEFsaWduID0gbWFjcm8uVGV4dEFsaWdubWVudDtcbmNvbnN0IFZlcnRpY2FsQWxpZ24gPSBtYWNyby5WZXJ0aWNhbFRleHRBbGlnbm1lbnQ7XG5jb25zdCBSaWNoVGV4dENoaWxkTmFtZSA9IFwiUklDSFRFWFRfQ0hJTERcIjtcbmNvbnN0IFJpY2hUZXh0Q2hpbGRJbWFnZU5hbWUgPSBcIlJJQ0hURVhUX0ltYWdlX0NISUxEXCI7XG5jb25zdCBDYWNoZU1vZGUgPSBjYy5MYWJlbC5DYWNoZU1vZGU7XG5cbi8vIFJldHVybnMgYSBmdW5jdGlvbiwgdGhhdCwgYXMgbG9uZyBhcyBpdCBjb250aW51ZXMgdG8gYmUgaW52b2tlZCwgd2lsbCBub3Rcbi8vIGJlIHRyaWdnZXJlZC4gVGhlIGZ1bmN0aW9uIHdpbGwgYmUgY2FsbGVkIGFmdGVyIGl0IHN0b3BzIGJlaW5nIGNhbGxlZCBmb3Jcbi8vIE4gbWlsbGlzZWNvbmRzLiBJZiBgaW1tZWRpYXRlYCBpcyBwYXNzZWQsIHRyaWdnZXIgdGhlIGZ1bmN0aW9uIG9uIHRoZVxuLy8gbGVhZGluZyBlZGdlLCBpbnN0ZWFkIG9mIHRoZSB0cmFpbGluZy5cbmZ1bmN0aW9uIGRlYm91bmNlKGZ1bmMsIHdhaXQsIGltbWVkaWF0ZSkge1xuICAgIGxldCB0aW1lb3V0O1xuICAgIHJldHVybiBmdW5jdGlvbiAoKSB7XG4gICAgICAgIGxldCBjb250ZXh0ID0gdGhpcztcbiAgICAgICAgbGV0IGxhdGVyID0gZnVuY3Rpb24gKCkge1xuICAgICAgICAgICAgdGltZW91dCA9IG51bGw7XG4gICAgICAgICAgICBpZiAoIWltbWVkaWF0ZSkgZnVuYy5hcHBseShjb250ZXh0LCBhcmd1bWVudHMpO1xuICAgICAgICB9O1xuICAgICAgICBsZXQgY2FsbE5vdyA9IGltbWVkaWF0ZSAmJiAhdGltZW91dDtcbiAgICAgICAgY2xlYXJUaW1lb3V0KHRpbWVvdXQpO1xuICAgICAgICB0aW1lb3V0ID0gc2V0VGltZW91dChsYXRlciwgd2FpdCk7XG4gICAgICAgIGlmIChjYWxsTm93KSBmdW5jLmFwcGx5KGNvbnRleHQsIGFyZ3VtZW50cyk7XG4gICAgfTtcbn1cblxuLyoqXG4gKiBSaWNoVGV4dCBwb29sXG4gKi9cbmxldCBwb29sID0gbmV3IGpzLlBvb2woZnVuY3Rpb24gKG5vZGUpIHtcbiAgICBpZiAoQ0NfRURJVE9SKSB7XG4gICAgICAgIGNjLmlzVmFsaWQobm9kZSkgJiYgbm9kZS5kZXN0cm95KCk7XG4gICAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9XG4gICAgaWYgKENDX0RFVikge1xuICAgICAgICBjYy5hc3NlcnQoIW5vZGUuX3BhcmVudCwgJ1JlY3ljbGluZyBub2RlXFwncyBwYXJlbnQgc2hvdWxkIGJlIG51bGwhJyk7XG4gICAgfVxuICAgIGlmICghY2MuaXNWYWxpZChub2RlKSkge1xuICAgICAgICByZXR1cm4gZmFsc2U7XG4gICAgfSBlbHNlIHtcbiAgICAgICAgbGV0IG91dGxpbmUgPSBub2RlLmdldENvbXBvbmVudChjYy5MYWJlbE91dGxpbmUpO1xuICAgICAgICBpZiAob3V0bGluZSkge1xuICAgICAgICAgICAgb3V0bGluZS53aWR0aCA9IDA7XG4gICAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gdHJ1ZTtcbn0sIDIwKTtcblxucG9vbC5nZXQgPSBmdW5jdGlvbiAoc3RyaW5nLCByaWNodGV4dCkge1xuICAgIGxldCBsYWJlbE5vZGUgPSB0aGlzLl9nZXQoKTtcbiAgICBpZiAoIWxhYmVsTm9kZSkge1xuICAgICAgICBsYWJlbE5vZGUgPSBuZXcgY2MuUHJpdmF0ZU5vZGUoUmljaFRleHRDaGlsZE5hbWUpO1xuICAgIH1cblxuICAgIGxhYmVsTm9kZS5zZXRQb3NpdGlvbigwLCAwKTtcbiAgICBsYWJlbE5vZGUuc2V0QW5jaG9yUG9pbnQoMC41LCAwLjUpO1xuICAgIGxhYmVsTm9kZS5za2V3WCA9IDA7XG5cbiAgICBsZXQgbGFiZWxDb21wb25lbnQgPSBsYWJlbE5vZGUuZ2V0Q29tcG9uZW50KGNjLkxhYmVsKTtcbiAgICBpZiAoIWxhYmVsQ29tcG9uZW50KSB7XG4gICAgICAgIGxhYmVsQ29tcG9uZW50ID0gbGFiZWxOb2RlLmFkZENvbXBvbmVudChjYy5MYWJlbCk7XG4gICAgfVxuXG4gICAgbGFiZWxDb21wb25lbnQuc3RyaW5nID0gXCJcIjtcbiAgICBsYWJlbENvbXBvbmVudC5ob3Jpem9udGFsQWxpZ24gPSBIb3Jpem9udGFsQWxpZ24uTEVGVDtcbiAgICBsYWJlbENvbXBvbmVudC52ZXJ0aWNhbEFsaWduID0gVmVydGljYWxBbGlnbi5DRU5URVI7XG5cbiAgICByZXR1cm4gbGFiZWxOb2RlO1xufTtcblxuLyoqXG4gKiAhI2VuIFRoZSBSaWNoVGV4dCBDb21wb25lbnQuXG4gKiAhI3poIOWvjOaWh+acrOe7hOS7tlxuICogQGNsYXNzIFJpY2hUZXh0XG4gKiBAZXh0ZW5kcyBDb21wb25lbnRcbiAqL1xubGV0IFJpY2hUZXh0ID0gY2MuQ2xhc3Moe1xuICAgIG5hbWU6ICdjYy5SaWNoVGV4dCcsXG4gICAgZXh0ZW5kczogY2MuQ29tcG9uZW50LFxuXG4gICAgY3RvcjogZnVuY3Rpb24gKCkge1xuICAgICAgICB0aGlzLl90ZXh0QXJyYXkgPSBudWxsO1xuICAgICAgICB0aGlzLl9sYWJlbFNlZ21lbnRzID0gW107XG4gICAgICAgIHRoaXMuX2xhYmVsU2VnbWVudHNDYWNoZSA9IFtdO1xuICAgICAgICB0aGlzLl9saW5lc1dpZHRoID0gW107XG5cbiAgICAgICAgaWYgKENDX0VESVRPUikge1xuICAgICAgICAgICAgdGhpcy5fdXNlckRlZmluZWRGb250ID0gbnVsbDtcbiAgICAgICAgICAgIHRoaXMuX3VwZGF0ZVJpY2hUZXh0U3RhdHVzID0gZGVib3VuY2UodGhpcy5fdXBkYXRlUmljaFRleHQsIDIwMCk7XG4gICAgICAgIH1cbiAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICB0aGlzLl91cGRhdGVSaWNoVGV4dFN0YXR1cyA9IHRoaXMuX3VwZGF0ZVJpY2hUZXh0O1xuICAgICAgICB9XG4gICAgfSxcblxuICAgIGVkaXRvcjogQ0NfRURJVE9SICYmIHtcbiAgICAgICAgbWVudTogJ2kxOG46TUFJTl9NRU5VLmNvbXBvbmVudC5yZW5kZXJlcnMvUmljaFRleHQnLFxuICAgICAgICBoZWxwOiAnaTE4bjpDT01QT05FTlQuaGVscF91cmwucmljaHRleHQnLFxuICAgICAgICBpbnNwZWN0b3I6ICdwYWNrYWdlczovL2luc3BlY3Rvci9pbnNwZWN0b3JzL2NvbXBzL3JpY2h0ZXh0LmpzJyxcbiAgICAgICAgZXhlY3V0ZUluRWRpdE1vZGU6IHRydWVcbiAgICB9LFxuXG4gICAgcHJvcGVydGllczoge1xuICAgICAgICAvKipcbiAgICAgICAgICogISNlbiBDb250ZW50IHN0cmluZyBvZiBSaWNoVGV4dC5cbiAgICAgICAgICogISN6aCDlr4zmlofmnKzmmL7npLrnmoTmlofmnKzlhoXlrrnjgIJcbiAgICAgICAgICogQHByb3BlcnR5IHtTdHJpbmd9IHN0cmluZ1xuICAgICAgICAgKi9cbiAgICAgICAgc3RyaW5nOiB7XG4gICAgICAgICAgICBkZWZhdWx0OiAnPGNvbG9yPSMwMGZmMDA+UmljaDwvYz48Y29sb3I9IzBmZmZmZj5UZXh0PC9jb2xvcj4nLFxuICAgICAgICAgICAgbXVsdGlsaW5lOiB0cnVlLFxuICAgICAgICAgICAgdG9vbHRpcDogQ0NfREVWICYmICdpMThuOkNPTVBPTkVOVC5yaWNodGV4dC5zdHJpbmcnLFxuICAgICAgICAgICAgbm90aWZ5OiBmdW5jdGlvbiAoKSB7XG4gICAgICAgICAgICAgICAgdGhpcy5fdXBkYXRlUmljaFRleHRTdGF0dXMoKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfSxcblxuICAgICAgICAvKipcbiAgICAgICAgICogISNlbiBIb3Jpem9udGFsIEFsaWdubWVudCBvZiBlYWNoIGxpbmUgaW4gUmljaFRleHQuXG4gICAgICAgICAqICEjemgg5paH5pys5YaF5a6555qE5rC05bmz5a+56b2Q5pa55byP44CCXG4gICAgICAgICAqIEBwcm9wZXJ0eSB7bWFjcm8uVGV4dEFsaWdubWVudH0gaG9yaXpvbnRhbEFsaWduXG4gICAgICAgICAqL1xuICAgICAgICBob3Jpem9udGFsQWxpZ246IHtcbiAgICAgICAgICAgIGRlZmF1bHQ6IEhvcml6b250YWxBbGlnbi5MRUZULFxuICAgICAgICAgICAgdHlwZTogSG9yaXpvbnRhbEFsaWduLFxuICAgICAgICAgICAgdG9vbHRpcDogQ0NfREVWICYmICdpMThuOkNPTVBPTkVOVC5yaWNodGV4dC5ob3Jpem9udGFsX2FsaWduJyxcbiAgICAgICAgICAgIGFuaW1hdGFibGU6IGZhbHNlLFxuICAgICAgICAgICAgbm90aWZ5OiBmdW5jdGlvbiAob2xkVmFsdWUpIHtcbiAgICAgICAgICAgICAgICBpZiAodGhpcy5ob3Jpem9udGFsQWxpZ24gPT09IG9sZFZhbHVlKSByZXR1cm47XG5cbiAgICAgICAgICAgICAgICB0aGlzLl9sYXlvdXREaXJ0eSA9IHRydWU7XG4gICAgICAgICAgICAgICAgdGhpcy5fdXBkYXRlUmljaFRleHRTdGF0dXMoKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfSxcblxuICAgICAgICAvKipcbiAgICAgICAgICogISNlbiBGb250IHNpemUgb2YgUmljaFRleHQuXG4gICAgICAgICAqICEjemgg5a+M5paH5pys5a2X5L2T5aSn5bCP44CCXG4gICAgICAgICAqIEBwcm9wZXJ0eSB7TnVtYmVyfSBmb250U2l6ZVxuICAgICAgICAgKi9cbiAgICAgICAgZm9udFNpemU6IHtcbiAgICAgICAgICAgIGRlZmF1bHQ6IDQwLFxuICAgICAgICAgICAgdG9vbHRpcDogQ0NfREVWICYmICdpMThuOkNPTVBPTkVOVC5yaWNodGV4dC5mb250X3NpemUnLFxuICAgICAgICAgICAgbm90aWZ5OiBmdW5jdGlvbiAob2xkVmFsdWUpIHtcbiAgICAgICAgICAgICAgICBpZiAodGhpcy5mb250U2l6ZSA9PT0gb2xkVmFsdWUpIHJldHVybjtcblxuICAgICAgICAgICAgICAgIHRoaXMuX2xheW91dERpcnR5ID0gdHJ1ZTtcbiAgICAgICAgICAgICAgICB0aGlzLl91cGRhdGVSaWNoVGV4dFN0YXR1cygpO1xuICAgICAgICAgICAgfVxuICAgICAgICB9LFxuXG4gICAgICAgIC8qKlxuICAgICAgICAgKiAhI2VuIEN1c3RvbSBTeXN0ZW0gZm9udCBvZiBSaWNoVGV4dFxuICAgICAgICAgKiAhI3poIOWvjOaWh+acrOWumuWItuezu+e7n+Wtl+S9k1xuICAgICAgICAgKiBAcHJvcGVydHkge1N0cmluZ30gZm9udEZhbWlseVxuICAgICAgICAgKi9cbiAgICAgICAgX2ZvbnRGYW1pbHk6IFwiQXJpYWxcIixcbiAgICAgICAgZm9udEZhbWlseToge1xuICAgICAgICAgICAgdG9vbHRpcDogQ0NfREVWICYmICdpMThuOkNPTVBPTkVOVC5yaWNodGV4dC5mb250X2ZhbWlseScsXG4gICAgICAgICAgICBnZXQgKCkge1xuICAgICAgICAgICAgICAgIHJldHVybiB0aGlzLl9mb250RmFtaWx5O1xuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHNldCAodmFsdWUpIHtcbiAgICAgICAgICAgICAgICBpZiAodGhpcy5fZm9udEZhbWlseSA9PT0gdmFsdWUpIHJldHVybjtcbiAgICAgICAgICAgICAgICB0aGlzLl9mb250RmFtaWx5ID0gdmFsdWU7XG4gICAgICAgICAgICAgICAgdGhpcy5fbGF5b3V0RGlydHkgPSB0cnVlO1xuICAgICAgICAgICAgICAgIHRoaXMuX3VwZGF0ZVJpY2hUZXh0U3RhdHVzKCk7XG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgYW5pbWF0YWJsZTogZmFsc2VcbiAgICAgICAgfSxcblxuICAgICAgICAvKipcbiAgICAgICAgICogISNlbiBDdXN0b20gVFRGIGZvbnQgb2YgUmljaFRleHRcbiAgICAgICAgICogISN6aCAg5a+M5paH5pys5a6a5Yi25a2X5L2TXG4gICAgICAgICAqIEBwcm9wZXJ0eSB7Y2MuVFRGRm9udH0gZm9udFxuICAgICAgICAgKi9cbiAgICAgICAgZm9udDoge1xuICAgICAgICAgICAgZGVmYXVsdDogbnVsbCxcbiAgICAgICAgICAgIHR5cGU6IGNjLlRURkZvbnQsXG4gICAgICAgICAgICB0b29sdGlwOiBDQ19ERVYgJiYgJ2kxOG46Q09NUE9ORU5ULnJpY2h0ZXh0LmZvbnQnLFxuICAgICAgICAgICAgbm90aWZ5OiBmdW5jdGlvbiAob2xkVmFsdWUpIHtcbiAgICAgICAgICAgICAgICBpZiAodGhpcy5mb250ID09PSBvbGRWYWx1ZSkgcmV0dXJuO1xuXG4gICAgICAgICAgICAgICAgdGhpcy5fbGF5b3V0RGlydHkgPSB0cnVlO1xuICAgICAgICAgICAgICAgIGlmICh0aGlzLmZvbnQpIHtcbiAgICAgICAgICAgICAgICAgICAgaWYgKENDX0VESVRPUikge1xuICAgICAgICAgICAgICAgICAgICAgICAgdGhpcy5fdXNlckRlZmluZWRGb250ID0gdGhpcy5mb250O1xuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgICAgIHRoaXMudXNlU3lzdGVtRm9udCA9IGZhbHNlO1xuICAgICAgICAgICAgICAgICAgICB0aGlzLl9vblRURkxvYWRlZCgpO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgICAgICAgICAgdGhpcy51c2VTeXN0ZW1Gb250ID0gdHJ1ZTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgdGhpcy5fdXBkYXRlUmljaFRleHRTdGF0dXMoKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfSxcblxuICAgICAgICAvKipcbiAgICAgICAgICogISNlbiBXaGV0aGVyIHVzZSBzeXN0ZW0gZm9udCBuYW1lIG9yIG5vdC5cbiAgICAgICAgICogISN6aCDmmK/lkKbkvb/nlKjns7vnu5/lrZfkvZPjgIJcbiAgICAgICAgICogQHByb3BlcnR5IHtCb29sZWFufSB1c2VTeXN0ZW1Gb250XG4gICAgICAgICAqL1xuICAgICAgICBfaXNTeXN0ZW1Gb250VXNlZDogdHJ1ZSxcbiAgICAgICAgdXNlU3lzdGVtRm9udDoge1xuICAgICAgICAgICAgZ2V0ICgpIHtcbiAgICAgICAgICAgICAgICByZXR1cm4gdGhpcy5faXNTeXN0ZW1Gb250VXNlZDtcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBzZXQgKHZhbHVlKSB7XG4gICAgICAgICAgICAgICAgaWYgKHRoaXMuX2lzU3lzdGVtRm9udFVzZWQgPT09IHZhbHVlKSB7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybjtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgdGhpcy5faXNTeXN0ZW1Gb250VXNlZCA9IHZhbHVlO1xuXG4gICAgICAgICAgICAgICAgaWYgKENDX0VESVRPUikge1xuICAgICAgICAgICAgICAgICAgICBpZiAodmFsdWUpIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIHRoaXMuZm9udCA9IG51bGw7XG4gICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgZWxzZSBpZiAodGhpcy5fdXNlckRlZmluZWRGb250KSB7XG4gICAgICAgICAgICAgICAgICAgICAgICB0aGlzLmZvbnQgPSB0aGlzLl91c2VyRGVmaW5lZEZvbnQ7XG4gICAgICAgICAgICAgICAgICAgICAgICByZXR1cm47XG4gICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICB9XG5cbiAgICAgICAgICAgICAgICB0aGlzLl9sYXlvdXREaXJ0eSA9IHRydWU7XG4gICAgICAgICAgICAgICAgdGhpcy5fdXBkYXRlUmljaFRleHRTdGF0dXMoKTtcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBhbmltYXRhYmxlOiBmYWxzZSxcbiAgICAgICAgICAgIHRvb2x0aXA6IENDX0RFViAmJiAnaTE4bjpDT01QT05FTlQucmljaHRleHQuc3lzdGVtX2ZvbnQnLFxuICAgICAgICB9LFxuXG4gICAgICAgIC8qKlxuICAgICAgICAgKiAhI2VuIFRoZSBjYWNoZSBtb2RlIG9mIGxhYmVsLiBUaGlzIG1vZGUgb25seSBzdXBwb3J0cyBzeXN0ZW0gZm9udHMuXG4gICAgICAgICAqICEjemgg5paH5pys57yT5a2Y5qih5byPLCDor6XmqKHlvI/lj6rmlK/mjIHns7vnu5/lrZfkvZPjgIJcbiAgICAgICAgICogQHByb3BlcnR5IHtMYWJlbC5DYWNoZU1vZGV9IGNhY2hlTW9kZVxuICAgICAgICAgKi9cbiAgICAgICAgY2FjaGVNb2RlOiB7XG4gICAgICAgICAgICBkZWZhdWx0OiBDYWNoZU1vZGUuTk9ORSxcbiAgICAgICAgICAgIHR5cGU6IENhY2hlTW9kZSxcbiAgICAgICAgICAgIHRvb2x0aXA6IENDX0RFViAmJiAnaTE4bjpDT01QT05FTlQubGFiZWwuY2FjaGVNb2RlJyxcbiAgICAgICAgICAgIG5vdGlmeSAob2xkVmFsdWUpIHtcbiAgICAgICAgICAgICAgICBpZiAodGhpcy5jYWNoZU1vZGUgPT09IG9sZFZhbHVlKSByZXR1cm47XG5cbiAgICAgICAgICAgICAgICB0aGlzLl91cGRhdGVSaWNoVGV4dFN0YXR1cygpO1xuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIGFuaW1hdGFibGU6IGZhbHNlXG4gICAgICAgIH0sXG5cbiAgICAgICAgLyoqXG4gICAgICAgICAqICEjZW4gVGhlIG1heGltaXplIHdpZHRoIG9mIHRoZSBSaWNoVGV4dFxuICAgICAgICAgKiAhI3poIOWvjOaWh+acrOeahOacgOWkp+WuveW6plxuICAgICAgICAgKiBAcHJvcGVydHkge051bWJlcn0gbWF4V2lkdGhcbiAgICAgICAgICovXG4gICAgICAgIG1heFdpZHRoOiB7XG4gICAgICAgICAgICBkZWZhdWx0OiAwLFxuICAgICAgICAgICAgdG9vbHRpcDogQ0NfREVWICYmICdpMThuOkNPTVBPTkVOVC5yaWNodGV4dC5tYXhfd2lkdGgnLFxuICAgICAgICAgICAgbm90aWZ5OiBmdW5jdGlvbiAob2xkVmFsdWUpIHtcbiAgICAgICAgICAgICAgICBpZiAodGhpcy5tYXhXaWR0aCA9PT0gb2xkVmFsdWUpIHJldHVybjtcblxuICAgICAgICAgICAgICAgIHRoaXMuX2xheW91dERpcnR5ID0gdHJ1ZTtcbiAgICAgICAgICAgICAgICB0aGlzLl91cGRhdGVSaWNoVGV4dFN0YXR1cygpO1xuICAgICAgICAgICAgfVxuICAgICAgICB9LFxuXG4gICAgICAgIC8qKlxuICAgICAgICAgKiAhI2VuIExpbmUgSGVpZ2h0IG9mIFJpY2hUZXh0LlxuICAgICAgICAgKiAhI3poIOWvjOaWh+acrOihjOmrmOOAglxuICAgICAgICAgKiBAcHJvcGVydHkge051bWJlcn0gbGluZUhlaWdodFxuICAgICAgICAgKi9cbiAgICAgICAgbGluZUhlaWdodDoge1xuICAgICAgICAgICAgZGVmYXVsdDogNDAsXG4gICAgICAgICAgICB0b29sdGlwOiBDQ19ERVYgJiYgJ2kxOG46Q09NUE9ORU5ULnJpY2h0ZXh0LmxpbmVfaGVpZ2h0JyxcbiAgICAgICAgICAgIG5vdGlmeTogZnVuY3Rpb24gKG9sZFZhbHVlKSB7XG4gICAgICAgICAgICAgICAgaWYgKHRoaXMubGluZUhlaWdodCA9PT0gb2xkVmFsdWUpIHJldHVybjtcblxuICAgICAgICAgICAgICAgIHRoaXMuX2xheW91dERpcnR5ID0gdHJ1ZTtcbiAgICAgICAgICAgICAgICB0aGlzLl91cGRhdGVSaWNoVGV4dFN0YXR1cygpO1xuICAgICAgICAgICAgfVxuICAgICAgICB9LFxuXG4gICAgICAgIC8qKlxuICAgICAgICAgKiAhI2VuIFRoZSBpbWFnZSBhdGxhcyBmb3IgdGhlIGltZyB0YWcuIEZvciBlYWNoIHNyYyB2YWx1ZSBpbiB0aGUgaW1nIHRhZywgdGhlcmUgc2hvdWxkIGJlIGEgdmFsaWQgc3ByaXRlRnJhbWUgaW4gdGhlIGltYWdlIGF0bGFzLlxuICAgICAgICAgKiAhI3poIOWvueS6jiBpbWcg5qCH562+6YeM6Z2i55qEIHNyYyDlsZ7mgKflkI3np7DvvIzpg73pnIDopoHlnKggaW1hZ2VBdGxhcyDph4zpnaLmib7liLDkuIDkuKrmnInmlYjnmoQgc3ByaXRlRnJhbWXvvIzlkKbliJkgaW1nIHRhZyDkvJrliKTlrprkuLrml6DmlYjjgIJcbiAgICAgICAgICogQHByb3BlcnR5IHtTcHJpdGVBdGxhc30gaW1hZ2VBdGxhc1xuICAgICAgICAgKi9cbiAgICAgICAgaW1hZ2VBdGxhczoge1xuICAgICAgICAgICAgZGVmYXVsdDogbnVsbCxcbiAgICAgICAgICAgIHR5cGU6IGNjLlNwcml0ZUF0bGFzLFxuICAgICAgICAgICAgdG9vbHRpcDogQ0NfREVWICYmICdpMThuOkNPTVBPTkVOVC5yaWNodGV4dC5pbWFnZV9hdGxhcycsXG4gICAgICAgICAgICBub3RpZnk6IGZ1bmN0aW9uIChvbGRWYWx1ZSkge1xuICAgICAgICAgICAgICAgIGlmICh0aGlzLmltYWdlQXRsYXMgPT09IG9sZFZhbHVlKSByZXR1cm47XG5cbiAgICAgICAgICAgICAgICB0aGlzLl9sYXlvdXREaXJ0eSA9IHRydWU7XG4gICAgICAgICAgICAgICAgdGhpcy5fdXBkYXRlUmljaFRleHRTdGF0dXMoKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfSxcblxuICAgICAgICAvKipcbiAgICAgICAgICogISNlblxuICAgICAgICAgKiBPbmNlIGNoZWNrZWQsIHRoZSBSaWNoVGV4dCB3aWxsIGJsb2NrIGFsbCBpbnB1dCBldmVudHMgKG1vdXNlIGFuZCB0b3VjaCkgd2l0aGluXG4gICAgICAgICAqIHRoZSBib3VuZGluZyBib3ggb2YgdGhlIG5vZGUsIHByZXZlbnRpbmcgdGhlIGlucHV0IGZyb20gcGVuZXRyYXRpbmcgaW50byB0aGUgdW5kZXJseWluZyBub2RlLlxuICAgICAgICAgKiAhI3poXG4gICAgICAgICAqIOmAieS4reatpOmAiemhueWQju+8jFJpY2hUZXh0IOWwhumYu+atouiKgueCuei+ueeVjOahhuS4reeahOaJgOaciei+k+WFpeS6i+S7tu+8iOm8oOagh+WSjOinpuaRuO+8ie+8jOS7juiAjOmYsuatoui+k+WFpeS6i+S7tuepv+mAj+WIsOW6leWxguiKgueCueOAglxuICAgICAgICAgKiBAcHJvcGVydHkge0Jvb2xlYW59IGhhbmRsZVRvdWNoRXZlbnRcbiAgICAgICAgICogQGRlZmF1bHQgdHJ1ZVxuICAgICAgICAgKi9cbiAgICAgICAgaGFuZGxlVG91Y2hFdmVudDoge1xuICAgICAgICAgICAgZGVmYXVsdDogdHJ1ZSxcbiAgICAgICAgICAgIHRvb2x0aXA6IENDX0RFViAmJiAnaTE4bjpDT01QT05FTlQucmljaHRleHQuaGFuZGxlVG91Y2hFdmVudCcsXG4gICAgICAgICAgICBub3RpZnk6IGZ1bmN0aW9uIChvbGRWYWx1ZSkge1xuICAgICAgICAgICAgICAgIGlmICh0aGlzLmhhbmRsZVRvdWNoRXZlbnQgPT09IG9sZFZhbHVlKSByZXR1cm47XG4gICAgICAgICAgICAgICAgaWYgKHRoaXMuZW5hYmxlZEluSGllcmFyY2h5KSB7XG4gICAgICAgICAgICAgICAgICAgIHRoaXMuaGFuZGxlVG91Y2hFdmVudCA/IHRoaXMuX2FkZEV2ZW50TGlzdGVuZXJzKCkgOiB0aGlzLl9yZW1vdmVFdmVudExpc3RlbmVycygpO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgIH0sXG5cbiAgICBzdGF0aWNzOiB7XG4gICAgICAgIEhvcml6b250YWxBbGlnbjogSG9yaXpvbnRhbEFsaWduLFxuICAgICAgICBWZXJ0aWNhbEFsaWduOiBWZXJ0aWNhbEFsaWduXG4gICAgfSxcblxuICAgIG9uRW5hYmxlICgpIHtcbiAgICAgICAgaWYgKHRoaXMuaGFuZGxlVG91Y2hFdmVudCkge1xuICAgICAgICAgICAgdGhpcy5fYWRkRXZlbnRMaXN0ZW5lcnMoKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLl91cGRhdGVSaWNoVGV4dCgpO1xuICAgICAgICB0aGlzLl9hY3RpdmF0ZUNoaWxkcmVuKHRydWUpO1xuICAgIH0sXG5cbiAgICBvbkRpc2FibGUgKCkge1xuICAgICAgICBpZiAodGhpcy5oYW5kbGVUb3VjaEV2ZW50KSB7XG4gICAgICAgICAgICB0aGlzLl9yZW1vdmVFdmVudExpc3RlbmVycygpO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMuX2FjdGl2YXRlQ2hpbGRyZW4oZmFsc2UpO1xuICAgIH0sXG5cbiAgICBzdGFydCAoKSB7XG4gICAgICAgIHRoaXMuX29uVFRGTG9hZGVkKCk7XG4gICAgfSxcblxuICAgIF9vbkNvbG9yQ2hhbmdlZCAocGFyZW50Q29sb3IpIHtcbiAgICAgICAgbGV0IGNoaWxkcmVuID0gdGhpcy5ub2RlLmNoaWxkcmVuO1xuICAgICAgICBjaGlsZHJlbi5mb3JFYWNoKGZ1bmN0aW9uIChjaGlsZE5vZGUpIHtcbiAgICAgICAgICAgIGNoaWxkTm9kZS5jb2xvciA9IHBhcmVudENvbG9yO1xuICAgICAgICB9KTtcbiAgICB9LFxuXG4gICAgX2FkZEV2ZW50TGlzdGVuZXJzICgpIHtcbiAgICAgICAgdGhpcy5ub2RlLm9uKGNjLk5vZGUuRXZlbnRUeXBlLlRPVUNIX0VORCwgdGhpcy5fb25Ub3VjaEVuZGVkLCB0aGlzKTtcbiAgICAgICAgdGhpcy5ub2RlLm9uKGNjLk5vZGUuRXZlbnRUeXBlLkNPTE9SX0NIQU5HRUQsIHRoaXMuX29uQ29sb3JDaGFuZ2VkLCB0aGlzKTtcbiAgICB9LFxuXG4gICAgX3JlbW92ZUV2ZW50TGlzdGVuZXJzICgpIHtcbiAgICAgICAgdGhpcy5ub2RlLm9mZihjYy5Ob2RlLkV2ZW50VHlwZS5UT1VDSF9FTkQsIHRoaXMuX29uVG91Y2hFbmRlZCwgdGhpcyk7XG4gICAgICAgIHRoaXMubm9kZS5vZmYoY2MuTm9kZS5FdmVudFR5cGUuQ09MT1JfQ0hBTkdFRCwgdGhpcy5fb25Db2xvckNoYW5nZWQsIHRoaXMpO1xuICAgIH0sXG5cbiAgICBfdXBkYXRlTGFiZWxTZWdtZW50VGV4dEF0dHJpYnV0ZXMgKCkge1xuICAgICAgICB0aGlzLl9sYWJlbFNlZ21lbnRzLmZvckVhY2goZnVuY3Rpb24gKGl0ZW0pIHtcbiAgICAgICAgICAgIHRoaXMuX2FwcGx5VGV4dEF0dHJpYnV0ZShpdGVtLCBudWxsLCB0cnVlKTtcbiAgICAgICAgfS5iaW5kKHRoaXMpKTtcbiAgICB9LFxuXG4gICAgX2NyZWF0ZUZvbnRMYWJlbCAoc3RyaW5nKSB7XG4gICAgICAgIHJldHVybiBwb29sLmdldChzdHJpbmcsIHRoaXMpO1xuICAgIH0sXG5cbiAgICBfb25UVEZMb2FkZWQgKCkge1xuICAgICAgICBpZiAodGhpcy5mb250IGluc3RhbmNlb2YgY2MuVFRGRm9udCkge1xuICAgICAgICAgICAgaWYgKHRoaXMuZm9udC5fbmF0aXZlQXNzZXQpIHtcbiAgICAgICAgICAgICAgICB0aGlzLl9sYXlvdXREaXJ0eSA9IHRydWU7XG4gICAgICAgICAgICAgICAgdGhpcy5fdXBkYXRlUmljaFRleHQoKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGVsc2Uge1xuICAgICAgICAgICAgICAgIGxldCBzZWxmID0gdGhpcztcbiAgICAgICAgICAgICAgICBjYy5sb2FkZXIubG9hZCh0aGlzLmZvbnQubmF0aXZlVXJsLCBmdW5jdGlvbiAoZXJyLCBmb250RmFtaWx5KSB7XG4gICAgICAgICAgICAgICAgICAgIHNlbGYuX2xheW91dERpcnR5ID0gdHJ1ZTtcbiAgICAgICAgICAgICAgICAgICAgc2VsZi5fdXBkYXRlUmljaFRleHQoKTtcbiAgICAgICAgICAgICAgICB9KTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIHRoaXMuX2xheW91dERpcnR5ID0gdHJ1ZTtcbiAgICAgICAgICAgIHRoaXMuX3VwZGF0ZVJpY2hUZXh0KCk7XG4gICAgICAgIH1cbiAgICB9LFxuXG4gICAgX21lYXN1cmVUZXh0IChzdHlsZUluZGV4LCBzdHJpbmcpIHtcbiAgICAgICAgbGV0IHNlbGYgPSB0aGlzO1xuICAgICAgICBsZXQgZnVuYyA9IGZ1bmN0aW9uIChzdHJpbmcpIHtcbiAgICAgICAgICAgIGxldCBsYWJlbDtcbiAgICAgICAgICAgIGlmIChzZWxmLl9sYWJlbFNlZ21lbnRzQ2FjaGUubGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgICAgICAgbGFiZWwgPSBzZWxmLl9jcmVhdGVGb250TGFiZWwoc3RyaW5nKTtcbiAgICAgICAgICAgICAgICBzZWxmLl9sYWJlbFNlZ21lbnRzQ2FjaGUucHVzaChsYWJlbCk7XG4gICAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgICAgIGxhYmVsID0gc2VsZi5fbGFiZWxTZWdtZW50c0NhY2hlWzBdO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgbGFiZWwuX3N0eWxlSW5kZXggPSBzdHlsZUluZGV4O1xuICAgICAgICAgICAgc2VsZi5fYXBwbHlUZXh0QXR0cmlidXRlKGxhYmVsLCBzdHJpbmcsIHRydWUpO1xuICAgICAgICAgICAgbGV0IGxhYmVsU2l6ZSA9IGxhYmVsLmdldENvbnRlbnRTaXplKCk7XG4gICAgICAgICAgICByZXR1cm4gbGFiZWxTaXplLndpZHRoO1xuICAgICAgICB9O1xuICAgICAgICBpZiAoc3RyaW5nKSB7XG4gICAgICAgICAgICByZXR1cm4gZnVuYyhzdHJpbmcpO1xuICAgICAgICB9XG4gICAgICAgIGVsc2Uge1xuICAgICAgICAgICAgcmV0dXJuIGZ1bmM7XG4gICAgICAgIH1cbiAgICB9LFxuXG4gICAgX29uVG91Y2hFbmRlZCAoZXZlbnQpIHtcbiAgICAgICAgbGV0IGNvbXBvbmVudHMgPSB0aGlzLm5vZGUuZ2V0Q29tcG9uZW50cyhjYy5Db21wb25lbnQpO1xuXG4gICAgICAgIGZvciAobGV0IGkgPSAwOyBpIDwgdGhpcy5fbGFiZWxTZWdtZW50cy5sZW5ndGg7ICsraSkge1xuICAgICAgICAgICAgbGV0IGxhYmVsU2VnbWVudCA9IHRoaXMuX2xhYmVsU2VnbWVudHNbaV07XG4gICAgICAgICAgICBsZXQgY2xpY2tIYW5kbGVyID0gbGFiZWxTZWdtZW50Ll9jbGlja0hhbmRsZXI7XG4gICAgICAgICAgICBsZXQgY2xpY2tQYXJhbSA9IGxhYmVsU2VnbWVudC5fY2xpY2tQYXJhbTtcbiAgICAgICAgICAgIGlmIChjbGlja0hhbmRsZXIgJiYgdGhpcy5fY29udGFpbnNUb3VjaExvY2F0aW9uKGxhYmVsU2VnbWVudCwgZXZlbnQudG91Y2guZ2V0TG9jYXRpb24oKSkpIHtcbiAgICAgICAgICAgICAgICBjb21wb25lbnRzLmZvckVhY2goZnVuY3Rpb24gKGNvbXBvbmVudCkge1xuICAgICAgICAgICAgICAgICAgICBpZiAoY29tcG9uZW50LmVuYWJsZWRJbkhpZXJhcmNoeSAmJiBjb21wb25lbnRbY2xpY2tIYW5kbGVyXSkge1xuICAgICAgICAgICAgICAgICAgICAgICAgY29tcG9uZW50W2NsaWNrSGFuZGxlcl0oZXZlbnQsIGNsaWNrUGFyYW0pO1xuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICAgICAgZXZlbnQuc3RvcFByb3BhZ2F0aW9uKCk7XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICB9LFxuXG4gICAgX2NvbnRhaW5zVG91Y2hMb2NhdGlvbiAobGFiZWwsIHBvaW50KSB7XG4gICAgICAgIGxldCBteVJlY3QgPSBsYWJlbC5nZXRCb3VuZGluZ0JveFRvV29ybGQoKTtcbiAgICAgICAgcmV0dXJuIG15UmVjdC5jb250YWlucyhwb2ludCk7XG4gICAgfSxcblxuICAgIF9yZXNldFN0YXRlICgpIHtcbiAgICAgICAgbGV0IGNoaWxkcmVuID0gdGhpcy5ub2RlLmNoaWxkcmVuO1xuICAgICAgICBmb3IgKGxldCBpID0gY2hpbGRyZW4ubGVuZ3RoIC0gMTsgaSA+PSAwOyBpLS0pIHtcbiAgICAgICAgICAgIGxldCBjaGlsZCA9IGNoaWxkcmVuW2ldO1xuICAgICAgICAgICAgaWYgKGNoaWxkLm5hbWUgPT09IFJpY2hUZXh0Q2hpbGROYW1lIHx8IGNoaWxkLm5hbWUgPT09IFJpY2hUZXh0Q2hpbGRJbWFnZU5hbWUpIHtcbiAgICAgICAgICAgICAgICBpZiAoY2hpbGQucGFyZW50ID09PSB0aGlzLm5vZGUpIHtcbiAgICAgICAgICAgICAgICAgICAgY2hpbGQucGFyZW50ID0gbnVsbDtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICAgICAgICAgIC8vIEluIGNhc2UgY2hpbGQucGFyZW50ICE9PSB0aGlzLm5vZGUsIGNoaWxkIGNhbm5vdCBiZSByZW1vdmVkIGZyb20gY2hpbGRyZW5cbiAgICAgICAgICAgICAgICAgICAgY2hpbGRyZW4uc3BsaWNlKGksIDEpO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBpZiAoY2hpbGQubmFtZSA9PT0gUmljaFRleHRDaGlsZE5hbWUpIHtcbiAgICAgICAgICAgICAgICAgICAgcG9vbC5wdXQoY2hpbGQpO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuX2xhYmVsU2VnbWVudHMubGVuZ3RoID0gMDtcbiAgICAgICAgdGhpcy5fbGFiZWxTZWdtZW50c0NhY2hlLmxlbmd0aCA9IDA7XG4gICAgICAgIHRoaXMuX2xpbmVzV2lkdGgubGVuZ3RoID0gMDtcbiAgICAgICAgdGhpcy5fbGluZU9mZnNldFggPSAwO1xuICAgICAgICB0aGlzLl9saW5lQ291bnQgPSAxO1xuICAgICAgICB0aGlzLl9sYWJlbFdpZHRoID0gMDtcbiAgICAgICAgdGhpcy5fbGFiZWxIZWlnaHQgPSAwO1xuICAgICAgICB0aGlzLl9sYXlvdXREaXJ0eSA9IHRydWU7XG4gICAgfSxcblxuICAgIG9uUmVzdG9yZTogQ0NfRURJVE9SICYmIGZ1bmN0aW9uICgpIHtcbiAgICAgICAgLy8gVE9ETzogcmVmaW5lIHVuZG8vcmVkbyBzeXN0ZW1cbiAgICAgICAgLy8gQmVjYXVzZSB1bmRvL3JlZG8gd2lsbCBub3QgY2FsbCBvbkVuYWJsZS9vbkRpc2FibGUsXG4gICAgICAgIC8vIHdlIG5lZWQgY2FsbCBvbkVuYWJsZS9vbkRpc2FibGUgbWFudWFsbHkgdG8gYWN0aXZlL2Rpc2FjdGl2ZSBjaGlsZHJlbiBub2Rlcy5cbiAgICAgICAgaWYgKHRoaXMuZW5hYmxlZEluSGllcmFyY2h5KSB7XG4gICAgICAgICAgICB0aGlzLm9uRW5hYmxlKCk7XG4gICAgICAgIH1cbiAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICB0aGlzLm9uRGlzYWJsZSgpO1xuICAgICAgICB9XG4gICAgfSxcblxuICAgIF9hY3RpdmF0ZUNoaWxkcmVuIChhY3RpdmUpIHtcbiAgICAgICAgZm9yIChsZXQgaSA9IHRoaXMubm9kZS5jaGlsZHJlbi5sZW5ndGggLSAxOyBpID49IDA7IGktLSkge1xuICAgICAgICAgICAgbGV0IGNoaWxkID0gdGhpcy5ub2RlLmNoaWxkcmVuW2ldO1xuICAgICAgICAgICAgaWYgKGNoaWxkLm5hbWUgPT09IFJpY2hUZXh0Q2hpbGROYW1lIHx8IGNoaWxkLm5hbWUgPT09IFJpY2hUZXh0Q2hpbGRJbWFnZU5hbWUpIHtcbiAgICAgICAgICAgICAgICBjaGlsZC5hY3RpdmUgPSBhY3RpdmU7XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICB9LFxuXG4gICAgX2FkZExhYmVsU2VnbWVudCAoc3RyaW5nVG9rZW4sIHN0eWxlSW5kZXgpIHtcbiAgICAgICAgbGV0IGxhYmVsU2VnbWVudDtcbiAgICAgICAgaWYgKHRoaXMuX2xhYmVsU2VnbWVudHNDYWNoZS5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICAgIGxhYmVsU2VnbWVudCA9IHRoaXMuX2NyZWF0ZUZvbnRMYWJlbChzdHJpbmdUb2tlbik7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICBsYWJlbFNlZ21lbnQgPSB0aGlzLl9sYWJlbFNlZ21lbnRzQ2FjaGUucG9wKCk7XG4gICAgICAgIH1cbiAgICAgICAgbGFiZWxTZWdtZW50Ll9zdHlsZUluZGV4ID0gc3R5bGVJbmRleDtcbiAgICAgICAgbGFiZWxTZWdtZW50Ll9saW5lQ291bnQgPSB0aGlzLl9saW5lQ291bnQ7XG4gICAgICAgIGxhYmVsU2VnbWVudC5hY3RpdmUgPSB0aGlzLm5vZGUuYWN0aXZlO1xuXG4gICAgICAgIGxhYmVsU2VnbWVudC5zZXRBbmNob3JQb2ludCgwLCAwKTtcbiAgICAgICAgdGhpcy5fYXBwbHlUZXh0QXR0cmlidXRlKGxhYmVsU2VnbWVudCwgc3RyaW5nVG9rZW4pO1xuXG4gICAgICAgIHRoaXMubm9kZS5hZGRDaGlsZChsYWJlbFNlZ21lbnQpO1xuICAgICAgICB0aGlzLl9sYWJlbFNlZ21lbnRzLnB1c2gobGFiZWxTZWdtZW50KTtcblxuICAgICAgICByZXR1cm4gbGFiZWxTZWdtZW50O1xuICAgIH0sXG5cbiAgICBfdXBkYXRlUmljaFRleHRXaXRoTWF4V2lkdGggKGxhYmVsU3RyaW5nLCBsYWJlbFdpZHRoLCBzdHlsZUluZGV4KSB7XG4gICAgICAgIGxldCBmcmFnbWVudFdpZHRoID0gbGFiZWxXaWR0aDtcbiAgICAgICAgbGV0IGxhYmVsU2VnbWVudDtcblxuICAgICAgICBpZiAodGhpcy5fbGluZU9mZnNldFggPiAwICYmIGZyYWdtZW50V2lkdGggKyB0aGlzLl9saW5lT2Zmc2V0WCA+IHRoaXMubWF4V2lkdGgpIHtcbiAgICAgICAgICAgIC8vY29uY2F0IHByZXZpb3VzIGxpbmVcbiAgICAgICAgICAgIGxldCBjaGVja1N0YXJ0SW5kZXggPSAwO1xuICAgICAgICAgICAgd2hpbGUgKHRoaXMuX2xpbmVPZmZzZXRYIDw9IHRoaXMubWF4V2lkdGgpIHtcbiAgICAgICAgICAgICAgICBsZXQgY2hlY2tFbmRJbmRleCA9IHRoaXMuX2dldEZpcnN0V29yZExlbihsYWJlbFN0cmluZyxcbiAgICAgICAgICAgICAgICAgICAgY2hlY2tTdGFydEluZGV4LFxuICAgICAgICAgICAgICAgICAgICBsYWJlbFN0cmluZy5sZW5ndGgpO1xuICAgICAgICAgICAgICAgIGxldCBjaGVja1N0cmluZyA9IGxhYmVsU3RyaW5nLnN1YnN0cihjaGVja1N0YXJ0SW5kZXgsIGNoZWNrRW5kSW5kZXgpO1xuICAgICAgICAgICAgICAgIGxldCBjaGVja1N0cmluZ1dpZHRoID0gdGhpcy5fbWVhc3VyZVRleHQoc3R5bGVJbmRleCwgY2hlY2tTdHJpbmcpO1xuXG4gICAgICAgICAgICAgICAgaWYgKHRoaXMuX2xpbmVPZmZzZXRYICsgY2hlY2tTdHJpbmdXaWR0aCA8PSB0aGlzLm1heFdpZHRoKSB7XG4gICAgICAgICAgICAgICAgICAgIHRoaXMuX2xpbmVPZmZzZXRYICs9IGNoZWNrU3RyaW5nV2lkdGg7XG4gICAgICAgICAgICAgICAgICAgIGNoZWNrU3RhcnRJbmRleCArPSBjaGVja0VuZEluZGV4O1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBlbHNlIHtcblxuICAgICAgICAgICAgICAgICAgICBpZiAoY2hlY2tTdGFydEluZGV4ID4gMCkge1xuICAgICAgICAgICAgICAgICAgICAgICAgbGV0IHJlbWFpbmluZ1N0cmluZyA9IGxhYmVsU3RyaW5nLnN1YnN0cigwLCBjaGVja1N0YXJ0SW5kZXgpO1xuICAgICAgICAgICAgICAgICAgICAgICAgdGhpcy5fYWRkTGFiZWxTZWdtZW50KHJlbWFpbmluZ1N0cmluZywgc3R5bGVJbmRleCk7XG4gICAgICAgICAgICAgICAgICAgICAgICBsYWJlbFN0cmluZyA9IGxhYmVsU3RyaW5nLnN1YnN0cihjaGVja1N0YXJ0SW5kZXgsIGxhYmVsU3RyaW5nLmxlbmd0aCk7XG4gICAgICAgICAgICAgICAgICAgICAgICBmcmFnbWVudFdpZHRoID0gdGhpcy5fbWVhc3VyZVRleHQoc3R5bGVJbmRleCwgbGFiZWxTdHJpbmcpO1xuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgICAgIHRoaXMuX3VwZGF0ZUxpbmVJbmZvKCk7XG4gICAgICAgICAgICAgICAgICAgIGJyZWFrO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICBpZiAoZnJhZ21lbnRXaWR0aCA+IHRoaXMubWF4V2lkdGgpIHtcbiAgICAgICAgICAgIGxldCBmcmFnbWVudHMgPSB0ZXh0VXRpbHMuZnJhZ21lbnRUZXh0KGxhYmVsU3RyaW5nLFxuICAgICAgICAgICAgICAgIGZyYWdtZW50V2lkdGgsXG4gICAgICAgICAgICAgICAgdGhpcy5tYXhXaWR0aCxcbiAgICAgICAgICAgICAgICB0aGlzLl9tZWFzdXJlVGV4dChzdHlsZUluZGV4KSk7XG4gICAgICAgICAgICBmb3IgKGxldCBrID0gMDsgayA8IGZyYWdtZW50cy5sZW5ndGg7ICsraykge1xuICAgICAgICAgICAgICAgIGxldCBzcGxpdFN0cmluZyA9IGZyYWdtZW50c1trXTtcbiAgICAgICAgICAgICAgICBsYWJlbFNlZ21lbnQgPSB0aGlzLl9hZGRMYWJlbFNlZ21lbnQoc3BsaXRTdHJpbmcsIHN0eWxlSW5kZXgpO1xuICAgICAgICAgICAgICAgIGxldCBsYWJlbFNpemUgPSBsYWJlbFNlZ21lbnQuZ2V0Q29udGVudFNpemUoKTtcbiAgICAgICAgICAgICAgICB0aGlzLl9saW5lT2Zmc2V0WCArPSBsYWJlbFNpemUud2lkdGg7XG4gICAgICAgICAgICAgICAgaWYgKGZyYWdtZW50cy5sZW5ndGggPiAxICYmIGsgPCBmcmFnbWVudHMubGVuZ3RoIC0gMSkge1xuICAgICAgICAgICAgICAgICAgICB0aGlzLl91cGRhdGVMaW5lSW5mbygpO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIHRoaXMuX2xpbmVPZmZzZXRYICs9IGZyYWdtZW50V2lkdGg7XG4gICAgICAgICAgICB0aGlzLl9hZGRMYWJlbFNlZ21lbnQobGFiZWxTdHJpbmcsIHN0eWxlSW5kZXgpO1xuICAgICAgICB9XG4gICAgfSxcblxuICAgIF9pc0xhc3RDb21wb25lbnRDUiAoc3RyaW5nVG9rZW4pIHtcbiAgICAgICAgcmV0dXJuIHN0cmluZ1Rva2VuLmxlbmd0aCAtIDEgPT09IHN0cmluZ1Rva2VuLmxhc3RJbmRleE9mKFwiXFxuXCIpO1xuICAgIH0sXG5cbiAgICBfdXBkYXRlTGluZUluZm8gKCkge1xuICAgICAgICB0aGlzLl9saW5lc1dpZHRoLnB1c2godGhpcy5fbGluZU9mZnNldFgpO1xuICAgICAgICB0aGlzLl9saW5lT2Zmc2V0WCA9IDA7XG4gICAgICAgIHRoaXMuX2xpbmVDb3VudCsrO1xuICAgIH0sXG5cbiAgICBfbmVlZHNVcGRhdGVUZXh0TGF5b3V0IChuZXdUZXh0QXJyYXkpIHtcbiAgICAgICAgaWYgKHRoaXMuX2xheW91dERpcnR5IHx8ICF0aGlzLl90ZXh0QXJyYXkgfHwgIW5ld1RleHRBcnJheSkge1xuICAgICAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAodGhpcy5fdGV4dEFycmF5Lmxlbmd0aCAhPT0gbmV3VGV4dEFycmF5Lmxlbmd0aCkge1xuICAgICAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICAgIH1cblxuICAgICAgICBmb3IgKGxldCBpID0gMDsgaSA8IHRoaXMuX3RleHRBcnJheS5sZW5ndGg7ICsraSkge1xuICAgICAgICAgICAgbGV0IG9sZEl0ZW0gPSB0aGlzLl90ZXh0QXJyYXlbaV07XG4gICAgICAgICAgICBsZXQgbmV3SXRlbSA9IG5ld1RleHRBcnJheVtpXTtcbiAgICAgICAgICAgIGlmIChvbGRJdGVtLnRleHQgIT09IG5ld0l0ZW0udGV4dCkge1xuICAgICAgICAgICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICAgICAgbGV0IG9sZFN0eWxlID0gb2xkSXRlbS5zdHlsZSwgbmV3U3R5bGUgPSBuZXdJdGVtLnN0eWxlO1xuICAgICAgICAgICAgICAgIGlmIChvbGRTdHlsZSkge1xuICAgICAgICAgICAgICAgICAgICBpZiAobmV3U3R5bGUpIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIGlmICghb2xkU3R5bGUub3V0bGluZSAhPT0gIW5ld1N0eWxlLm91dGxpbmUpIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICByZXR1cm4gdHJ1ZTtcbiAgICAgICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgIGlmIChvbGRTdHlsZS5zaXplICE9PSBuZXdTdHlsZS5zaXplXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgfHwgIW9sZFN0eWxlLml0YWxpYyAhPT0gIW5ld1N0eWxlLml0YWxpY1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIHx8IG9sZFN0eWxlLmlzSW1hZ2UgIT09IG5ld1N0eWxlLmlzSW1hZ2UpIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICByZXR1cm4gdHJ1ZTtcbiAgICAgICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgIGlmIChvbGRTdHlsZS5zcmMgIT09IG5ld1N0eWxlLnNyYyB8fFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgIG9sZFN0eWxlLmltYWdlQWxpZ24gIT09IG5ld1N0eWxlLmltYWdlQWxpZ24gfHxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBvbGRTdHlsZS5pbWFnZUhlaWdodCAhPT0gbmV3U3R5bGUuaW1hZ2VIZWlnaHQgfHxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBvbGRTdHlsZS5pbWFnZVdpZHRoICE9PSBuZXdTdHlsZS5pbWFnZVdpZHRoIHx8XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgb2xkU3R5bGUuaW1hZ2VPZmZzZXQgIT09IG5ld1N0eWxlLmltYWdlT2Zmc2V0KSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICBpZiAob2xkU3R5bGUuc2l6ZSB8fCBvbGRTdHlsZS5pdGFsaWMgfHwgb2xkU3R5bGUuaXNJbWFnZSB8fCBvbGRTdHlsZS5vdXRsaW5lKSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICAgICAgICAgIGlmIChuZXdTdHlsZSkge1xuICAgICAgICAgICAgICAgICAgICAgICAgaWYgKG5ld1N0eWxlLnNpemUgfHwgbmV3U3R5bGUuaXRhbGljIHx8IG5ld1N0eWxlLmlzSW1hZ2UgfHwgbmV3U3R5bGUub3V0bGluZSkge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9LFxuXG4gICAgX2FkZFJpY2hUZXh0SW1hZ2VFbGVtZW50IChyaWNoVGV4dEVsZW1lbnQpIHtcbiAgICAgICAgbGV0IHNwcml0ZUZyYW1lTmFtZSA9IHJpY2hUZXh0RWxlbWVudC5zdHlsZS5zcmM7XG4gICAgICAgIGxldCBzcHJpdGVGcmFtZSA9IHRoaXMuaW1hZ2VBdGxhcy5nZXRTcHJpdGVGcmFtZShzcHJpdGVGcmFtZU5hbWUpO1xuICAgICAgICBpZiAoc3ByaXRlRnJhbWUpIHtcbiAgICAgICAgICAgIGxldCBzcHJpdGVOb2RlID0gbmV3IGNjLlByaXZhdGVOb2RlKFJpY2hUZXh0Q2hpbGRJbWFnZU5hbWUpO1xuICAgICAgICAgICAgbGV0IHNwcml0ZUNvbXBvbmVudCA9IHNwcml0ZU5vZGUuYWRkQ29tcG9uZW50KGNjLlNwcml0ZSk7XG4gICAgICAgICAgICBzd2l0Y2ggKHJpY2hUZXh0RWxlbWVudC5zdHlsZS5pbWFnZUFsaWduKVxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIGNhc2UgJ3RvcCc6XG4gICAgICAgICAgICAgICAgICAgIHNwcml0ZU5vZGUuc2V0QW5jaG9yUG9pbnQoMCwgMSk7XG4gICAgICAgICAgICAgICAgICAgIGJyZWFrO1xuICAgICAgICAgICAgICAgIGNhc2UgJ2NlbnRlcic6XG4gICAgICAgICAgICAgICAgICAgIHNwcml0ZU5vZGUuc2V0QW5jaG9yUG9pbnQoMCwgMC41KTtcbiAgICAgICAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgICAgICAgZGVmYXVsdDpcbiAgICAgICAgICAgICAgICAgICAgc3ByaXRlTm9kZS5zZXRBbmNob3JQb2ludCgwLCAwKTtcbiAgICAgICAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBpZiAocmljaFRleHRFbGVtZW50LnN0eWxlLmltYWdlT2Zmc2V0KSBzcHJpdGVOb2RlLl9pbWFnZU9mZnNldCA9IHJpY2hUZXh0RWxlbWVudC5zdHlsZS5pbWFnZU9mZnNldDtcbiAgICAgICAgICAgIHNwcml0ZUNvbXBvbmVudC50eXBlID0gY2MuU3ByaXRlLlR5cGUuU0xJQ0VEO1xuICAgICAgICAgICAgc3ByaXRlQ29tcG9uZW50LnNpemVNb2RlID0gY2MuU3ByaXRlLlNpemVNb2RlLkNVU1RPTTtcbiAgICAgICAgICAgIHRoaXMubm9kZS5hZGRDaGlsZChzcHJpdGVOb2RlKTtcbiAgICAgICAgICAgIHRoaXMuX2xhYmVsU2VnbWVudHMucHVzaChzcHJpdGVOb2RlKTtcblxuICAgICAgICAgICAgbGV0IHNwcml0ZVJlY3QgPSBzcHJpdGVGcmFtZS5nZXRSZWN0KCk7XG4gICAgICAgICAgICBsZXQgc2NhbGVGYWN0b3IgPSAxO1xuICAgICAgICAgICAgbGV0IHNwcml0ZVdpZHRoID0gc3ByaXRlUmVjdC53aWR0aDtcbiAgICAgICAgICAgIGxldCBzcHJpdGVIZWlnaHQgPSBzcHJpdGVSZWN0LmhlaWdodDtcbiAgICAgICAgICAgIGxldCBleHBlY3RXaWR0aCA9IHJpY2hUZXh0RWxlbWVudC5zdHlsZS5pbWFnZVdpZHRoO1xuICAgICAgICAgICAgbGV0IGV4cGVjdEhlaWdodCA9IHJpY2hUZXh0RWxlbWVudC5zdHlsZS5pbWFnZUhlaWdodDtcblxuICAgICAgICAgICAgaWYgKGV4cGVjdEhlaWdodCA+IDApIHtcbiAgICAgICAgICAgICAgICBzY2FsZUZhY3RvciA9IGV4cGVjdEhlaWdodCAvIHNwcml0ZUhlaWdodDtcbiAgICAgICAgICAgICAgICBzcHJpdGVXaWR0aCA9IHNwcml0ZVdpZHRoICogc2NhbGVGYWN0b3I7XG4gICAgICAgICAgICAgICAgc3ByaXRlSGVpZ2h0ID0gc3ByaXRlSGVpZ2h0ICogc2NhbGVGYWN0b3I7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgICAgICBzY2FsZUZhY3RvciA9IHRoaXMubGluZUhlaWdodCAvIHNwcml0ZUhlaWdodDtcbiAgICAgICAgICAgICAgICBzcHJpdGVXaWR0aCA9IHNwcml0ZVdpZHRoICogc2NhbGVGYWN0b3I7XG4gICAgICAgICAgICAgICAgc3ByaXRlSGVpZ2h0ID0gc3ByaXRlSGVpZ2h0ICogc2NhbGVGYWN0b3I7XG4gICAgICAgICAgICB9XG5cbiAgICAgICAgICAgIGlmIChleHBlY3RXaWR0aCA+IDApIHNwcml0ZVdpZHRoID0gZXhwZWN0V2lkdGg7XG5cbiAgICAgICAgICAgIGlmICh0aGlzLm1heFdpZHRoID4gMCkge1xuICAgICAgICAgICAgICAgIGlmICh0aGlzLl9saW5lT2Zmc2V0WCArIHNwcml0ZVdpZHRoID4gdGhpcy5tYXhXaWR0aCkge1xuICAgICAgICAgICAgICAgICAgICB0aGlzLl91cGRhdGVMaW5lSW5mbygpO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICB0aGlzLl9saW5lT2Zmc2V0WCArPSBzcHJpdGVXaWR0aDtcblxuICAgICAgICAgICAgfVxuICAgICAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICAgICAgdGhpcy5fbGluZU9mZnNldFggKz0gc3ByaXRlV2lkdGg7XG4gICAgICAgICAgICAgICAgaWYgKHRoaXMuX2xpbmVPZmZzZXRYID4gdGhpcy5fbGFiZWxXaWR0aCkge1xuICAgICAgICAgICAgICAgICAgICB0aGlzLl9sYWJlbFdpZHRoID0gdGhpcy5fbGluZU9mZnNldFg7XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfVxuICAgICAgICAgICAgc3ByaXRlQ29tcG9uZW50LnNwcml0ZUZyYW1lID0gc3ByaXRlRnJhbWU7XG4gICAgICAgICAgICBzcHJpdGVOb2RlLnNldENvbnRlbnRTaXplKHNwcml0ZVdpZHRoLCBzcHJpdGVIZWlnaHQpO1xuICAgICAgICAgICAgc3ByaXRlTm9kZS5fbGluZUNvdW50ID0gdGhpcy5fbGluZUNvdW50O1xuXG4gICAgICAgICAgICBpZiAocmljaFRleHRFbGVtZW50LnN0eWxlLmV2ZW50KSB7XG4gICAgICAgICAgICAgICAgaWYgKHJpY2hUZXh0RWxlbWVudC5zdHlsZS5ldmVudC5jbGljaykge1xuICAgICAgICAgICAgICAgICAgICBzcHJpdGVOb2RlLl9jbGlja0hhbmRsZXIgPSByaWNoVGV4dEVsZW1lbnQuc3R5bGUuZXZlbnQuY2xpY2s7XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIGlmIChyaWNoVGV4dEVsZW1lbnQuc3R5bGUuZXZlbnQucGFyYW0pIHtcbiAgICAgICAgICAgICAgICAgICAgc3ByaXRlTm9kZS5fY2xpY2tQYXJhbSA9IHJpY2hUZXh0RWxlbWVudC5zdHlsZS5ldmVudC5wYXJhbTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICAgICAgICAgIHNwcml0ZU5vZGUuX2NsaWNrUGFyYW0gPSAnJztcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgICAgICBzcHJpdGVOb2RlLl9jbGlja0hhbmRsZXIgPSBudWxsO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIGVsc2Uge1xuICAgICAgICAgICAgY2Mud2FybklEKDQ0MDApO1xuICAgICAgICB9XG4gICAgfSxcblxuICAgIF91cGRhdGVSaWNoVGV4dCAoKSB7XG4gICAgICAgIGlmICghdGhpcy5lbmFibGVkSW5IaWVyYXJjaHkpIHJldHVybjtcblxuICAgICAgICBsZXQgbmV3VGV4dEFycmF5ID0gX2h0bWxUZXh0UGFyc2VyLnBhcnNlKHRoaXMuc3RyaW5nKTtcbiAgICAgICAgaWYgKCF0aGlzLl9uZWVkc1VwZGF0ZVRleHRMYXlvdXQobmV3VGV4dEFycmF5KSkge1xuICAgICAgICAgICAgdGhpcy5fdGV4dEFycmF5ID0gbmV3VGV4dEFycmF5O1xuICAgICAgICAgICAgdGhpcy5fdXBkYXRlTGFiZWxTZWdtZW50VGV4dEF0dHJpYnV0ZXMoKTtcbiAgICAgICAgICAgIHJldHVybjtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuX3RleHRBcnJheSA9IG5ld1RleHRBcnJheTtcbiAgICAgICAgdGhpcy5fcmVzZXRTdGF0ZSgpO1xuXG4gICAgICAgIGxldCBsYXN0RW1wdHlMaW5lID0gZmFsc2U7XG4gICAgICAgIGxldCBsYWJlbDtcbiAgICAgICAgbGV0IGxhYmVsU2l6ZTtcblxuICAgICAgICBmb3IgKGxldCBpID0gMDsgaSA8IHRoaXMuX3RleHRBcnJheS5sZW5ndGg7ICsraSkge1xuICAgICAgICAgICAgbGV0IHJpY2hUZXh0RWxlbWVudCA9IHRoaXMuX3RleHRBcnJheVtpXTtcbiAgICAgICAgICAgIGxldCB0ZXh0ID0gcmljaFRleHRFbGVtZW50LnRleHQ7XG4gICAgICAgICAgICAvL2hhbmRsZSA8YnIvPiA8aW1nIC8+IHRhZ1xuICAgICAgICAgICAgaWYgKHRleHQgPT09IFwiXCIpIHtcbiAgICAgICAgICAgICAgICBpZiAocmljaFRleHRFbGVtZW50LnN0eWxlICYmIHJpY2hUZXh0RWxlbWVudC5zdHlsZS5uZXdsaW5lKSB7XG4gICAgICAgICAgICAgICAgICAgIHRoaXMuX3VwZGF0ZUxpbmVJbmZvKCk7XG4gICAgICAgICAgICAgICAgICAgIGNvbnRpbnVlO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBpZiAocmljaFRleHRFbGVtZW50LnN0eWxlICYmIHJpY2hUZXh0RWxlbWVudC5zdHlsZS5pc0ltYWdlICYmIHRoaXMuaW1hZ2VBdGxhcykge1xuICAgICAgICAgICAgICAgICAgICB0aGlzLl9hZGRSaWNoVGV4dEltYWdlRWxlbWVudChyaWNoVGV4dEVsZW1lbnQpO1xuICAgICAgICAgICAgICAgICAgICBjb250aW51ZTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBsZXQgbXVsdGlsaW5lVGV4dHMgPSB0ZXh0LnNwbGl0KFwiXFxuXCIpO1xuXG4gICAgICAgICAgICBmb3IgKGxldCBqID0gMDsgaiA8IG11bHRpbGluZVRleHRzLmxlbmd0aDsgKytqKSB7XG4gICAgICAgICAgICAgICAgbGV0IGxhYmVsU3RyaW5nID0gbXVsdGlsaW5lVGV4dHNbal07XG4gICAgICAgICAgICAgICAgaWYgKGxhYmVsU3RyaW5nID09PSBcIlwiKSB7XG4gICAgICAgICAgICAgICAgICAgIC8vZm9yIGNvbnRpbnVlcyBcXG5cbiAgICAgICAgICAgICAgICAgICAgaWYgKHRoaXMuX2lzTGFzdENvbXBvbmVudENSKHRleHQpXG4gICAgICAgICAgICAgICAgICAgICAgICAmJiBqID09PSBtdWx0aWxpbmVUZXh0cy5sZW5ndGggLSAxKSB7XG4gICAgICAgICAgICAgICAgICAgICAgICBjb250aW51ZTtcbiAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgICAgICB0aGlzLl91cGRhdGVMaW5lSW5mbygpO1xuICAgICAgICAgICAgICAgICAgICBsYXN0RW1wdHlMaW5lID0gdHJ1ZTtcbiAgICAgICAgICAgICAgICAgICAgY29udGludWU7XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIGxhc3RFbXB0eUxpbmUgPSBmYWxzZTtcblxuICAgICAgICAgICAgICAgIGlmICh0aGlzLm1heFdpZHRoID4gMCkge1xuICAgICAgICAgICAgICAgICAgICBsZXQgbGFiZWxXaWR0aCA9IHRoaXMuX21lYXN1cmVUZXh0KGksIGxhYmVsU3RyaW5nKTtcbiAgICAgICAgICAgICAgICAgICAgdGhpcy5fdXBkYXRlUmljaFRleHRXaXRoTWF4V2lkdGgobGFiZWxTdHJpbmcsIGxhYmVsV2lkdGgsIGkpO1xuXG4gICAgICAgICAgICAgICAgICAgIGlmIChtdWx0aWxpbmVUZXh0cy5sZW5ndGggPiAxICYmIGogPCBtdWx0aWxpbmVUZXh0cy5sZW5ndGggLSAxKSB7XG4gICAgICAgICAgICAgICAgICAgICAgICB0aGlzLl91cGRhdGVMaW5lSW5mbygpO1xuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIGVsc2Uge1xuICAgICAgICAgICAgICAgICAgICBsYWJlbCA9IHRoaXMuX2FkZExhYmVsU2VnbWVudChsYWJlbFN0cmluZywgaSk7XG4gICAgICAgICAgICAgICAgICAgIGxhYmVsU2l6ZSA9IGxhYmVsLmdldENvbnRlbnRTaXplKCk7XG5cbiAgICAgICAgICAgICAgICAgICAgdGhpcy5fbGluZU9mZnNldFggKz0gbGFiZWxTaXplLndpZHRoO1xuICAgICAgICAgICAgICAgICAgICBpZiAodGhpcy5fbGluZU9mZnNldFggPiB0aGlzLl9sYWJlbFdpZHRoKSB7XG4gICAgICAgICAgICAgICAgICAgICAgICB0aGlzLl9sYWJlbFdpZHRoID0gdGhpcy5fbGluZU9mZnNldFg7XG4gICAgICAgICAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgICAgICAgICBpZiAobXVsdGlsaW5lVGV4dHMubGVuZ3RoID4gMSAmJiBqIDwgbXVsdGlsaW5lVGV4dHMubGVuZ3RoIC0gMSkge1xuICAgICAgICAgICAgICAgICAgICAgICAgdGhpcy5fdXBkYXRlTGluZUluZm8oKTtcbiAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICBpZiAoIWxhc3RFbXB0eUxpbmUpIHtcbiAgICAgICAgICAgIHRoaXMuX2xpbmVzV2lkdGgucHVzaCh0aGlzLl9saW5lT2Zmc2V0WCk7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAodGhpcy5tYXhXaWR0aCA+IDApIHtcbiAgICAgICAgICAgIHRoaXMuX2xhYmVsV2lkdGggPSB0aGlzLm1heFdpZHRoO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMuX2xhYmVsSGVpZ2h0ID0gKHRoaXMuX2xpbmVDb3VudCArIHRleHRVdGlscy5CQVNFTElORV9SQVRJTykgKiB0aGlzLmxpbmVIZWlnaHQ7XG5cbiAgICAgICAgLy8gdHJpZ2dlciBcInNpemUtY2hhbmdlZFwiIGV2ZW50XG4gICAgICAgIHRoaXMubm9kZS5zZXRDb250ZW50U2l6ZSh0aGlzLl9sYWJlbFdpZHRoLCB0aGlzLl9sYWJlbEhlaWdodCk7XG5cbiAgICAgICAgdGhpcy5fdXBkYXRlUmljaFRleHRQb3NpdGlvbigpO1xuICAgICAgICB0aGlzLl9sYXlvdXREaXJ0eSA9IGZhbHNlO1xuICAgIH0sXG5cbiAgICBfZ2V0Rmlyc3RXb3JkTGVuICh0ZXh0LCBzdGFydEluZGV4LCB0ZXh0TGVuKSB7XG4gICAgICAgIGxldCBjaGFyYWN0ZXIgPSB0ZXh0LmNoYXJBdChzdGFydEluZGV4KTtcbiAgICAgICAgaWYgKHRleHRVdGlscy5pc1VuaWNvZGVDSksoY2hhcmFjdGVyKVxuICAgICAgICAgICAgfHwgdGV4dFV0aWxzLmlzVW5pY29kZVNwYWNlKGNoYXJhY3RlcikpIHtcbiAgICAgICAgICAgIHJldHVybiAxO1xuICAgICAgICB9XG5cbiAgICAgICAgbGV0IGxlbiA9IDE7XG4gICAgICAgIGZvciAobGV0IGluZGV4ID0gc3RhcnRJbmRleCArIDE7IGluZGV4IDwgdGV4dExlbjsgKytpbmRleCkge1xuICAgICAgICAgICAgY2hhcmFjdGVyID0gdGV4dC5jaGFyQXQoaW5kZXgpO1xuICAgICAgICAgICAgaWYgKHRleHRVdGlscy5pc1VuaWNvZGVTcGFjZShjaGFyYWN0ZXIpXG4gICAgICAgICAgICAgICAgfHwgdGV4dFV0aWxzLmlzVW5pY29kZUNKSyhjaGFyYWN0ZXIpKSB7XG4gICAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBsZW4rKztcbiAgICAgICAgfVxuXG4gICAgICAgIHJldHVybiBsZW47XG4gICAgfSxcblxuICAgIF91cGRhdGVSaWNoVGV4dFBvc2l0aW9uICgpIHtcbiAgICAgICAgbGV0IG5leHRUb2tlblggPSAwO1xuICAgICAgICBsZXQgbmV4dExpbmVJbmRleCA9IDE7XG4gICAgICAgIGxldCB0b3RhbExpbmVDb3VudCA9IHRoaXMuX2xpbmVDb3VudDtcbiAgICAgICAgZm9yIChsZXQgaSA9IDA7IGkgPCB0aGlzLl9sYWJlbFNlZ21lbnRzLmxlbmd0aDsgKytpKSB7XG4gICAgICAgICAgICBsZXQgbGFiZWwgPSB0aGlzLl9sYWJlbFNlZ21lbnRzW2ldO1xuICAgICAgICAgICAgbGV0IGxpbmVDb3VudCA9IGxhYmVsLl9saW5lQ291bnQ7XG4gICAgICAgICAgICBpZiAobGluZUNvdW50ID4gbmV4dExpbmVJbmRleCkge1xuICAgICAgICAgICAgICAgIG5leHRUb2tlblggPSAwO1xuICAgICAgICAgICAgICAgIG5leHRMaW5lSW5kZXggPSBsaW5lQ291bnQ7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBsZXQgbGluZU9mZnNldFggPSAwO1xuICAgICAgICAgICAgLy8gbGV0IG5vZGVBbmNob3JYT2Zmc2V0ID0gKDAuNSAtIHRoaXMubm9kZS5hbmNob3JYKSAqIHRoaXMuX2xhYmVsV2lkdGg7XG4gICAgICAgICAgICBzd2l0Y2ggKHRoaXMuaG9yaXpvbnRhbEFsaWduKSB7XG4gICAgICAgICAgICAgICAgY2FzZSBIb3Jpem9udGFsQWxpZ24uTEVGVDpcbiAgICAgICAgICAgICAgICAgICAgbGluZU9mZnNldFggPSAtIHRoaXMuX2xhYmVsV2lkdGggLyAyO1xuICAgICAgICAgICAgICAgICAgICBicmVhaztcbiAgICAgICAgICAgICAgICBjYXNlIEhvcml6b250YWxBbGlnbi5DRU5URVI6XG4gICAgICAgICAgICAgICAgICAgIGxpbmVPZmZzZXRYID0gLSB0aGlzLl9saW5lc1dpZHRoW2xpbmVDb3VudCAtIDFdIC8gMjtcbiAgICAgICAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgICAgICAgY2FzZSBIb3Jpem9udGFsQWxpZ24uUklHSFQ6XG4gICAgICAgICAgICAgICAgICAgIGxpbmVPZmZzZXRYID0gdGhpcy5fbGFiZWxXaWR0aCAvIDIgLSB0aGlzLl9saW5lc1dpZHRoW2xpbmVDb3VudCAtIDFdO1xuICAgICAgICAgICAgICAgICAgICBicmVhaztcbiAgICAgICAgICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgICAgICAgICAgICBicmVhaztcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGxhYmVsLnggPSBuZXh0VG9rZW5YICsgbGluZU9mZnNldFg7XG5cbiAgICAgICAgICAgIGxldCBsYWJlbFNpemUgPSBsYWJlbC5nZXRDb250ZW50U2l6ZSgpO1xuXG4gICAgICAgICAgICBsYWJlbC55ID0gdGhpcy5saW5lSGVpZ2h0ICogKHRvdGFsTGluZUNvdW50IC0gbGluZUNvdW50KSAtIHRoaXMuX2xhYmVsSGVpZ2h0IC8gMjtcblxuICAgICAgICAgICAgaWYgKGxpbmVDb3VudCA9PT0gbmV4dExpbmVJbmRleCkge1xuICAgICAgICAgICAgICAgIG5leHRUb2tlblggKz0gbGFiZWxTaXplLndpZHRoO1xuICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICBsZXQgc3ByaXRlID0gbGFiZWwuZ2V0Q29tcG9uZW50KGNjLlNwcml0ZSk7XG4gICAgICAgICAgICBpZiAoc3ByaXRlKSB7XG4gICAgICAgICAgICAgICAgLy8gYWRqdXN0IGltZyBhbGlnbiAoZnJvbSA8aW1nIGFsaWduPXRvcHxjZW50ZXJ8Ym90dG9tPilcbiAgICAgICAgICAgICAgICBsZXQgbGluZUhlaWdodFNldCA9IHRoaXMubGluZUhlaWdodDtcbiAgICAgICAgICAgICAgICBsZXQgbGluZUhlaWdodFJlYWwgPSB0aGlzLmxpbmVIZWlnaHQgKiAoMSArIHRleHRVdGlscy5CQVNFTElORV9SQVRJTyk7IC8vc2luZ2xlIGxpbmUgbm9kZSBoZWlnaHRcbiAgICAgICAgICAgICAgICBzd2l0Y2ggKGxhYmVsLmFuY2hvclkpXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBjYXNlIDE6XG4gICAgICAgICAgICAgICAgICAgICAgICBsYWJlbC55ICs9ICggbGluZUhlaWdodFNldCArICggKCBsaW5lSGVpZ2h0UmVhbCAtIGxpbmVIZWlnaHRTZXQpIC8gMiApICk7XG4gICAgICAgICAgICAgICAgICAgICAgICBicmVhaztcbiAgICAgICAgICAgICAgICAgICAgY2FzZSAwLjU6XG4gICAgICAgICAgICAgICAgICAgICAgICBsYWJlbC55ICs9ICggbGluZUhlaWdodFJlYWwgLyAyICk7XG4gICAgICAgICAgICAgICAgICAgICAgICBicmVhaztcbiAgICAgICAgICAgICAgICAgICAgZGVmYXVsdDpcbiAgICAgICAgICAgICAgICAgICAgICAgIGxhYmVsLnkgKz0gKCAobGluZUhlaWdodFJlYWwgLSBsaW5lSGVpZ2h0U2V0KSAvIDIgKTtcbiAgICAgICAgICAgICAgICAgICAgICAgIGJyZWFrO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAvLyBhZGp1c3QgaW1nIG9mZnNldCAoZnJvbSA8aW1nIG9mZnNldD0xMnwxMiwzND4pXG4gICAgICAgICAgICAgICAgaWYgKGxhYmVsLl9pbWFnZU9mZnNldClcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIGxldCBvZmZzZXRzID0gbGFiZWwuX2ltYWdlT2Zmc2V0LnNwbGl0KCcsJyk7XG4gICAgICAgICAgICAgICAgICAgIGlmIChvZmZzZXRzLmxlbmd0aCA9PT0gMSAmJiBvZmZzZXRzWzBdKVxuICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICBsZXQgb2Zmc2V0WSA9IHBhcnNlRmxvYXQob2Zmc2V0c1swXSk7XG4gICAgICAgICAgICAgICAgICAgICAgICBpZiAoTnVtYmVyLmlzSW50ZWdlcihvZmZzZXRZKSkgbGFiZWwueSArPSBvZmZzZXRZO1xuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgICAgIGVsc2UgaWYob2Zmc2V0cy5sZW5ndGggPT09IDIpXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIGxldCBvZmZzZXRYID0gcGFyc2VGbG9hdChvZmZzZXRzWzBdKTtcbiAgICAgICAgICAgICAgICAgICAgICAgIGxldCBvZmZzZXRZID0gcGFyc2VGbG9hdChvZmZzZXRzWzFdKTtcbiAgICAgICAgICAgICAgICAgICAgICAgIGlmIChOdW1iZXIuaXNJbnRlZ2VyKG9mZnNldFgpKSBsYWJlbC54ICs9IG9mZnNldFg7XG4gICAgICAgICAgICAgICAgICAgICAgICBpZiAoTnVtYmVyLmlzSW50ZWdlcihvZmZzZXRZKSkgbGFiZWwueSArPSBvZmZzZXRZO1xuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICAvL2FkanVzdCB5IGZvciBsYWJlbCB3aXRoIG91dGxpbmVcbiAgICAgICAgICAgIGxldCBvdXRsaW5lID0gbGFiZWwuZ2V0Q29tcG9uZW50KGNjLkxhYmVsT3V0bGluZSk7XG4gICAgICAgICAgICBpZiAob3V0bGluZSAmJiBvdXRsaW5lLndpZHRoKSBsYWJlbC55ID0gbGFiZWwueSAtIG91dGxpbmUud2lkdGg7XG4gICAgICAgIH1cbiAgICB9LFxuXG4gICAgX2NvbnZlcnRMaXRlcmFsQ29sb3JWYWx1ZSAoY29sb3IpIHtcbiAgICAgICAgbGV0IGNvbG9yVmFsdWUgPSBjb2xvci50b1VwcGVyQ2FzZSgpO1xuICAgICAgICBpZiAoY2MuQ29sb3JbY29sb3JWYWx1ZV0pIHtcbiAgICAgICAgICAgIHJldHVybiBjYy5Db2xvcltjb2xvclZhbHVlXTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIGxldCBvdXQgPSBjYy5jb2xvcigpO1xuICAgICAgICAgICAgcmV0dXJuIG91dC5mcm9tSEVYKGNvbG9yKTtcbiAgICAgICAgfVxuICAgIH0sXG5cbiAgICAvLyBXaGVuIHN0cmluZyBpcyBudWxsLCBpdCBtZWFucyB0aGF0IHRoZSB0ZXh0IGRvZXMgbm90IG5lZWQgdG8gYmUgdXBkYXRlZC5cbiAgICBfYXBwbHlUZXh0QXR0cmlidXRlIChsYWJlbE5vZGUsIHN0cmluZywgZm9yY2UpIHtcbiAgICAgICAgbGV0IGxhYmVsQ29tcG9uZW50ID0gbGFiZWxOb2RlLmdldENvbXBvbmVudChjYy5MYWJlbCk7XG4gICAgICAgIGlmICghbGFiZWxDb21wb25lbnQpIHtcbiAgICAgICAgICAgIHJldHVybjtcbiAgICAgICAgfVxuXG4gICAgICAgIGxldCBpbmRleCA9IGxhYmVsTm9kZS5fc3R5bGVJbmRleDtcblxuICAgICAgICBsZXQgdGV4dFN0eWxlID0gbnVsbDtcbiAgICAgICAgaWYgKHRoaXMuX3RleHRBcnJheVtpbmRleF0pIHtcbiAgICAgICAgICAgIHRleHRTdHlsZSA9IHRoaXMuX3RleHRBcnJheVtpbmRleF0uc3R5bGU7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAodGV4dFN0eWxlICYmIHRleHRTdHlsZS5jb2xvcikge1xuICAgICAgICAgICAgbGFiZWxOb2RlLmNvbG9yID0gdGhpcy5fY29udmVydExpdGVyYWxDb2xvclZhbHVlKHRleHRTdHlsZS5jb2xvcik7XG4gICAgICAgIH1lbHNlIHtcbiAgICAgICAgICAgIGxhYmVsTm9kZS5jb2xvciA9IHRoaXMubm9kZS5jb2xvcjtcbiAgICAgICAgfVxuXG4gICAgICAgIGxhYmVsQ29tcG9uZW50LmNhY2hlTW9kZSA9IHRoaXMuY2FjaGVNb2RlO1xuXG4gICAgICAgIGxldCBpc0Fzc2V0ID0gdGhpcy5mb250IGluc3RhbmNlb2YgY2MuRm9udDtcbiAgICAgICAgaWYgKGlzQXNzZXQgJiYgIXRoaXMuX2lzU3lzdGVtRm9udFVzZWQpIHtcbiAgICAgICAgICAgIGxhYmVsQ29tcG9uZW50LmZvbnQgPSB0aGlzLmZvbnQ7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICBsYWJlbENvbXBvbmVudC5mb250RmFtaWx5ID0gdGhpcy5mb250RmFtaWx5O1xuICAgICAgICB9XG5cbiAgICAgICAgbGFiZWxDb21wb25lbnQudXNlU3lzdGVtRm9udCA9IHRoaXMuX2lzU3lzdGVtRm9udFVzZWQ7XG4gICAgICAgIGxhYmVsQ29tcG9uZW50LmxpbmVIZWlnaHQgPSB0aGlzLmxpbmVIZWlnaHQ7XG4gICAgICAgIGxhYmVsQ29tcG9uZW50LmVuYWJsZUJvbGQgPSB0ZXh0U3R5bGUgJiYgdGV4dFN0eWxlLmJvbGQ7XG4gICAgICAgIGxhYmVsQ29tcG9uZW50LmVuYWJsZUl0YWxpY3MgPSB0ZXh0U3R5bGUgJiYgdGV4dFN0eWxlLml0YWxpYztcbiAgICAgICAgLy9UT0RPOiB0ZW1wb3JhcnkgaW1wbGVtZW50YXRpb24sIHRoZSBpdGFsaWMgZWZmZWN0IHNob3VsZCBiZSBpbXBsZW1lbnRlZCBpbiB0aGUgaW50ZXJuYWwgb2YgbGFiZWwtYXNzZW1ibGVyLlxuICAgICAgICBpZiAodGV4dFN0eWxlICYmIHRleHRTdHlsZS5pdGFsaWMpIHtcbiAgICAgICAgICAgIGxhYmVsTm9kZS5za2V3WCA9IDEyO1xuICAgICAgICB9XG5cbiAgICAgICAgbGFiZWxDb21wb25lbnQuZW5hYmxlVW5kZXJsaW5lID0gdGV4dFN0eWxlICYmIHRleHRTdHlsZS51bmRlcmxpbmU7XG5cbiAgICAgICAgaWYgKHRleHRTdHlsZSAmJiB0ZXh0U3R5bGUub3V0bGluZSkge1xuICAgICAgICAgICAgbGV0IGxhYmVsT3V0bGluZUNvbXBvbmVudCA9IGxhYmVsTm9kZS5nZXRDb21wb25lbnQoY2MuTGFiZWxPdXRsaW5lKTtcbiAgICAgICAgICAgIGlmICghbGFiZWxPdXRsaW5lQ29tcG9uZW50KSB7XG4gICAgICAgICAgICAgICAgbGFiZWxPdXRsaW5lQ29tcG9uZW50ID0gbGFiZWxOb2RlLmFkZENvbXBvbmVudChjYy5MYWJlbE91dGxpbmUpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgbGFiZWxPdXRsaW5lQ29tcG9uZW50LmNvbG9yID0gdGhpcy5fY29udmVydExpdGVyYWxDb2xvclZhbHVlKHRleHRTdHlsZS5vdXRsaW5lLmNvbG9yKTtcbiAgICAgICAgICAgIGxhYmVsT3V0bGluZUNvbXBvbmVudC53aWR0aCA9IHRleHRTdHlsZS5vdXRsaW5lLndpZHRoO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKHRleHRTdHlsZSAmJiB0ZXh0U3R5bGUuc2l6ZSkge1xuICAgICAgICAgICAgbGFiZWxDb21wb25lbnQuZm9udFNpemUgPSB0ZXh0U3R5bGUuc2l6ZTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIGxhYmVsQ29tcG9uZW50LmZvbnRTaXplID0gdGhpcy5mb250U2l6ZTtcbiAgICAgICAgfVxuXG4gICAgICAgIGlmIChzdHJpbmcgIT09IG51bGwpIHtcbiAgICAgICAgICAgIGlmICh0eXBlb2Ygc3RyaW5nICE9PSAnc3RyaW5nJykge1xuICAgICAgICAgICAgICAgIHN0cmluZyA9ICcnICsgc3RyaW5nO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgbGFiZWxDb21wb25lbnQuc3RyaW5nID0gc3RyaW5nO1xuICAgICAgICB9XG5cbiAgICAgICAgZm9yY2UgJiYgbGFiZWxDb21wb25lbnQuX2ZvcmNlVXBkYXRlUmVuZGVyRGF0YSgpO1xuXG4gICAgICAgIGlmICh0ZXh0U3R5bGUgJiYgdGV4dFN0eWxlLmV2ZW50KSB7XG4gICAgICAgICAgICBpZiAodGV4dFN0eWxlLmV2ZW50LmNsaWNrKSB7XG4gICAgICAgICAgICAgICAgbGFiZWxOb2RlLl9jbGlja0hhbmRsZXIgPSB0ZXh0U3R5bGUuZXZlbnQuY2xpY2s7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBpZiAodGV4dFN0eWxlLmV2ZW50LnBhcmFtKSB7XG4gICAgICAgICAgICAgICAgbGFiZWxOb2RlLl9jbGlja1BhcmFtID0gdGV4dFN0eWxlLmV2ZW50LnBhcmFtO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICAgICAgbGFiZWxOb2RlLl9jbGlja1BhcmFtID0gJyc7XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICBsYWJlbE5vZGUuX2NsaWNrSGFuZGxlciA9IG51bGw7XG4gICAgICAgIH1cbiAgICB9LFxuXG4gICAgb25EZXN0cm95ICgpIHtcbiAgICAgICAgZm9yIChsZXQgaSA9IDA7IGkgPCB0aGlzLl9sYWJlbFNlZ21lbnRzLmxlbmd0aDsgKytpKSB7XG4gICAgICAgICAgICB0aGlzLl9sYWJlbFNlZ21lbnRzW2ldLnJlbW92ZUZyb21QYXJlbnQoKTtcbiAgICAgICAgICAgIHBvb2wucHV0KHRoaXMuX2xhYmVsU2VnbWVudHNbaV0pO1xuICAgICAgICB9XG4gICAgfSxcbn0pO1xuXG5jYy5SaWNoVGV4dCA9IG1vZHVsZS5leHBvcnRzID0gUmljaFRleHQ7XG4iXSwic291cmNlUm9vdCI6Ii8ifQ==