
                (function() {
                    var nodeEnv = typeof require !== 'undefined' && typeof process !== 'undefined';
                    var __module = nodeEnv ? module : {exports:{}};
                    var __filename = 'engine-dev/cocos2d/core/components/CCButton.js';
                    var __require = nodeEnv ? function (request) {
                        return require(request);
                    } : function (request) {
                        return __quick_compile_engine__.require(request, __filename);
                    };
                    function __define (exports, require, module) {
                        if (!nodeEnv) {__quick_compile_engine__.registerModule(__filename, module);}"use strict";

/****************************************************************************
 Copyright (c) 2013-2016 Chukong Technologies Inc.
 Copyright (c) 2017-2018 Xiamen Yaji Software Co., Ltd.

 https://www.cocos.com/

 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated engine source code (the "Software"), a limited,
  worldwide, royalty-free, non-assignable, revocable and non-exclusive license
 to use Cocos Creator solely to develop games on your target platforms. You shall
  not use Cocos Creator software for developing other software or tools that's
  used for developing games. You are not granted to publish, distribute,
  sublicense, and/or sell copies of Cocos Creator.

 The software or tools in this License Agreement are licensed, not sold.
 Xiamen Yaji Software Co., Ltd. reserves all rights not expressly granted to you.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 THE SOFTWARE.
 ****************************************************************************/
var Component = require('./CCComponent');

var GraySpriteState = require('../utils/gray-sprite-state');
/**
 * !#en Enum for transition type.
 * !#zh 过渡类型
 * @enum Button.Transition
 */


var Transition = cc.Enum({
  /**
   * !#en The none type.
   * !#zh 不做任何过渡
   * @property {Number} NONE
   */
  NONE: 0,

  /**
   * !#en The color type.
   * !#zh 颜色过渡
   * @property {Number} COLOR
   */
  COLOR: 1,

  /**
   * !#en The sprite type.
   * !#zh 精灵过渡
   * @property {Number} SPRITE
   */
  SPRITE: 2,

  /**
   * !#en The scale type
   * !#zh 缩放过渡
   * @property {Number} SCALE
   */
  SCALE: 3
});
var State = cc.Enum({
  NORMAL: 0,
  HOVER: 1,
  PRESSED: 2,
  DISABLED: 3
});
/**
 * !#en
 * Button has 4 Transition types<br/>
 * When Button state changed:<br/>
 *  If Transition type is Button.Transition.NONE, Button will do nothing<br/>
 *  If Transition type is Button.Transition.COLOR, Button will change target's color<br/>
 *  If Transition type is Button.Transition.SPRITE, Button will change target Sprite's sprite<br/>
 *  If Transition type is Button.Transition.SCALE, Button will change target node's scale<br/>
 *
 * Button will trigger 5 events:<br/>
 *  Button.EVENT_TOUCH_DOWN<br/>
 *  Button.EVENT_TOUCH_UP<br/>
 *  Button.EVENT_HOVER_IN<br/>
 *  Button.EVENT_HOVER_MOVE<br/>
 *  Button.EVENT_HOVER_OUT<br/>
 *  User can get the current clicked node with 'event.target' from event object which is passed as parameter in the callback function of click event.
 *
 * !#zh
 * 按钮组件。可以被按下，或者点击。
 *
 * 按钮可以通过修改 Transition 来设置按钮状态过渡的方式：
 * 
 *   - Button.Transition.NONE   // 不做任何过渡
 *   - Button.Transition.COLOR  // 进行颜色之间过渡
 *   - Button.Transition.SPRITE // 进行精灵之间过渡
 *   - Button.Transition.SCALE // 进行缩放过渡
 *
 * 按钮可以绑定事件（但是必须要在按钮的 Node 上才能绑定事件）：<br/>
 * 以下事件可以在全平台上都触发：
 * 
 *   - cc.Node.EventType.TOUCH_START  // 按下时事件
 *   - cc.Node.EventType.TOUCH_Move   // 按住移动后事件
 *   - cc.Node.EventType.TOUCH_END    // 按下后松开后事件
 *   - cc.Node.EventType.TOUCH_CANCEL // 按下取消事件
 * 
 * 以下事件只在 PC 平台上触发：
 * 
 *   - cc.Node.EventType.MOUSE_DOWN  // 鼠标按下时事件
 *   - cc.Node.EventType.MOUSE_MOVE  // 鼠标按住移动后事件
 *   - cc.Node.EventType.MOUSE_ENTER // 鼠标进入目标事件
 *   - cc.Node.EventType.MOUSE_LEAVE // 鼠标离开目标事件
 *   - cc.Node.EventType.MOUSE_UP    // 鼠标松开事件
 *   - cc.Node.EventType.MOUSE_WHEEL // 鼠标滚轮事件
 * 
 * 用户可以通过获取 __点击事件__ 回调函数的参数 event 的 target 属性获取当前点击对象。
 * @class Button
 * @extends Component
 * @uses GraySpriteState
 * @example
 *
 * // Add an event to the button.
 * button.node.on(cc.Node.EventType.TOUCH_START, function (event) {
 *     cc.log("This is a callback after the trigger event");
 * });

 * // You could also add a click event
 * //Note: In this way, you can't get the touch event info, so use it wisely.
 * button.node.on('click', function (button) {
 *    //The event is a custom event, you could get the Button component via first argument
 * })
 *
 */

var Button = cc.Class({
  name: 'cc.Button',
  "extends": Component,
  mixins: [GraySpriteState],
  ctor: function ctor() {
    this._pressed = false;
    this._hovered = false;
    this._fromColor = null;
    this._toColor = null;
    this._time = 0;
    this._transitionFinished = true; // init _originalScale in __preload()

    this._fromScale = cc.Vec2.ZERO;
    this._toScale = cc.Vec2.ZERO;
    this._originalScale = null;
    this._graySpriteMaterial = null;
    this._spriteMaterial = null;
    this._sprite = null;
  },
  editor: CC_EDITOR && {
    menu: 'i18n:MAIN_MENU.component.ui/Button',
    help: 'i18n:COMPONENT.help_url.button',
    inspector: 'packages://inspector/inspectors/comps/button.js',
    executeInEditMode: true
  },
  properties: {
    /**
     * !#en
     * Whether the Button is disabled.
     * If true, the Button will trigger event and do transition.
     * !#zh
     * 按钮事件是否被响应，如果为 false，则按钮将被禁用。
     * @property {Boolean} interactable
     * @default true
     */
    interactable: {
      "default": true,
      tooltip: CC_DEV && 'i18n:COMPONENT.button.interactable',
      notify: function notify() {
        this._updateState();

        if (!this.interactable) {
          this._resetState();
        }
      },
      animatable: false
    },
    _resizeToTarget: {
      animatable: false,
      set: function set(value) {
        if (value) {
          this._resizeNodeToTargetNode();
        }
      }
    },

    /**
     * !#en When this flag is true, Button target sprite will turn gray when interactable is false.
     * !#zh 如果这个标记为 true，当 button 的 interactable 属性为 false 的时候，会使用内置 shader 让 button 的 target 节点的 sprite 组件变灰
     * @property {Boolean} enableAutoGrayEffect
     */
    enableAutoGrayEffect: {
      "default": false,
      tooltip: CC_DEV && 'i18n:COMPONENT.button.auto_gray_effect',
      notify: function notify() {
        this._updateDisabledState();
      }
    },

    /**
     * !#en Transition type
     * !#zh 按钮状态改变时过渡方式。
     * @property {Button.Transition} transition
     * @default Button.Transition.Node
     */
    transition: {
      "default": Transition.NONE,
      tooltip: CC_DEV && 'i18n:COMPONENT.button.transition',
      type: Transition,
      animatable: false,
      notify: function notify(oldValue) {
        this._updateTransition(oldValue);
      },
      formerlySerializedAs: 'transition'
    },
    // color transition

    /**
     * !#en Normal state color.
     * !#zh 普通状态下按钮所显示的颜色。
     * @property {Color} normalColor
     */
    normalColor: {
      "default": cc.Color.WHITE,
      displayName: 'Normal',
      tooltip: CC_DEV && 'i18n:COMPONENT.button.normal_color',
      notify: function notify() {
        if (this.transition === Transition.Color && this._getButtonState() === State.NORMAL) {
          this._getTarget().opacity = this.normalColor.a;
        }

        this._updateState();
      }
    },

    /**
     * !#en Pressed state color
     * !#zh 按下状态时按钮所显示的颜色。
     * @property {Color} pressedColor
     */
    pressedColor: {
      "default": cc.color(211, 211, 211),
      displayName: 'Pressed',
      tooltip: CC_DEV && 'i18n:COMPONENT.button.pressed_color',
      notify: function notify() {
        if (this.transition === Transition.Color && this._getButtonState() === State.PRESSED) {
          this._getTarget().opacity = this.pressedColor.a;
        }

        this._updateState();
      },
      formerlySerializedAs: 'pressedColor'
    },

    /**
     * !#en Hover state color
     * !#zh 悬停状态下按钮所显示的颜色。
     * @property {Color} hoverColor
     */
    hoverColor: {
      "default": cc.Color.WHITE,
      displayName: 'Hover',
      tooltip: CC_DEV && 'i18n:COMPONENT.button.hover_color',
      notify: function notify() {
        if (this.transition === Transition.Color && this._getButtonState() === State.HOVER) {
          this._getTarget().opacity = this.hoverColor.a;
        }

        this._updateState();
      },
      formerlySerializedAs: 'hoverColor'
    },

    /**
     * !#en Disabled state color
     * !#zh 禁用状态下按钮所显示的颜色。
     * @property {Color} disabledColor
     */
    disabledColor: {
      "default": cc.color(124, 124, 124),
      displayName: 'Disabled',
      tooltip: CC_DEV && 'i18n:COMPONENT.button.disabled_color',
      notify: function notify() {
        if (this.transition === Transition.Color && this._getButtonState() === State.DISABLED) {
          this._getTarget().opacity = this.disabledColor.a;
        }

        this._updateState();
      }
    },

    /**
     * !#en Color and Scale transition duration
     * !#zh 颜色过渡和缩放过渡时所需时间
     * @property {Number} duration
     */
    duration: {
      "default": 0.1,
      range: [0, 10],
      tooltip: CC_DEV && 'i18n:COMPONENT.button.duration'
    },

    /**
     * !#en  When user press the button, the button will zoom to a scale.
     * The final scale of the button  equals (button original scale * zoomScale)
     * !#zh 当用户点击按钮后，按钮会缩放到一个值，这个值等于 Button 原始 scale * zoomScale
     * @property {Number} zoomScale
     */
    zoomScale: {
      "default": 1.2,
      tooltip: CC_DEV && 'i18n:COMPONENT.button.zoom_scale'
    },
    // sprite transition

    /**
     * !#en Normal state sprite
     * !#zh 普通状态下按钮所显示的 Sprite 。
     * @property {SpriteFrame} normalSprite
     */
    normalSprite: {
      "default": null,
      type: cc.SpriteFrame,
      displayName: 'Normal',
      tooltip: CC_DEV && 'i18n:COMPONENT.button.normal_sprite',
      notify: function notify() {
        this._updateState();
      }
    },

    /**
     * !#en Pressed state sprite
     * !#zh 按下状态时按钮所显示的 Sprite 。
     * @property {SpriteFrame} pressedSprite
     */
    pressedSprite: {
      "default": null,
      type: cc.SpriteFrame,
      displayName: 'Pressed',
      tooltip: CC_DEV && 'i18n:COMPONENT.button.pressed_sprite',
      formerlySerializedAs: 'pressedSprite',
      notify: function notify() {
        this._updateState();
      }
    },

    /**
     * !#en Hover state sprite
     * !#zh 悬停状态下按钮所显示的 Sprite 。
     * @property {SpriteFrame} hoverSprite
     */
    hoverSprite: {
      "default": null,
      type: cc.SpriteFrame,
      displayName: 'Hover',
      tooltip: CC_DEV && 'i18n:COMPONENT.button.hover_sprite',
      formerlySerializedAs: 'hoverSprite',
      notify: function notify() {
        this._updateState();
      }
    },

    /**
     * !#en Disabled state sprite
     * !#zh 禁用状态下按钮所显示的 Sprite 。
     * @property {SpriteFrame} disabledSprite
     */
    disabledSprite: {
      "default": null,
      type: cc.SpriteFrame,
      displayName: 'Disabled',
      tooltip: CC_DEV && 'i18n:COMPONENT.button.disabled_sprite',
      notify: function notify() {
        this._updateState();
      }
    },

    /**
     * !#en
     * Transition target.
     * When Button state changed:
     *  If Transition type is Button.Transition.NONE, Button will do nothing
     *  If Transition type is Button.Transition.COLOR, Button will change target's color
     *  If Transition type is Button.Transition.SPRITE, Button will change target Sprite's sprite
     * !#zh
     * 需要过渡的目标。
     * 当前按钮状态改变规则：
     * -如果 Transition type 选择 Button.Transition.NONE，按钮不做任何过渡。
     * -如果 Transition type 选择 Button.Transition.COLOR，按钮会对目标颜色进行颜色之间的过渡。
     * -如果 Transition type 选择 Button.Transition.Sprite，按钮会对目标 Sprite 进行 Sprite 之间的过渡。
     * @property {Node} target
     */
    target: {
      "default": null,
      type: cc.Node,
      tooltip: CC_DEV && "i18n:COMPONENT.button.target",
      notify: function notify(oldValue) {
        this._applyTarget();

        if (oldValue && this.target !== oldValue) {
          this._unregisterTargetEvent(oldValue);
        }
      }
    },

    /**
     * !#en If Button is clicked, it will trigger event's handler
     * !#zh 按钮的点击事件列表。
     * @property {Component.EventHandler[]} clickEvents
     */
    clickEvents: {
      "default": [],
      type: cc.Component.EventHandler,
      tooltip: CC_DEV && 'i18n:COMPONENT.button.click_events'
    }
  },
  statics: {
    Transition: Transition
  },
  __preload: function __preload() {
    this._applyTarget();

    this._resetState();
  },
  _resetState: function _resetState() {
    this._pressed = false;
    this._hovered = false; // // Restore button status

    var target = this._getTarget();

    var transition = this.transition;
    var originalScale = this._originalScale;

    if (transition === Transition.COLOR && this.interactable) {
      this._setTargetColor(this.normalColor);
    } else if (transition === Transition.SCALE && originalScale) {
      target.setScale(originalScale.x, originalScale.y);
    }

    this._transitionFinished = true;
  },
  onEnable: function onEnable() {
    // check sprite frames
    if (this.normalSprite) {
      this.normalSprite.ensureLoadTexture();
    }

    if (this.hoverSprite) {
      this.hoverSprite.ensureLoadTexture();
    }

    if (this.pressedSprite) {
      this.pressedSprite.ensureLoadTexture();
    }

    if (this.disabledSprite) {
      this.disabledSprite.ensureLoadTexture();
    }

    if (!CC_EDITOR) {
      this._registerNodeEvent();
    }
  },
  onDisable: function onDisable() {
    this._resetState();

    if (!CC_EDITOR) {
      this._unregisterNodeEvent();
    }
  },
  _getTarget: function _getTarget() {
    return this.target ? this.target : this.node;
  },
  _onTargetSpriteFrameChanged: function _onTargetSpriteFrameChanged(comp) {
    if (this.transition === Transition.SPRITE) {
      this._setCurrentStateSprite(comp.spriteFrame);
    }
  },
  _onTargetColorChanged: function _onTargetColorChanged(color) {
    if (this.transition === Transition.COLOR) {
      this._setCurrentStateColor(color);
    }
  },
  _onTargetScaleChanged: function _onTargetScaleChanged() {
    var target = this._getTarget(); // update _originalScale if target scale changed


    if (this._originalScale) {
      if (this.transition !== Transition.SCALE || this._transitionFinished) {
        this._originalScale.x = target.scaleX;
        this._originalScale.y = target.scaleY;
      }
    }
  },
  _setTargetColor: function _setTargetColor(color) {
    var target = this._getTarget();

    target.color = color;
    target.opacity = color.a;
  },
  _getStateColor: function _getStateColor(state) {
    switch (state) {
      case State.NORMAL:
        return this.normalColor;

      case State.HOVER:
        return this.hoverColor;

      case State.PRESSED:
        return this.pressedColor;

      case State.DISABLED:
        return this.disabledColor;
    }
  },
  _getStateSprite: function _getStateSprite(state) {
    switch (state) {
      case State.NORMAL:
        return this.normalSprite;

      case State.HOVER:
        return this.hoverSprite;

      case State.PRESSED:
        return this.pressedSprite;

      case State.DISABLED:
        return this.disabledSprite;
    }
  },
  _setCurrentStateColor: function _setCurrentStateColor(color) {
    switch (this._getButtonState()) {
      case State.NORMAL:
        this.normalColor = color;
        break;

      case State.HOVER:
        this.hoverColor = color;
        break;

      case State.PRESSED:
        this.pressedColor = color;
        break;

      case State.DISABLED:
        this.disabledColor = color;
        break;
    }
  },
  _setCurrentStateSprite: function _setCurrentStateSprite(spriteFrame) {
    switch (this._getButtonState()) {
      case State.NORMAL:
        this.normalSprite = spriteFrame;
        break;

      case State.HOVER:
        this.hoverSprite = spriteFrame;
        break;

      case State.PRESSED:
        this.pressedSprite = spriteFrame;
        break;

      case State.DISABLED:
        this.disabledSprite = spriteFrame;
        break;
    }
  },
  update: function update(dt) {
    var target = this._getTarget();

    if (this._transitionFinished) return;
    if (this.transition !== Transition.COLOR && this.transition !== Transition.SCALE) return;
    this.time += dt;
    var ratio = 1.0;

    if (this.duration > 0) {
      ratio = this.time / this.duration;
    } // clamp ratio


    if (ratio >= 1) {
      ratio = 1;
    }

    if (this.transition === Transition.COLOR) {
      var color = this._fromColor.lerp(this._toColor, ratio);

      this._setTargetColor(color);
    } // Skip if _originalScale is invalid
    else if (this.transition === Transition.SCALE && this._originalScale) {
        target.scale = this._fromScale.lerp(this._toScale, ratio);
      }

    if (ratio === 1) {
      this._transitionFinished = true;
    }
  },
  _registerNodeEvent: function _registerNodeEvent() {
    this.node.on(cc.Node.EventType.TOUCH_START, this._onTouchBegan, this);
    this.node.on(cc.Node.EventType.TOUCH_MOVE, this._onTouchMove, this);
    this.node.on(cc.Node.EventType.TOUCH_END, this._onTouchEnded, this);
    this.node.on(cc.Node.EventType.TOUCH_CANCEL, this._onTouchCancel, this);
    this.node.on(cc.Node.EventType.MOUSE_ENTER, this._onMouseMoveIn, this);
    this.node.on(cc.Node.EventType.MOUSE_LEAVE, this._onMouseMoveOut, this);
  },
  _unregisterNodeEvent: function _unregisterNodeEvent() {
    this.node.off(cc.Node.EventType.TOUCH_START, this._onTouchBegan, this);
    this.node.off(cc.Node.EventType.TOUCH_MOVE, this._onTouchMove, this);
    this.node.off(cc.Node.EventType.TOUCH_END, this._onTouchEnded, this);
    this.node.off(cc.Node.EventType.TOUCH_CANCEL, this._onTouchCancel, this);
    this.node.off(cc.Node.EventType.MOUSE_ENTER, this._onMouseMoveIn, this);
    this.node.off(cc.Node.EventType.MOUSE_LEAVE, this._onMouseMoveOut, this);
  },
  _registerTargetEvent: function _registerTargetEvent(target) {
    if (CC_EDITOR) {
      target.on('spriteframe-changed', this._onTargetSpriteFrameChanged, this);
      target.on(cc.Node.EventType.COLOR_CHANGED, this._onTargetColorChanged, this);
    }

    target.on(cc.Node.EventType.SCALE_CHANGED, this._onTargetScaleChanged, this);
  },
  _unregisterTargetEvent: function _unregisterTargetEvent(target) {
    if (CC_EDITOR) {
      target.off('spriteframe-changed', this._onTargetSpriteFrameChanged, this);
      target.off(cc.Node.EventType.COLOR_CHANGED, this._onTargetColorChanged, this);
    }

    target.off(cc.Node.EventType.SCALE_CHANGED, this._onTargetScaleChanged, this);
  },
  _getTargetSprite: function _getTargetSprite(target) {
    var sprite = null;

    if (target) {
      sprite = target.getComponent(cc.Sprite);
    }

    return sprite;
  },
  _applyTarget: function _applyTarget() {
    var target = this._getTarget();

    this._sprite = this._getTargetSprite(target);

    if (!this._originalScale) {
      this._originalScale = cc.Vec2.ZERO;
    }

    this._originalScale.x = target.scaleX;
    this._originalScale.y = target.scaleY;

    this._registerTargetEvent(target);
  },
  // touch event handler
  _onTouchBegan: function _onTouchBegan(event) {
    if (!this.interactable || !this.enabledInHierarchy) return;
    this._pressed = true;

    this._updateState();

    event.stopPropagation();
  },
  _onTouchMove: function _onTouchMove(event) {
    if (!this.interactable || !this.enabledInHierarchy || !this._pressed) return; // mobile phone will not emit _onMouseMoveOut,
    // so we have to do hit test when touch moving

    var touch = event.touch;

    var hit = this.node._hitTest(touch.getLocation());

    var target = this._getTarget();

    var originalScale = this._originalScale;

    if (this.transition === Transition.SCALE && originalScale) {
      if (hit) {
        this._fromScale.x = originalScale.x;
        this._fromScale.y = originalScale.y;
        this._toScale.x = originalScale.x * this.zoomScale;
        this._toScale.y = originalScale.y * this.zoomScale;
        this._transitionFinished = false;
      } else {
        this.time = 0;
        this._transitionFinished = true;
        target.setScale(originalScale.x, originalScale.y);
      }
    } else {
      var state;

      if (hit) {
        state = State.PRESSED;
      } else {
        state = State.NORMAL;
      }

      this._applyTransition(state);
    }

    event.stopPropagation();
  },
  _onTouchEnded: function _onTouchEnded(event) {
    if (!this.interactable || !this.enabledInHierarchy) return;

    if (this._pressed) {
      cc.Component.EventHandler.emitEvents(this.clickEvents, event);
      this.node.emit('click', this);
    }

    this._pressed = false;

    this._updateState();

    event.stopPropagation();
  },
  _onTouchCancel: function _onTouchCancel() {
    if (!this.interactable || !this.enabledInHierarchy) return;
    this._pressed = false;

    this._updateState();
  },
  _onMouseMoveIn: function _onMouseMoveIn() {
    if (this._pressed || !this.interactable || !this.enabledInHierarchy) return;
    if (this.transition === Transition.SPRITE && !this.hoverSprite) return;

    if (!this._hovered) {
      this._hovered = true;

      this._updateState();
    }
  },
  _onMouseMoveOut: function _onMouseMoveOut() {
    if (this._hovered) {
      this._hovered = false;

      this._updateState();
    }
  },
  // state handler
  _updateState: function _updateState() {
    var state = this._getButtonState();

    this._applyTransition(state);

    this._updateDisabledState();
  },
  _getButtonState: function _getButtonState() {
    var state;

    if (!this.interactable) {
      state = State.DISABLED;
    } else if (this._pressed) {
      state = State.PRESSED;
    } else if (this._hovered) {
      state = State.HOVER;
    } else {
      state = State.NORMAL;
    }

    return state;
  },
  _updateColorTransitionImmediately: function _updateColorTransitionImmediately(state) {
    var color = this._getStateColor(state);

    this._setTargetColor(color);

    this._fromColor = color.clone();
    this._toColor = color;
  },
  _updateColorTransition: function _updateColorTransition(state) {
    if (CC_EDITOR || state === State.DISABLED) {
      this._updateColorTransitionImmediately(state);
    } else {
      var target = this._getTarget();

      var color = this._getStateColor(state);

      this._fromColor = target.color.clone();
      this._toColor = color;
      this.time = 0;
      this._transitionFinished = false;
    }
  },
  _updateSpriteTransition: function _updateSpriteTransition(state) {
    var sprite = this._getStateSprite(state);

    if (this._sprite && sprite) {
      this._sprite.spriteFrame = sprite;
    }
  },
  _updateScaleTransition: function _updateScaleTransition(state) {
    if (state === State.PRESSED) {
      this._zoomUp();
    } else {
      this._zoomBack();
    }
  },
  _zoomUp: function _zoomUp() {
    // skip before __preload()
    if (!this._originalScale) {
      return;
    }

    this._fromScale.x = this._originalScale.x;
    this._fromScale.y = this._originalScale.y;
    this._toScale.x = this._originalScale.x * this.zoomScale;
    this._toScale.y = this._originalScale.y * this.zoomScale;
    this.time = 0;
    this._transitionFinished = false;
  },
  _zoomBack: function _zoomBack() {
    // skip before __preload()
    if (!this._originalScale) {
      return;
    }

    var target = this._getTarget();

    this._fromScale.x = target.scaleX;
    this._fromScale.y = target.scaleY;
    this._toScale.x = this._originalScale.x;
    this._toScale.y = this._originalScale.y;
    this.time = 0;
    this._transitionFinished = false;
  },
  _updateTransition: function _updateTransition(oldTransition) {
    // Reset to normal data when change transition.
    if (oldTransition === Transition.COLOR) {
      this._updateColorTransitionImmediately(State.NORMAL);
    } else if (oldTransition === Transition.SPRITE) {
      this._updateSpriteTransition(State.NORMAL);
    }

    this._updateState();
  },
  _applyTransition: function _applyTransition(state) {
    var transition = this.transition;

    if (transition === Transition.COLOR) {
      this._updateColorTransition(state);
    } else if (transition === Transition.SPRITE) {
      this._updateSpriteTransition(state);
    } else if (transition === Transition.SCALE) {
      this._updateScaleTransition(state);
    }
  },
  _resizeNodeToTargetNode: CC_EDITOR && function () {
    this.node.setContentSize(this._getTarget().getContentSize());
  },
  _updateDisabledState: function _updateDisabledState() {
    if (this._sprite) {
      var useGrayMaterial = false;

      if (this.enableAutoGrayEffect) {
        if (!(this.transition === Transition.SPRITE && this.disabledSprite)) {
          if (!this.interactable) {
            useGrayMaterial = true;
          }
        }
      }

      this._switchGrayMaterial(useGrayMaterial, this._sprite);
    }
  }
});
cc.Button = module.exports = Button;
/**
 * !#en
 * Note: This event is emitted from the node to which the component belongs.
 * !#zh
 * 注意：此事件是从该组件所属的 Node 上面派发出来的，需要用 node.on 来监听。
 * @event click
 * @param {Event.EventCustom} event
 * @param {Button} button - The Button component.
 */
                    }
                    if (nodeEnv) {
                        __define(__module.exports, __require, __module);
                    }
                    else {
                        __quick_compile_engine__.registerModuleFunc(__filename, function () {
                            __define(__module.exports, __require, __module);
                        });
                    }
                })();
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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