
                (function() {
                    var nodeEnv = typeof require !== 'undefined' && typeof process !== 'undefined';
                    var __module = nodeEnv ? module : {exports:{}};
                    var __filename = 'engine-dev/cocos2d/core/components/CCAnimation.js';
                    var __require = nodeEnv ? function (request) {
                        return require(request);
                    } : function (request) {
                        return __quick_compile_engine__.require(request, __filename);
                    };
                    function __define (exports, require, module) {
                        if (!nodeEnv) {__quick_compile_engine__.registerModule(__filename, module);}"use strict";

/****************************************************************************
 Copyright (c) 2013-2016 Chukong Technologies Inc.
 Copyright (c) 2017-2018 Xiamen Yaji Software Co., Ltd.

 https://www.cocos.com/

 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated engine source code (the "Software"), a limited,
  worldwide, royalty-free, non-assignable, revocable and non-exclusive license
 to use Cocos Creator solely to develop games on your target platforms. You shall
  not use Cocos Creator software for developing other software or tools that's
  used for developing games. You are not granted to publish, distribute,
  sublicense, and/or sell copies of Cocos Creator.

 The software or tools in this License Agreement are licensed, not sold.
 Xiamen Yaji Software Co., Ltd. reserves all rights not expressly granted to you.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 THE SOFTWARE.
 ****************************************************************************/
var AnimationAnimator = require('../../animation/animation-animator');

var AnimationClip = require('../../animation/animation-clip');

var EventTarget = require('../event/event-target');

var js = require('../platform/js');

var equalClips = CC_EDITOR ? function (clip1, clip2) {
  return clip1 === clip2 || clip1 && clip2 && (clip1.name === clip2.name || clip1._uuid === clip2._uuid);
} : function (clip1, clip2) {
  return clip1 === clip2;
};
/**
 * !#en The event type supported by Animation
 * !#zh Animation 支持的事件类型
 * @class Animation.EventType
 * @static
 * @namespace Animationd
 */

var EventType = cc.Enum({
  /**
   * !#en Emit when begin playing animation
   * !#zh 开始播放时触发
   * @property {String} PLAY
   * @static
   */
  PLAY: 'play',

  /**
   * !#en Emit when stop playing animation
   * !#zh 停止播放时触发
   * @property {String} STOP
   * @static
   */
  STOP: 'stop',

  /**
   * !#en Emit when pause animation
   * !#zh 暂停播放时触发
   * @property {String} PAUSE   
   * @static
   */
  PAUSE: 'pause',

  /**
   * !#en Emit when resume animation
   * !#zh 恢复播放时触发
   * @property {String} RESUME
   * @static
   */
  RESUME: 'resume',

  /**
   * !#en If animation repeat count is larger than 1, emit when animation play to the last frame
   * !#zh 假如动画循环次数大于 1，当动画播放到最后一帧时触发
   * @property {String} LASTFRAME
   * @static
   */
  LASTFRAME: 'lastframe',

  /**
   * !#en Emit when finish playing animation
   * !#zh 动画播放完成时触发
   * @property {String} FINISHED
   * @static
   */
  FINISHED: 'finished'
});
/**
 * !#en The animation component is used to play back animations.
 *   
 * Animation provide several events to register：
 *  - play : Emit when begin playing animation
 *  - stop : Emit when stop playing animation
 *  - pause : Emit when pause animation
 *  - resume : Emit when resume animation
 *  - lastframe : If animation repeat count is larger than 1, emit when animation play to the last frame
 *  - finished : Emit when finish playing animation
 *
 * !#zh Animation 组件用于播放动画。
 *   
 * Animation 提供了一系列可注册的事件：
 *  - play : 开始播放时
 *  - stop : 停止播放时
 *  - pause : 暂停播放时
 *  - resume : 恢复播放时
 *  - lastframe : 假如动画循环次数大于 1，当动画播放到最后一帧时
 *  - finished : 动画播放完成时
 * 
 * @class Animation
 * @extends Component
 * @uses EventTarget
 */

var Animation = cc.Class({
  name: 'cc.Animation',
  "extends": require('./CCComponent'),
  mixins: [EventTarget],
  editor: CC_EDITOR && {
    menu: 'i18n:MAIN_MENU.component.others/Animation',
    help: 'i18n:COMPONENT.help_url.animation',
    executeInEditMode: true
  },
  statics: {
    EventType: EventType
  },
  ctor: function ctor() {
    cc.EventTarget.call(this); // The actual implement for Animation

    this._animator = null;
    this._nameToState = js.createMap(true);
    this._didInit = false;
    this._currentClip = null;
  },
  properties: {
    _defaultClip: {
      "default": null,
      type: AnimationClip
    },

    /**
     * !#en Animation will play the default clip when start game.
     * !#zh 在勾选自动播放或调用 play() 时默认播放的动画剪辑。
     * @property defaultClip
     * @type {AnimationClip}
     */
    defaultClip: {
      type: AnimationClip,
      get: function get() {
        return this._defaultClip;
      },
      set: function set(value) {
        if (!CC_EDITOR || cc.engine && cc.engine.isPlaying) {
          return;
        }

        if (this._defaultClip) {
          this.removeClip(this._defaultClip, true);
        }

        if (value) {
          this.addClip(value);
        }

        this._defaultClip = value;
      },
      tooltip: CC_DEV && 'i18n:COMPONENT.animation.default_clip'
    },

    /**
     * !#en Current played clip.
     * !#zh 当前播放的动画剪辑。
     * @property currentClip
     * @type {AnimationClip}
     */
    currentClip: {
      get: function get() {
        return this._currentClip;
      },
      set: function set(value) {
        this._currentClip = value;
      },
      type: AnimationClip,
      visible: false
    },
    // This property is used to watch clip changes in editor.
    // Don't use in your game, use addClip/removeClip instead.
    _writableClips: {
      get: function get() {
        return this._clips;
      },
      set: function set(val) {
        this._didInit = false;
        this._clips = val;

        this._init();
      },
      type: [AnimationClip]
    },

    /**
     * !#en All the clips used in this animation.
     * !#zh 通过脚本可以访问并播放的 AnimationClip 列表。
     * @property _clips
     * @type {AnimationClip[]}
     * @private
     */
    _clips: {
      "default": [],
      type: [AnimationClip],
      tooltip: CC_DEV && 'i18n:COMPONENT.animation.clips',
      visible: true
    },

    /**
     * !#en Whether the animation should auto play the default clip when start game.
     * !#zh 是否在运行游戏后自动播放默认动画剪辑。
     * @property playOnLoad
     * @type {Boolean}
     * @default true
     */
    playOnLoad: {
      "default": false,
      tooltip: CC_DEV && 'i18n:COMPONENT.animation.play_on_load'
    }
  },
  start: function start() {
    if (!CC_EDITOR && this.playOnLoad && this._defaultClip) {
      var isPlaying = this._animator && this._animator.isPlaying;

      if (!isPlaying) {
        var state = this.getAnimationState(this._defaultClip.name);

        this._animator.playState(state);
      }
    }
  },
  onEnable: function onEnable() {
    if (this._animator) {
      this._animator.resume();
    }
  },
  onDisable: function onDisable() {
    if (this._animator) {
      this._animator.pause();
    }
  },
  onDestroy: function onDestroy() {
    this.stop();
  },
  ///////////////////////////////////////////////////////////////////////////////
  // Public Methods
  ///////////////////////////////////////////////////////////////////////////////

  /**
   * !#en Get all the clips used in this animation.
   * !#zh 获取动画组件上的所有动画剪辑。
   * @method getClips
   * @return {AnimationClip[]}
   */
  getClips: function getClips() {
    return this._clips;
  },

  /**
   * !#en Plays an animation and stop other animations.
   * !#zh 播放指定的动画，并且停止当前正在播放动画。如果没有指定动画，则播放默认动画。
   * @method play
   * @param {String} [name] - The name of animation to play. If no name is supplied then the default animation will be played.
   * @param {Number} [startTime] - play an animation from startTime
   * @return {AnimationState} - The AnimationState of playing animation. In cases where the animation can't be played (ie, there is no default animation or no animation with the specified name), the function will return null.
   * @example
   * var animCtrl = this.node.getComponent(cc.Animation);
   * animCtrl.play("linear");
   */
  play: function play(name, startTime) {
    var state = this.playAdditive(name, startTime);

    this._animator.stopStatesExcept(state);

    return state;
  },

  /**
   * !#en
   * Plays an additive animation, it will not stop other animations.
   * If there are other animations playing, then will play several animations at the same time.
   * !#zh 播放指定的动画（将不会停止当前播放的动画）。如果没有指定动画，则播放默认动画。
   * @method playAdditive
   * @param {String} [name] - The name of animation to play. If no name is supplied then the default animation will be played.
   * @param {Number} [startTime] - play an animation from startTime
   * @return {AnimationState} - The AnimationState of playing animation. In cases where the animation can't be played (ie, there is no default animation or no animation with the specified name), the function will return null.
   * @example
   * // linear_1 and linear_2 at the same time playing.
   * var animCtrl = this.node.getComponent(cc.Animation);
   * animCtrl.playAdditive("linear_1");
   * animCtrl.playAdditive("linear_2");
   */
  playAdditive: function playAdditive(name, startTime) {
    this._init();

    var state = this.getAnimationState(name || this._defaultClip && this._defaultClip.name);

    if (state) {
      this.enabled = true;
      var animator = this._animator;

      if (animator.isPlaying && state.isPlaying) {
        if (state.isPaused) {
          animator.resumeState(state);
        } else {
          animator.stopState(state);
          animator.playState(state, startTime);
        }
      } else {
        animator.playState(state, startTime);
      } // Animation cannot be played when the component is not enabledInHierarchy.
      // That would cause an error for the animation lost the reference after destroying the node.
      // If users play the animation when the component is not enabledInHierarchy,
      // we pause the animator here so that it will automatically resume the animation when users enable the component.


      if (!this.enabledInHierarchy) {
        animator.pause();
      }

      this.currentClip = state.clip;
    }

    return state;
  },

  /**
   * !#en Stops an animation named name. If no name is supplied then stops all playing animations that were started with this Animation. <br/>
   * Stopping an animation also Rewinds it to the Start.
   * !#zh 停止指定的动画。如果没有指定名字，则停止当前正在播放的动画。
   * @method stop
   * @param {String} [name] - The animation to stop, if not supplied then stops all playing animations.
   */
  stop: function stop(name) {
    if (!this._didInit) {
      return;
    }

    if (name) {
      var state = this._nameToState[name];

      if (state) {
        this._animator.stopState(state);
      }
    } else {
      this._animator.stop();
    }
  },

  /**
   * !#en Pauses an animation named name. If no name is supplied then pauses all playing animations that were started with this Animation.
   * !#zh 暂停当前或者指定的动画。如果没有指定名字，则暂停当前正在播放的动画。
   * @method pause
   * @param {String} [name] - The animation to pauses, if not supplied then pauses all playing animations.
   */
  pause: function pause(name) {
    if (!this._didInit) {
      return;
    }

    if (name) {
      var state = this._nameToState[name];

      if (state) {
        this._animator.pauseState(state);
      }
    } else {
      this.enabled = false;
    }
  },

  /**
   * !#en Resumes an animation named name. If no name is supplied then resumes all paused animations that were started with this Animation.
   * !#zh 重新播放指定的动画，如果没有指定名字，则重新播放当前正在播放的动画。
   * @method resume
   * @param {String} [name] - The animation to resumes, if not supplied then resumes all paused animations.
   */
  resume: function resume(name) {
    if (!this._didInit) {
      return;
    }

    if (name) {
      var state = this._nameToState[name];

      if (state) {
        this._animator.resumeState(state);
      }
    } else {
      this.enabled = true;
    }
  },

  /**
   * !#en Make an animation named name go to the specified time. If no name is supplied then make all animations go to the specified time.
   * !#zh 设置指定动画的播放时间。如果没有指定名字，则设置当前播放动画的播放时间。
   * @method setCurrentTime
   * @param {Number} [time] - The time to go to
   * @param {String} [name] - Specified animation name, if not supplied then make all animations go to the time.
   */
  setCurrentTime: function setCurrentTime(time, name) {
    this._init();

    if (name) {
      var state = this._nameToState[name];

      if (state) {
        this._animator.setStateTime(state, time);
      }
    } else {
      this._animator.setStateTime(time);
    }
  },

  /**
   * !#en Returns the animation state named name. If no animation with the specified name, the function will return null.
   * !#zh 获取当前或者指定的动画状态，如果未找到指定动画剪辑则返回 null。
   * @method getAnimationState
   * @param {String} name
   * @return {AnimationState}
   */
  getAnimationState: function getAnimationState(name) {
    this._init();

    var state = this._nameToState[name];

    if (CC_EDITOR && (!state || !cc.js.array.contains(this._clips, state.clip))) {
      this._didInit = false;

      if (this._animator) {
        this._animator.stop();
      }

      this._init();

      state = this._nameToState[name];
    }

    if (state && !state.curveLoaded) {
      this._animator._reloadClip(state);
    }

    return state || null;
  },

  /**
   * !#en Adds a clip to the animation with name newName. If a clip with that name already exists it will be replaced with the new clip.
   * !#zh 添加动画剪辑，并且可以重新设置该动画剪辑的名称。
   * @method addClip
   * @param {AnimationClip} clip - the clip to add
   * @param {String} [newName]
   * @return {AnimationState} - The AnimationState which gives full control over the animation clip.
   */
  addClip: function addClip(clip, newName) {
    if (!clip) {
      cc.warnID(3900);
      return;
    }

    this._init(); // add clip


    if (!cc.js.array.contains(this._clips, clip)) {
      this._clips.push(clip);
    } // replace same name clip


    newName = newName || clip.name;
    var oldState = this._nameToState[newName];

    if (oldState) {
      if (oldState.clip === clip) {
        return oldState;
      } else {
        var index = this._clips.indexOf(oldState.clip);

        if (index !== -1) {
          this._clips.splice(index, 1);
        }
      }
    } // replace state


    var newState = new cc.AnimationState(clip, newName);
    this._nameToState[newName] = newState;
    return newState;
  },

  /**
   * !#en 
   * Remove clip from the animation list. This will remove the clip and any animation states based on it.
   * If there are animation states depand on the clip are playing or clip is defaultClip, it will not delete the clip.
   * But if force is true, then will always remove the clip and any animation states based on it. If clip is defaultClip, defaultClip will be reset to null
   * !#zh
   * 从动画列表中移除指定的动画剪辑，<br/>
   * 如果依赖于 clip 的 AnimationState 正在播放或者 clip 是 defaultClip 的话，默认是不会删除 clip 的。
   * 但是如果 force 参数为 true，则会强制停止该动画，然后移除该动画剪辑和相关的动画。这时候如果 clip 是 defaultClip，defaultClip 将会被重置为 null。
   * @method removeClip
   * @param {AnimationClip} clip
   * @param {Boolean} [force=false] - If force is true, then will always remove the clip and any animation states based on it.
   */
  removeClip: function removeClip(clip, force) {
    if (!clip) {
      cc.warnID(3901);
      return;
    }

    this._init();

    var state;

    for (var name in this._nameToState) {
      state = this._nameToState[name];

      if (equalClips(state.clip, clip)) {
        break;
      }
    }

    if (clip === this._defaultClip) {
      if (force) this._defaultClip = null;else {
        if (!CC_TEST) cc.warnID(3902);
        return;
      }
    }

    if (state && state.isPlaying) {
      if (force) this.stop(state.name);else {
        if (!CC_TEST) cc.warnID(3903);
        return;
      }
    }

    this._clips = this._clips.filter(function (item) {
      return !equalClips(item, clip);
    });

    if (state) {
      delete this._nameToState[state.name];
    }
  },

  /**
   * !#en
   * Samples animations at the current state.<br/>
   * This is useful when you explicitly want to set up some animation state, and sample it once.
   * !#zh 对指定或当前动画进行采样。你可以手动将动画设置到某一个状态，然后采样一次。
   * @method sample
   * @param {String} name
   */
  sample: function sample(name) {
    this._init();

    if (name) {
      var state = this._nameToState[name];

      if (state) {
        state.sample();
      }
    } else {
      this._animator.sample();
    }
  },

  /**
   * !#en 
   * Register animation event callback.
   * The event arguments will provide the AnimationState which emit the event.
   * When play an animation, will auto register the event callback to the AnimationState, and unregister the event callback from the AnimationState when animation stopped.
   * !#zh
   * 注册动画事件回调。
   * 回调的事件里将会附上发送事件的 AnimationState。
   * 当播放一个动画时，会自动将事件注册到对应的 AnimationState 上，停止播放时会将事件从这个 AnimationState 上取消注册。
   * @method on
   * @param {String} type - A string representing the event type to listen for.
   * @param {Function} callback - The callback that will be invoked when the event is dispatched.
   *                              The callback is ignored if it is a duplicate (the callbacks are unique).
   * @param {cc.AnimationState} state 
   * @param {Object} [target] - The target (this object) to invoke the callback, can be null
   * @param {Boolean} [useCapture=false] - When set to true, the capture argument prevents callback
   *                              from being invoked when the event's eventPhase attribute value is BUBBLING_PHASE.
   *                              When false, callback will NOT be invoked when event's eventPhase attribute value is CAPTURING_PHASE.
   *                              Either way, callback will be invoked when event's eventPhase attribute value is AT_TARGET.
   *
   * @return {Function} - Just returns the incoming callback so you can save the anonymous function easier.
   * @typescript
   * on(type: string, callback: (event: Event.EventCustom) => void, target?: any, useCapture?: boolean): (event: Event.EventCustom) => void
   * on<T>(type: string, callback: (event: T) => void, target?: any, useCapture?: boolean): (event: T) => void
   * @example
   * onPlay: function (type, state) {
   *     // callback
   * }
   * 
   * // register event to all animation
   * animation.on('play', this.onPlay, this);
   */
  on: function on(type, callback, target, useCapture) {
    this._init();

    var ret = this._EventTargetOn(type, callback, target, useCapture);

    if (type === 'lastframe') {
      var states = this._nameToState;

      for (var name in states) {
        states[name]._lastframeEventOn = true;
      }
    }

    return ret;
  },

  /**
   * !#en
   * Unregister animation event callback.
   * !#zh
   * 取消注册动画事件回调。
   * @method off
   * @param {String} type - A string representing the event type being removed.
   * @param {Function} [callback] - The callback to remove.
   * @param {Object} [target] - The target (this object) to invoke the callback, if it's not given, only callback without target will be removed
   * @param {Boolean} [useCapture=false] - Specifies whether the callback being removed was registered as a capturing callback or not.
   *                              If not specified, useCapture defaults to false. If a callback was registered twice,
   *                              one with capture and one without, each must be removed separately. Removal of a capturing callback
   *                              does not affect a non-capturing version of the same listener, and vice versa.
   *
   * @example
   * // unregister event to all animation
   * animation.off('play', this.onPlay, this);
   */
  off: function off(type, callback, target, useCapture) {
    this._init();

    if (type === 'lastframe') {
      var states = this._nameToState;

      for (var name in states) {
        states[name]._lastframeEventOn = false;
      }
    }

    this._EventTargetOff(type, callback, target, useCapture);
  },
  ///////////////////////////////////////////////////////////////////////////////
  // Internal Methods
  ///////////////////////////////////////////////////////////////////////////////
  // Dont forget to call _init before every actual process in public methods.
  // Just invoking _init by onLoad is not enough because onLoad is called only if the entity is active.
  _init: function _init() {
    if (this._didInit) {
      return;
    }

    this._didInit = true;
    this._animator = new AnimationAnimator(this.node, this);

    this._createStates();
  },
  _createStates: function _createStates() {
    this._nameToState = js.createMap(true); // create animation states

    var state = null;
    var defaultClipState = false;

    for (var i = 0; i < this._clips.length; ++i) {
      var clip = this._clips[i];

      if (clip) {
        state = new cc.AnimationState(clip);

        if (CC_EDITOR) {
          this._animator._reloadClip(state);
        }

        this._nameToState[state.name] = state;

        if (equalClips(this._defaultClip, clip)) {
          defaultClipState = state;
        }
      }
    }

    if (this._defaultClip && !defaultClipState) {
      state = new cc.AnimationState(this._defaultClip);

      if (CC_EDITOR) {
        this._animator._reloadClip(state);
      }

      this._nameToState[state.name] = state;
    }
  }
});
Animation.prototype._EventTargetOn = EventTarget.prototype.on;
Animation.prototype._EventTargetOff = EventTarget.prototype.off;
cc.Animation = module.exports = Animation;
                    }
                    if (nodeEnv) {
                        __define(__module.exports, __require, __module);
                    }
                    else {
                        __quick_compile_engine__.registerModuleFunc(__filename, function () {
                            __define(__module.exports, __require, __module);
                        });
                    }
                })();
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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