
                (function() {
                    var nodeEnv = typeof require !== 'undefined' && typeof process !== 'undefined';
                    var __module = nodeEnv ? module : {exports:{}};
                    var __filename = 'engine-dev/cocos2d/audio/CCAudioEngine.js';
                    var __require = nodeEnv ? function (request) {
                        return require(request);
                    } : function (request) {
                        return __quick_compile_engine__.require(request, __filename);
                    };
                    function __define (exports, require, module) {
                        if (!nodeEnv) {__quick_compile_engine__.registerModule(__filename, module);}"use strict";

/****************************************************************************
 Copyright (c) 2008-2010 Ricardo Quesada
 Copyright (c) 2011-2012 cocos2d-x.org
 Copyright (c) 2013-2016 Chukong Technologies Inc.
 Copyright (c) 2017-2018 Xiamen Yaji Software Co., Ltd.

 http://www.cocos2d-x.org

 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated documentation files (the "Software"), to deal
 in the Software without restriction, including without limitation the rights
 to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the Software is
 furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 THE SOFTWARE.
 ****************************************************************************/
var Audio = require('./CCAudio');

var AudioClip = require('../core/assets/CCAudioClip');

var js = cc.js;
var _instanceId = 0;

var _id2audio = js.createMap(true);

var _url2id = {};
var _audioPool = [];

var recycleAudio = function recycleAudio(audio) {
  // In case repeatly recycle audio when users call audio.stop when audio finish playing
  if (!audio._shouldRecycleOnEnded) {
    return;
  }

  audio._finishCallback = null;
  audio.off('ended');
  audio.off('stop');
  audio.src = null; // In case repeatly recycle audio

  if (!_audioPool.includes(audio)) {
    if (_audioPool.length < 32) {
      _audioPool.push(audio);
    } else {
      audio.destroy();
    }
  }

  audio._shouldRecycleOnEnded = false;
};

var getAudioFromPath = function getAudioFromPath(path) {
  var id = _instanceId++;
  var list = _url2id[path];

  if (!list) {
    list = _url2id[path] = [];
  }

  if (audioEngine._maxAudioInstance <= list.length) {
    var oldId = list.shift();
    var oldAudio = getAudioFromId(oldId); // Stop will recycle audio automatically by event callback

    oldAudio.stop();
  }

  var audio = _audioPool.pop() || new Audio();

  var callback = function callback() {
    var audioInList = getAudioFromId(this.id);

    if (audioInList) {
      delete _id2audio[this.id];
      var index = list.indexOf(this.id);
      cc.js.array.fastRemoveAt(list, index);
    }

    recycleAudio(this);
  };

  audio.on('ended', function () {
    if (this._finishCallback) {
      this._finishCallback();
    }

    callback.call(this);
  }, audio);
  audio.on('stop', callback, audio);
  audio.id = id;
  _id2audio[id] = audio;
  list.push(id);
  return audio;
};

var getAudioFromId = function getAudioFromId(id) {
  return _id2audio[id];
};

var handleVolume = function handleVolume(volume) {
  if (volume === undefined) {
    // set default volume as 1
    volume = 1;
  } else if (typeof volume === 'string') {
    volume = Number.parseFloat(volume);
  }

  return volume;
};
/**
 * !#en cc.audioEngine is the singleton object, it provide simple audio APIs.
 * !#zh
 * cc.audioengine是单例对象。<br/>
 * 主要用来播放音频，播放的时候会返回一个 audioID，之后都可以通过这个 audioID 来操作这个音频对象。<br/>
 * 不使用的时候，请使用 cc.audioEngine.uncache(filePath); 进行资源释放 <br/>
 * 注意：<br/>
 * 在 Android 系统浏览器上，不同浏览器，不同版本的效果不尽相同。<br/>
 * 比如说：大多数浏览器都需要用户物理交互才可以开始播放音效，有一些不支持 WebAudio，<br/>
 * 有一些不支持多音轨播放。总之如果对音乐依赖比较强，请做尽可能多的测试。
 * @class audioEngine
 * @static
 */


var audioEngine = {
  AudioState: Audio.State,
  _maxWebAudioSize: 2097152,
  // 2048kb * 1024
  _maxAudioInstance: 24,
  _id2audio: _id2audio,

  /**
   * !#en Play audio.
   * !#zh 播放音频
   * @method play
   * @param {AudioClip} clip - The audio clip to play.
   * @param {Boolean} loop - Whether the music loop or not.
   * @param {Number} volume - Volume size.
   * @return {Number} audioId
   * @example
   * cc.loader.loadRes(url, cc.AudioClip, function (err, clip) {
   *     var audioID = cc.audioEngine.play(clip, false, 0.5);
   * });
   */
  play: function play(clip, loop, volume
  /*, profile*/
  ) {
    var path = clip;
    var audio;

    if (typeof clip === 'string') {
      // backward compatibility since 1.10
      cc.warnID(8401, 'cc.audioEngine', 'cc.AudioClip', 'AudioClip', 'cc.AudioClip', 'audio');
      path = clip; // load clip

      audio = getAudioFromPath(path);

      AudioClip._loadByUrl(path, function (err, clip) {
        if (clip) {
          audio.src = clip;
        }
      });
    } else {
      if (!clip) {
        return;
      }

      path = clip.nativeUrl;
      audio = getAudioFromPath(path);
      audio.src = clip;
    }

    audio._shouldRecycleOnEnded = true;
    audio.setLoop(loop || false);
    volume = handleVolume(volume);
    audio.setVolume(volume);
    audio.play();
    return audio.id;
  },

  /**
   * !#en Set audio loop.
   * !#zh 设置音频是否循环。
   * @method setLoop
   * @param {Number} audioID - audio id.
   * @param {Boolean} loop - Whether cycle.
   * @example
   * cc.audioEngine.setLoop(id, true);
   */
  setLoop: function setLoop(audioID, loop) {
    var audio = getAudioFromId(audioID);
    if (!audio || !audio.setLoop) return;
    audio.setLoop(loop);
  },

  /**
   * !#en Get audio cycle state.
   * !#zh 获取音频的循环状态。
   * @method isLoop
   * @param {Number} audioID - audio id.
   * @return {Boolean} Whether cycle.
   * @example
   * cc.audioEngine.isLoop(id);
   */
  isLoop: function isLoop(audioID) {
    var audio = getAudioFromId(audioID);
    if (!audio || !audio.getLoop) return false;
    return audio.getLoop();
  },

  /**
   * !#en Set the volume of audio.
   * !#zh 设置音量（0.0 ~ 1.0）。
   * @method setVolume
   * @param {Number} audioID - audio id.
   * @param {Number} volume - Volume must be in 0.0~1.0 .
   * @example
   * cc.audioEngine.setVolume(id, 0.5);
   */
  setVolume: function setVolume(audioID, volume) {
    var audio = getAudioFromId(audioID);

    if (audio) {
      audio.setVolume(volume);
    }
  },

  /**
   * !#en The volume of the music max value is 1.0,the min value is 0.0 .
   * !#zh 获取音量（0.0 ~ 1.0）。
   * @method getVolume
   * @param {Number} audioID - audio id.
   * @return {Number}
   * @example
   * var volume = cc.audioEngine.getVolume(id);
   */
  getVolume: function getVolume(audioID) {
    var audio = getAudioFromId(audioID);
    return audio ? audio.getVolume() : 1;
  },

  /**
   * !#en Set current time
   * !#zh 设置当前的音频时间。
   * @method setCurrentTime
   * @param {Number} audioID - audio id.
   * @param {Number} sec - current time.
   * @return {Boolean}
   * @example
   * cc.audioEngine.setCurrentTime(id, 2);
   */
  setCurrentTime: function setCurrentTime(audioID, sec) {
    var audio = getAudioFromId(audioID);

    if (audio) {
      audio.setCurrentTime(sec);
      return true;
    } else {
      return false;
    }
  },

  /**
   * !#en Get current time
   * !#zh 获取当前的音频播放时间。
   * @method getCurrentTime
   * @param {Number} audioID - audio id.
   * @return {Number} audio current time.
   * @example
   * var time = cc.audioEngine.getCurrentTime(id);
   */
  getCurrentTime: function getCurrentTime(audioID) {
    var audio = getAudioFromId(audioID);
    return audio ? audio.getCurrentTime() : 0;
  },

  /**
   * !#en Get audio duration
   * !#zh 获取音频总时长。
   * @method getDuration
   * @param {Number} audioID - audio id.
   * @return {Number} audio duration.
   * @example
   * var time = cc.audioEngine.getDuration(id);
   */
  getDuration: function getDuration(audioID) {
    var audio = getAudioFromId(audioID);
    return audio ? audio.getDuration() : 0;
  },

  /**
   * !#en Get audio state
   * !#zh 获取音频状态。
   * @method getState
   * @param {Number} audioID - audio id.
   * @return {audioEngine.AudioState} audio duration.
   * @example
   * var state = cc.audioEngine.getState(id);
   */
  getState: function getState(audioID) {
    var audio = getAudioFromId(audioID);
    return audio ? audio.getState() : this.AudioState.ERROR;
  },

  /**
   * !#en Set Audio finish callback
   * !#zh 设置一个音频结束后的回调
   * @method setFinishCallback
   * @param {Number} audioID - audio id.
   * @param {Function} callback - loaded callback.
   * @example
   * cc.audioEngine.setFinishCallback(id, function () {});
   */
  setFinishCallback: function setFinishCallback(audioID, callback) {
    var audio = getAudioFromId(audioID);
    if (!audio) return;
    audio._finishCallback = callback;
  },

  /**
   * !#en Pause playing audio.
   * !#zh 暂停正在播放音频。
   * @method pause
   * @param {Number} audioID - The return value of function play.
   * @example
   * cc.audioEngine.pause(audioID);
   */
  pause: function pause(audioID) {
    var audio = getAudioFromId(audioID);

    if (audio) {
      audio.pause();
      return true;
    } else {
      return false;
    }
  },
  _pauseIDCache: [],

  /**
   * !#en Pause all playing audio
   * !#zh 暂停现在正在播放的所有音频。
   * @method pauseAll
   * @example
   * cc.audioEngine.pauseAll();
   */
  pauseAll: function pauseAll() {
    for (var id in _id2audio) {
      var audio = _id2audio[id];
      var state = audio.getState();

      if (state === Audio.State.PLAYING) {
        this._pauseIDCache.push(id);

        audio.pause();
      }
    }
  },

  /**
   * !#en Resume playing audio.
   * !#zh 恢复播放指定的音频。
   * @method resume
   * @param {Number} audioID - The return value of function play.
   * @example
   * cc.audioEngine.resume(audioID);
   */
  resume: function resume(audioID) {
    var audio = getAudioFromId(audioID);

    if (audio) {
      audio.resume();
    }
  },

  /**
   * !#en Resume all playing audio.
   * !#zh 恢复播放所有之前暂停的所有音频。
   * @method resumeAll
   * @example
   * cc.audioEngine.resumeAll();
   */
  resumeAll: function resumeAll() {
    for (var i = 0; i < this._pauseIDCache.length; ++i) {
      var id = this._pauseIDCache[i];
      var audio = getAudioFromId(id);
      if (audio) audio.resume();
    }

    this._pauseIDCache.length = 0;
  },

  /**
   * !#en Stop playing audio.
   * !#zh 停止播放指定音频。
   * @method stop
   * @param {Number} audioID - The return value of function play.
   * @example
   * cc.audioEngine.stop(audioID);
   */
  stop: function stop(audioID) {
    var audio = getAudioFromId(audioID);

    if (audio) {
      // Stop will recycle audio automatically by event callback
      audio.stop();
      return true;
    } else {
      return false;
    }
  },

  /**
   * !#en Stop all playing audio.
   * !#zh 停止正在播放的所有音频。
   * @method stopAll
   * @example
   * cc.audioEngine.stopAll();
   */
  stopAll: function stopAll() {
    for (var id in _id2audio) {
      var audio = _id2audio[id];

      if (audio) {
        // Stop will recycle audio automatically by event callback
        audio.stop();
      }
    }
  },

  /**
   * !#en Set up an audio can generate a few examples.
   * !#zh 设置一个音频可以设置几个实例
   * @method setMaxAudioInstance
   * @param {Number} num - a number of instances to be created from within an audio
   * @example
   * cc.audioEngine.setMaxAudioInstance(20);
   */
  setMaxAudioInstance: function setMaxAudioInstance(num) {
    this._maxAudioInstance = num;
  },

  /**
   * !#en Getting audio can produce several examples.
   * !#zh 获取一个音频可以设置几个实例
   * @method getMaxAudioInstance
   * @return {Number} a - number of instances to be created from within an audio
   * @example
   * cc.audioEngine.getMaxAudioInstance();
   */
  getMaxAudioInstance: function getMaxAudioInstance() {
    return this._maxAudioInstance;
  },

  /**
   * !#en Unload the preloaded audio from internal buffer.
   * !#zh 卸载预加载的音频。
   * @method uncache
   * @param {AudioClip} clip
   * @example
   * cc.audioEngine.uncache(filePath);
   */
  uncache: function uncache(clip) {
    var filePath = clip;

    if (typeof clip === 'string') {
      // backward compatibility since 1.10
      cc.warnID(8401, 'cc.audioEngine', 'cc.AudioClip', 'AudioClip', 'cc.AudioClip', 'audio');
      filePath = clip;
    } else {
      if (!clip) {
        return;
      }

      filePath = clip.nativeUrl;
    }

    var list = _url2id[filePath];
    if (!list) return;

    while (list.length > 0) {
      var id = list.pop();
      var audio = _id2audio[id];

      if (audio) {
        // Stop will recycle audio automatically by event callback
        audio.stop();
        delete _id2audio[id];
      }
    }
  },

  /**
   * !#en Unload all audio from internal buffer.
   * !#zh 卸载所有音频。
   * @method uncacheAll
   * @example
   * cc.audioEngine.uncacheAll();
   */
  uncacheAll: function uncacheAll() {
    this.stopAll();
    var audio;

    for (var id in _id2audio) {
      audio = _id2audio[id];

      if (audio) {
        audio.destroy();
      }
    }

    while (audio = _audioPool.pop()) {
      audio.destroy();
    }

    _id2audio = js.createMap(true);
    _url2id = {};
  },

  /**
   * !#en Gets an audio profile by name.
   *
   * @param profileName A name of audio profile.
   * @return The audio profile.
   */
  getProfile: function getProfile(profileName) {},

  /**
   * !#en Preload audio file.
   * !#zh 预加载一个音频
   * @method preload
   * @param {String} filePath - The file path of an audio.
   * @param {Function} [callback] - The callback of an audio.
   * @example
   * cc.audioEngine.preload(path);
   * @deprecated `cc.audioEngine.preload` is deprecated, use `cc.loader.loadRes(url, cc.AudioClip)` instead please.
   */
  preload: function preload(filePath, callback) {
    if (CC_DEBUG) {
      cc.warn('`cc.audioEngine.preload` is deprecated, use `cc.loader.loadRes(url, cc.AudioClip)` instead please.');
    }

    cc.loader.load(filePath, callback && function (error) {
      if (!error) {
        callback();
      }
    });
  },

  /**
   * !#en Set a size, the unit is KB. Over this size is directly resolved into DOM nodes.
   * !#zh 设置一个以 KB 为单位的尺寸，大于这个尺寸的音频在加载的时候会强制使用 dom 方式加载
   * @method setMaxWebAudioSize
   * @param {Number} kb - The file path of an audio.
   * @example
   * cc.audioEngine.setMaxWebAudioSize(300);
   */
  // Because webAudio takes up too much memory，So allow users to manually choose
  setMaxWebAudioSize: function setMaxWebAudioSize(kb) {
    this._maxWebAudioSize = kb * 1024;
  },
  _breakCache: null,
  _break: function _break() {
    this._breakCache = [];

    for (var id in _id2audio) {
      var audio = _id2audio[id];
      var state = audio.getState();

      if (state === Audio.State.PLAYING) {
        this._breakCache.push(id);

        audio.pause();
      }
    }
  },
  _restore: function _restore() {
    if (!this._breakCache) return;

    while (this._breakCache.length > 0) {
      var id = this._breakCache.pop();

      var audio = getAudioFromId(id);
      if (audio && audio.resume) audio.resume();
    }

    this._breakCache = null;
  },
  ///////////////////////////////
  // Classification of interface
  _music: {
    id: -1,
    loop: false,
    volume: 1
  },
  _effect: {
    volume: 1,
    pauseCache: []
  },

  /**
   * !#en Play background music
   * !#zh 播放背景音乐
   * @method playMusic
   * @param {AudioClip} clip - The audio clip to play.
   * @param {Boolean} loop - Whether the music loop or not.
   * @return {Number} audioId
   * @example
   * cc.loader.loadRes(url, cc.AudioClip, function (err, clip) {
   *     var audioID = cc.audioEngine.playMusic(clip, false);
   * });
   */
  playMusic: function playMusic(clip, loop) {
    var music = this._music;
    this.stop(music.id);
    music.id = this.play(clip, loop, music.volume);
    music.loop = loop;
    return music.id;
  },

  /**
   * !#en Stop background music.
   * !#zh 停止播放背景音乐。
   * @method stopMusic
   * @example
   * cc.audioEngine.stopMusic();
   */
  stopMusic: function stopMusic() {
    this.stop(this._music.id);
  },

  /**
   * !#en Pause the background music.
   * !#zh 暂停播放背景音乐。
   * @method pauseMusic
   * @example
   * cc.audioEngine.pauseMusic();
   */
  pauseMusic: function pauseMusic() {
    this.pause(this._music.id);
    return this._music.id;
  },

  /**
   * !#en Resume playing background music.
   * !#zh 恢复播放背景音乐。
   * @method resumeMusic
   * @example
   * cc.audioEngine.resumeMusic();
   */
  resumeMusic: function resumeMusic() {
    this.resume(this._music.id);
    return this._music.id;
  },

  /**
   * !#en Get the volume(0.0 ~ 1.0).
   * !#zh 获取音量（0.0 ~ 1.0）。
   * @method getMusicVolume
   * @return {Number}
   * @example
   * var volume = cc.audioEngine.getMusicVolume();
   */
  getMusicVolume: function getMusicVolume() {
    return this._music.volume;
  },

  /**
   * !#en Set the background music volume.
   * !#zh 设置背景音乐音量（0.0 ~ 1.0）。
   * @method setMusicVolume
   * @param {Number} volume - Volume must be in 0.0~1.0.
   * @example
   * cc.audioEngine.setMusicVolume(0.5);
   */
  setMusicVolume: function setMusicVolume(volume) {
    volume = handleVolume(volume);
    var music = this._music;
    music.volume = volume;
    this.setVolume(music.id, music.volume);
    return music.volume;
  },

  /**
   * !#en Background music playing state
   * !#zh 背景音乐是否正在播放
   * @method isMusicPlaying
   * @return {Boolean}
   * @example
   * cc.audioEngine.isMusicPlaying();
   */
  isMusicPlaying: function isMusicPlaying() {
    return this.getState(this._music.id) === this.AudioState.PLAYING;
  },

  /**
   * !#en Play effect audio.
   * !#zh 播放音效
   * @method playEffect
   * @param {AudioClip} clip - The audio clip to play.
   * @param {Boolean} loop - Whether the music loop or not.
   * @return {Number} audioId
   * @example
   * cc.loader.loadRes(url, cc.AudioClip, function (err, clip) {
   *     var audioID = cc.audioEngine.playEffect(clip, false);
   * });
   */
  playEffect: function playEffect(clip, loop) {
    return this.play(clip, loop || false, this._effect.volume);
  },

  /**
   * !#en Set the volume of effect audio.
   * !#zh 设置音效音量（0.0 ~ 1.0）。
   * @method setEffectsVolume
   * @param {Number} volume - Volume must be in 0.0~1.0.
   * @example
   * cc.audioEngine.setEffectsVolume(0.5);
   */
  setEffectsVolume: function setEffectsVolume(volume) {
    volume = handleVolume(volume);
    var musicId = this._music.id;
    this._effect.volume = volume;

    for (var id in _id2audio) {
      var audio = _id2audio[id];
      if (!audio || audio.id === musicId) continue;
      audioEngine.setVolume(id, volume);
    }
  },

  /**
   * !#en The volume of the effect audio max value is 1.0,the min value is 0.0 .
   * !#zh 获取音效音量（0.0 ~ 1.0）。
   * @method getEffectsVolume
   * @return {Number}
   * @example
   * var volume = cc.audioEngine.getEffectsVolume();
   */
  getEffectsVolume: function getEffectsVolume() {
    return this._effect.volume;
  },

  /**
   * !#en Pause effect audio.
   * !#zh 暂停播放音效。
   * @method pauseEffect
   * @param {Number} audioID - audio id.
   * @example
   * cc.audioEngine.pauseEffect(audioID);
   */
  pauseEffect: function pauseEffect(audioID) {
    return this.pause(audioID);
  },

  /**
   * !#en Stop playing all the sound effects.
   * !#zh 暂停播放所有音效。
   * @method pauseAllEffects
   * @example
   * cc.audioEngine.pauseAllEffects();
   */
  pauseAllEffects: function pauseAllEffects() {
    var musicId = this._music.id;
    var effect = this._effect;
    effect.pauseCache.length = 0;

    for (var id in _id2audio) {
      var audio = _id2audio[id];
      if (!audio || audio.id === musicId) continue;
      var state = audio.getState();

      if (state === this.AudioState.PLAYING) {
        effect.pauseCache.push(id);
        audio.pause();
      }
    }
  },

  /**
   * !#en Resume effect audio.
   * !#zh 恢复播放音效音频。
   * @method resumeEffect
   * @param {Number} audioID - The return value of function play.
   * @example
   * cc.audioEngine.resumeEffect(audioID);
   */
  resumeEffect: function resumeEffect(id) {
    this.resume(id);
  },

  /**
   * !#en Resume all effect audio.
   * !#zh 恢复播放所有之前暂停的音效。
   * @method resumeAllEffects
   * @example
   * cc.audioEngine.resumeAllEffects();
   */
  resumeAllEffects: function resumeAllEffects() {
    var pauseIDCache = this._effect.pauseCache;

    for (var i = 0; i < pauseIDCache.length; ++i) {
      var id = pauseIDCache[i];
      var audio = _id2audio[id];
      if (audio) audio.resume();
    }
  },

  /**
   * !#en Stop playing the effect audio.
   * !#zh 停止播放音效。
   * @method stopEffect
   * @param {Number} audioID - audio id.
   * @example
   * cc.audioEngine.stopEffect(id);
   */
  stopEffect: function stopEffect(audioID) {
    return this.stop(audioID);
  },

  /**
   * !#en Stop playing all the effects.
   * !#zh 停止播放所有音效。
   * @method stopAllEffects
   * @example
   * cc.audioEngine.stopAllEffects();
   */
  stopAllEffects: function stopAllEffects() {
    var musicId = this._music.id;

    for (var id in _id2audio) {
      var audio = _id2audio[id];
      if (!audio || audio.id === musicId) continue;
      var state = audio.getState();

      if (state === audioEngine.AudioState.PLAYING) {
        audio.stop();
      }
    }
  }
};
module.exports = cc.audioEngine = audioEngine;
                    }
                    if (nodeEnv) {
                        __define(__module.exports, __require, __module);
                    }
                    else {
                        __quick_compile_engine__.registerModuleFunc(__filename, function () {
                            __define(__module.exports, __require, __module);
                        });
                    }
                })();
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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