
                (function() {
                    var nodeEnv = typeof require !== 'undefined' && typeof process !== 'undefined';
                    var __module = nodeEnv ? module : {exports:{}};
                    var __filename = 'engine-dev/cocos2d/actions/tween.js';
                    var __require = nodeEnv ? function (request) {
                        return require(request);
                    } : function (request) {
                        return __quick_compile_engine__.require(request, __filename);
                    };
                    function __define (exports, require, module) {
                        if (!nodeEnv) {__quick_compile_engine__.registerModule(__filename, module);}"use strict";

var _bezier = require("../animation/bezier");

var _tweenID = 0;
var TweenAction = cc.Class({
  name: 'cc.TweenAction',
  "extends": cc.ActionInterval,
  ctor: function ctor(duration, props, opts) {
    this._opts = opts = opts || Object.create(null);
    this._props = Object.create(null); // global easing or progress used for this action

    opts.progress = opts.progress || this.progress;

    if (opts.easing && typeof opts.easing === 'string') {
      var easingName = opts.easing;
      opts.easing = cc.easing[easingName];
      !opts.easing && cc.warnID(1031, easingName);
    }

    var relative = this._opts.relative;

    for (var name in props) {
      var value = props[name]; // property may have custom easing or progress function

      var easing = void 0,
          progress = void 0;

      if (value.value !== undefined && (value.easing || value.progress)) {
        if (typeof value.easing === 'string') {
          easing = cc.easing[value.easing];
          !easing && cc.warnID(1031, value.easing);
        } else {
          easing = value.easing;
        }

        progress = value.progress;
        value = value.value;
      }

      var isNumber = typeof value === 'number';

      if (!isNumber && (!value.lerp || relative && !value.add && !value.mul || !value.clone)) {
        cc.warn("Can not animate " + name + " property, because it do not have [lerp, (add|mul), clone] function.");
        continue;
      }

      var prop = Object.create(null);
      prop.value = value;
      prop.easing = easing;
      prop.progress = progress;
      this._props[name] = prop;
    }

    this._originProps = props;
    this.initWithDuration(duration);
  },
  clone: function clone() {
    var action = new TweenAction(this._duration, this._originProps, this._opts);

    this._cloneDecoration(action);

    return action;
  },
  startWithTarget: function startWithTarget(target) {
    cc.ActionInterval.prototype.startWithTarget.call(this, target);
    var relative = !!this._opts.relative;
    var props = this._props;

    for (var name in props) {
      var value = target[name];
      var prop = props[name];

      if (typeof value === 'number') {
        prop.start = value;
        prop.current = value;
        prop.end = relative ? value + prop.value : prop.value;
      } else {
        prop.start = value.clone();
        prop.current = value.clone();
        prop.end = relative ? (value.add || value.mul).call(value, prop.value) : prop.value;
      }
    }
  },
  update: function update(t) {
    var opts = this._opts;
    var easingTime = t;
    if (opts.easing) easingTime = opts.easing(t);
    var target = this.target;
    if (!target) return;
    var props = this._props;
    var progress = this._opts.progress;

    for (var name in props) {
      var prop = props[name];
      var time = prop.easing ? prop.easing(t) : easingTime;
      var current = prop.current = (prop.progress || progress)(prop.start, prop.end, prop.current, time);
      target[name] = current;
    }
  },
  progress: function progress(start, end, current, t) {
    if (typeof start === 'number') {
      current = start + (end - start) * t;
    } else {
      start.lerp(end, t, current);
    }

    return current;
  }
});
var SetAction = cc.Class({
  name: 'cc.SetAction',
  "extends": cc.ActionInstant,
  ctor: function ctor(props) {
    this._props = {};
    props !== undefined && this.init(props);
  },
  init: function init(props) {
    for (var name in props) {
      this._props[name] = props[name];
    }

    return true;
  },
  update: function update() {
    var props = this._props;
    var target = this.target;

    for (var name in props) {
      target[name] = props[name];
    }
  },
  clone: function clone() {
    var action = new SetAction();
    action.init(this._props);
    return action;
  }
});
/**
 * !#en
 * Tween provide a simple and flexible way to create action.
 * Tween's api is more flexible than cc.Action:
 *  - Support creating an action sequence in chained api,
 *  - Support animate any objects' any properties, not limited to node's properties.
 *    By contrast, cc.Action needs to create a new action class to support new node property.
 *  - Support working with cc.Action,
 *  - Support easing and progress function.
 * !#zh
 * Tween 提供了一个简单灵活的方法来创建 action。
 * 相对于 Cocos 传统的 cc.Action，cc.Tween 在创建动画上要灵活非常多：
 *  - 支持以链式结构的方式创建一个动画序列。
 *  - 支持对任意对象的任意属性进行缓动，不再局限于节点上的属性，而 cc.Action 添加一个属性的支持时还需要添加一个新的 action 类型。
 *  - 支持与 cc.Action 混用
 *  - 支持设置 {{#crossLink "Easing"}}{{/crossLink}} 或者 progress 函数
 * @class Tween
 * @example
 * cc.tween(node)
 *   .to(1, {scale: 2, position: cc.v3(100, 100, 100)})
 *   .call(() => { console.log('This is a callback'); })
 *   .by(1, {scale: 3, position: cc.v3(200, 200, 200)}, {easing: 'sineOutIn'})
 *   .start(cc.find('Canvas/cocos'));
 */

function Tween(target) {
  this._actions = [];
  this._finalAction = null;
  this._target = target;
  this._tag = cc.Action.TAG_INVALID;
}
/**
 * @method constructor
 * @param {Object} [target]
 */

/**
 * !#en Stop all tweens
 * !#zh 停止所有缓动
 * @method stopAll
 * @static
 */


Tween.stopAll = function () {
  cc.director.getActionManager().removeAllActions();
};
/**
 * !#en Stop all tweens by tag
 * !#zh 停止所有指定标签的缓动
 * @method stopAllByTag
 * @static
 * @param {number} tag
 */


Tween.stopAllByTag = function (tag) {
  cc.director.getActionManager().removeActionByTag(tag);
};
/**
 * !#en Stop all tweens by target
 * !#zh 停止所有指定对象的缓动
 * @method stopAllByTarget
 * @static
 * @param {Object} target
 */


Tween.stopAllByTarget = function (target) {
  cc.director.getActionManager().removeAllActionsFromTarget(target);
};
/**
 * !#en
 * Insert an action or tween to this sequence
 * !#zh
 * 插入一个 action 或者 tween 到队列中
 * @method then 
 * @param {Action|Tween} other
 * @return {Tween}
 */


Tween.prototype.then = function (other) {
  if (other instanceof cc.Action) {
    this._actions.push(other.clone());
  } else {
    this._actions.push(other._union());
  }

  return this;
};
/**
 * !#en
 * Set tween target
 * !#zh
 * 设置 tween 的 target
 * @method target
 * @param {Object} target
 * @return {Tween}
 */


Tween.prototype.target = function (target) {
  this._target = target;
  return this;
};
/**
 * !#en
 * Start this tween
 * !#zh
 * 运行当前 tween
 * @method start
 * @return {Tween}
 */


Tween.prototype.start = function () {
  var target = this._target;

  if (!target) {
    cc.warn('Please set target to tween first');
    return this;
  }

  if (target instanceof cc.Object && !target.isValid) {
    return;
  }

  if (this._finalAction) {
    cc.director.getActionManager().removeAction(this._finalAction);
  }

  this._finalAction = this._union();

  if (target._id === undefined) {
    target._id = ++_tweenID;
  }

  this._finalAction.setTag(this._tag);

  cc.director.getActionManager().addAction(this._finalAction, target, false);
  return this;
};
/**
 * !#en
 * Stop this tween
 * !#zh
 * 停止当前 tween
 * @method stop
 * @return {Tween}
 */


Tween.prototype.stop = function () {
  if (this._finalAction) {
    cc.director.getActionManager().removeAction(this._finalAction);
  }

  return this;
};
/**
 * !#en Sets tween tag
 * !#zh 设置缓动的标签
 * @method tag
 * @param {number} tag
 * @return {Tween}
 */


Tween.prototype.tag = function (tag) {
  this._tag = tag;
  return this;
};
/**
 * !#en
 * Clone a tween
 * !#zh
 * 克隆当前 tween
 * @method clone
 * @param {Object} [target]
 * @return {Tween}
 */


Tween.prototype.clone = function (target) {
  var action = this._union();

  return cc.tween(target).then(action.clone());
};
/**
 * !#en
 * Integrate all previous actions to an action.
 * !#zh
 * 将之前所有的 action 整合为一个 action。
 * @method union
 * @return {Tween}
 */


Tween.prototype.union = function () {
  var action = this._union();

  this._actions.length = 0;

  this._actions.push(action);

  return this;
};

Tween.prototype._union = function () {
  var actions = this._actions;

  if (actions.length === 1) {
    actions = actions[0];
  } else {
    actions = cc.sequence(actions);
  }

  return actions;
};

Object.assign(Tween.prototype, {
  /**
   * !#en Sets target's position property according to the bezier curve.
   * !#zh 按照贝塞尔路径设置目标的 position 属性。
   * @method bezierTo
   * @param {number} duration 
   * @param {cc.Vec2} c1
   * @param {cc.Vec2} c2
   * @param {cc.Vec2} to
   * @return {Tween}
   */
  bezierTo: function bezierTo(duration, c1, c2, to, opts) {
    var c0x = c1.x,
        c0y = c1.y,
        c1x = c2.x,
        c1y = c2.y;
    opts = opts || Object.create(null);

    opts.progress = function (start, end, current, t) {
      current.x = (0, _bezier.bezier)(start.x, c0x, c1x, end.x, t);
      current.y = (0, _bezier.bezier)(start.y, c0y, c1y, end.y, t);
      return current;
    };

    return this.to(duration, {
      position: to
    }, opts);
  },

  /**
   * !#en Sets target's position property according to the bezier curve.
   * !#zh 按照贝塞尔路径设置目标的 position 属性。
   * @method bezierBy
   * @param {number} duration 
   * @param {cc.Vec2} c1
   * @param {cc.Vec2} c2
   * @param {cc.Vec2} to
   * @return {Tween} 
   */
  bezierBy: function bezierBy(duration, c1, c2, to, opts) {
    var c0x = c1.x,
        c0y = c1.y,
        c1x = c2.x,
        c1y = c2.y;
    opts = opts || Object.create(null);

    opts.progress = function (start, end, current, t) {
      var sx = start.x,
          sy = start.y;
      current.x = (0, _bezier.bezier)(sx, c0x + sx, c1x + sx, end.x, t);
      current.y = (0, _bezier.bezier)(sy, c0y + sy, c1y + sy, end.y, t);
      return current;
    };

    return this.by(duration, {
      position: to
    }, opts);
  },

  /**
   * !#en Flips target's scaleX
   * !#zh 翻转目标的 scaleX 属性
   * @method flipX
   * @return {Tween}
   */
  flipX: function flipX() {
    var _this = this;

    return this.call(function () {
      _this._target.scaleX *= -1;
    }, this);
  },

  /**
   * !#en Flips target's scaleY
   * !#zh 翻转目标的 scaleY 属性
   * @method flipY
   * @return {Tween}
   */
  flipY: function flipY() {
    var _this2 = this;

    return this.call(function () {
      _this2._target.scaleY *= -1;
    }, this);
  },

  /**
   * !#en Blinks target by set target's opacity property
   * !#zh 通过设置目标的 opacity 属性达到闪烁效果
   * @method blink
   * @param {number} duration 
   * @param {number} times 
   * @param {Object} [opts] 
   * @param {Function} [opts.progress]
   * @param {Function|String} [opts.easing]
   * @return {Tween}
   */
  blink: function blink(duration, times, opts) {
    var slice = 1.0 / times;
    opts = opts || Object.create(null);

    opts.progress = function (start, end, current, t) {
      if (t >= 1) {
        return start;
      } else {
        var m = t % slice;
        return m > slice / 2 ? 255 : 0;
      }
    };

    return this.to(duration, {
      opacity: 1
    }, opts);
  }
});
var tmp_args = [];

function wrapAction(action) {
  return function () {
    tmp_args.length = 0;

    for (var l = arguments.length, i = 0; i < l; i++) {
      var arg = tmp_args[i] = arguments[i];

      if (arg instanceof Tween) {
        tmp_args[i] = arg._union();
      }
    }

    return action.apply(this, tmp_args);
  };
}

var actions = {
  /**
   * !#en
   * Add an action which calculate with absolute value
   * !#zh
   * 添加一个对属性进行绝对值计算的 action
   * @method to
   * @param {Number} duration 
   * @param {Object} props - {scale: 2, position: cc.v3(100, 100, 100)}
   * @param {Object} [opts] 
   * @param {Function} [opts.progress]
   * @param {Function|String} [opts.easing]
   * @return {Tween}
   * @typescript
   * to <OPTS extends Partial<{progress: Function, easing: Function|String}>> (duration: number, props: ConstructorType<T>, opts?: OPTS) : Tween
   */
  to: function to(duration, props, opts) {
    opts = opts || Object.create(null);
    opts.relative = false;
    return new TweenAction(duration, props, opts);
  },

  /**
   * !#en
   * Add an action which calculate with relative value
   * !#zh
   * 添加一个对属性进行相对值计算的 action
   * @method by
   * @param {Number} duration 
   * @param {Object} props - {scale: 2, position: cc.v3(100, 100, 100)}
   * @param {Object} [opts] 
   * @param {Function} [opts.progress]
   * @param {Function|String} [opts.easing]
   * @return {Tween}
   * @typescript
   * by <OPTS extends Partial<{progress: Function, easing: Function|String}>> (duration: number, props: ConstructorType<T>, opts?: OPTS) : Tween
   */
  by: function by(duration, props, opts) {
    opts = opts || Object.create(null);
    opts.relative = true;
    return new TweenAction(duration, props, opts);
  },

  /**
   * !#en
   * Directly set target properties
   * !#zh
   * 直接设置 target 的属性
   * @method set
   * @param {Object} props
   * @return {Tween}
   * @typescript
   * set (props: ConstructorType<T>) : Tween
   */
  set: function set(props) {
    return new SetAction(props);
  },

  /**
   * !#en
   * Add an delay action
   * !#zh
   * 添加一个延时 action
   * @method delay
   * @param {Number} duration 
   * @return {Tween}
   */
  delay: cc.delayTime,

  /**
   * !#en
   * Add an callback action
   * !#zh
   * 添加一个回调 action
   * @method call
   * @param {Function} callback
   * @return {Tween}
   */
  call: cc.callFunc,

  /**
   * !#en
   * Add an hide action
   * !#zh
   * 添加一个隐藏 action
   * @method hide
   * @return {Tween}
   */
  hide: cc.hide,

  /**
   * !#en
   * Add an show action
   * !#zh
   * 添加一个显示 action
   * @method show
   * @return {Tween}
   */
  show: cc.show,

  /**
   * !#en
   * Add an removeSelf action
   * !#zh
   * 添加一个移除自己 action
   * @method removeSelf
   * @return {Tween}
   */
  removeSelf: cc.removeSelf,

  /**
   * !#en
   * Add an sequence action
   * !#zh
   * 添加一个队列 action
   * @method sequence
   * @param {Action|Tween} action
   * @param {Action|Tween} ...actions
   * @return {Tween}
   */
  sequence: wrapAction(cc.sequence),

  /**
   * !#en
   * Add an parallel action
   * !#zh
   * 添加一个并行 action
   * @method parallel
   * @param {Action|Tween} action
   * @param {Action|Tween} ...actions
   * @return {Tween}
   */
  parallel: wrapAction(cc.spawn)
}; // these action will use previous action as their parameters

var previousAsInputActions = {
  /**
   * !#en
   * Add an repeat action. 
   * This action will integrate before actions to a sequence action as their parameters.
   * !#zh
   * 添加一个重复 action，这个 action 会将前一个动作作为他的参数。
   * @method repeat
   * @param {Number} repeatTimes 
   * @param {Action | Tween} [action]
   * @return {Tween}
   */
  repeat: cc.repeat,

  /**
   * !#en
   * Add an repeat forever action
   * This action will integrate before actions to a sequence action as their parameters.
   * !#zh
   * 添加一个永久重复 action，这个 action 会将前一个动作作为他的参数。
   * @method repeatForever
   * @param {Action | Tween} [action]
   * @return {Tween}
   */
  repeatForever: function repeatForever(action) {
    // TODO: fixed with cc.repeatForever
    return cc.repeat(action, 10e8);
  },

  /**
   * !#en
   * Add an reverse time action.
   * This action will integrate before actions to a sequence action as their parameters.
   * !#zh
   * 添加一个倒置时间 action，这个 action 会将前一个动作作为他的参数。
   * @method reverseTime
   * @param {Action | Tween} [action]
   * @return {Tween}
   */
  reverseTime: cc.reverseTime
};
var keys = Object.keys(actions);

var _loop = function _loop(i) {
  var key = keys[i];

  Tween.prototype[key] = function () {
    var action = actions[key].apply(this, arguments);

    this._actions.push(action);

    return this;
  };
};

for (var i = 0; i < keys.length; i++) {
  _loop(i);
}

keys = Object.keys(previousAsInputActions);

var _loop2 = function _loop2(_i) {
  var key = keys[_i];

  Tween.prototype[key] = function () {
    var actions = this._actions;
    var action = arguments[arguments.length - 1];
    var length = arguments.length - 1;

    if (action instanceof cc.Tween) {
      action = action._union();
    } else if (!(action instanceof cc.Action)) {
      action = actions[actions.length - 1];
      actions.length -= 1;
      length += 1;
    }

    var args = [action];

    for (var _i2 = 0; _i2 < length; _i2++) {
      args.push(arguments[_i2]);
    }

    action = previousAsInputActions[key].apply(this, args);
    actions.push(action);
    return this;
  };
};

for (var _i = 0; _i < keys.length; _i++) {
  _loop2(_i);
}
/**
 * @module cc
 */

/**
 * @method tween
 * @param {Object} [target] - the target to animate
 * @return {Tween}
 * @typescript
 * tween<T> (target?: Object) : Tween<T>
 */


cc.tween = function (target) {
  return new Tween(target);
};

cc.Tween = Tween;
                    }
                    if (nodeEnv) {
                        __define(__module.exports, __require, __module);
                    }
                    else {
                        __quick_compile_engine__.registerModuleFunc(__filename, function () {
                            __define(__module.exports, __require, __module);
                        });
                    }
                })();
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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