"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mkdirsSync = mkdirsSync;
exports.getServerIPAndPort = getServerIPAndPort;
exports.getClientIPAddress = getClientIPAddress;
exports.stripPrefixForIPV4MappedIPV6Address = stripPrefixForIPV4MappedIPV6Address;
exports.generateQrcode = generateQrcode;
exports.outputQRCodeOnTerminal = outputQRCodeOnTerminal;
exports.traverseDirSync = traverseDirSync;
exports.getProjectName = getProjectName;
exports.getClientFromRequest = getClientFromRequest;
exports.emitWSEvent = emitWSEvent;
exports.getInspectorUrl = getInspectorUrl;
exports.getDebugInfoFromRequest = getDebugInfoFromRequest;
exports.startChrome = startChrome;
exports.SOURCE = exports.getPort = exports.colorconsole = exports.LINK_MODE = void 0;

var _os = _interopRequireDefault(require("os"));

var _net = _interopRequireDefault(require("net"));

var _fs = _interopRequireDefault(require("fs"));

var _path = _interopRequireDefault(require("path"));

var _chalk = _interopRequireDefault(require("chalk"));

var _qrcodeTerminal = _interopRequireDefault(require("qrcode-terminal"));

var _chromeSimpleLauncher = _interopRequireDefault(require("chrome-simple-launcher"));

var LINK_MODE = {
  NULL: 0,
  WIFI: 1,
  ADB: 2
};
exports.LINK_MODE = LINK_MODE;
var logLevelMap = {};

function prependLevel(levelName, args) {
  !logLevelMap[levelName] && function (levelNameInner) {
    var logLevel = levelNameInner.toUpperCase().substr(0, 4);
    logLevel.paddEnd && logLevel.paddEnd(4);
    logLevelMap[levelName] = logLevel;
  }(levelName);

  if (typeof args[0] === 'string' && args[0].length > 1 && args[0][0] !== '[') {
    args[0] = "[".concat(logLevelMap[levelName], "] ").concat(args[0]);
  }
}
/**
 * 带颜色的 info, log, warn, error, trace 的输出工具函数
 */


var colorconsole = {
  trace: function trace() {
    var _console;

    for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
      args[_key] = arguments[_key];
    }

    prependLevel('trace', args);

    (_console = console).trace.apply(_console, args);
  },
  log: function log() {
    for (var _len2 = arguments.length, args = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
      args[_key2] = arguments[_key2];
    }

    prependLevel('log', args);
    console.log(_chalk["default"].green.apply(_chalk["default"], args));
  },
  info: function info() {
    for (var _len3 = arguments.length, args = new Array(_len3), _key3 = 0; _key3 < _len3; _key3++) {
      args[_key3] = arguments[_key3];
    }

    prependLevel('info', args);
    console.info(_chalk["default"].green.apply(_chalk["default"], args));
  },
  warn: function warn() {
    var _chalk$yellow;

    for (var _len4 = arguments.length, args = new Array(_len4), _key4 = 0; _key4 < _len4; _key4++) {
      args[_key4] = arguments[_key4];
    }

    prependLevel('warn', args);
    console.warn((_chalk$yellow = _chalk["default"].yellow).bold.apply(_chalk$yellow, args));
  },
  error: function error() {
    var _chalk$red;

    for (var _len5 = arguments.length, args = new Array(_len5), _key5 = 0; _key5 < _len5; _key5++) {
      args[_key5] = arguments[_key5];
    }

    prependLevel('error', args);
    console.error((_chalk$red = _chalk["default"].red).bold.apply(_chalk$red, args));
  },
  "throw": function _throw() {
    var _chalk$red2;

    throw new Error((_chalk$red2 = _chalk["default"].red).bold.apply(_chalk$red2, arguments));
  }
};
/**
 * 创建任意深度的路径的文件夹
 * @param dirname
 * @returns {boolean}
 */

exports.colorconsole = colorconsole;

function mkdirsSync(dirname) {
  if (_fs["default"].existsSync(dirname)) {
    return true;
  } else {
    if (mkdirsSync(_path["default"].dirname(dirname))) {
      _fs["default"].mkdirSync(dirname);

      return true;
    }
  }
}
/**
 * 取得server端的IP和port
 * @param port {number}
 * @returns {string}
 */


function getServerIPAndPort(port) {
  return (getIPv4IPAddress() || '127.0.0.1') + "".concat(port === 80 ? '' : ':' + port);
}
/**
 * 获取客户端ip
 * @param req node Http IncomingRequest Object
 * @returns {any|*|string}
 */


function getClientIPAddress(req) {
  var ip = req.headers['x-forwarded-for'] || req.connection && req.connection.remoteAddress || req.socket && req.socket.remoteAddress || req.connection && req.connection.socket && req.connection.socket.remoteAddress;
  return stripPrefixForIPV4MappedIPV6Address(ip);
}
/**
 * 检测可能是IpV4-mapped IpV6 格式的ip
 * - https://en.wikipedia.org/wiki/IPv6#IPv4-mapped_IPv6_addresses
 *
 * @param ip  IpV4-mapped IpV6 string
 * @returns {*}
 */


function stripPrefixForIPV4MappedIPV6Address(ip) {
  if (/^::1$/.test(ip)) {
    ip = '127.0.0.1';
  }

  if (/^::.{0,4}:(\d{1,3}\.){3}\d{1,3}/.test(ip)) {
    ip = ip.replace(/^.*:/, '');
  }

  return ip;
}
/**
 * 二维码生成
 * @param {*} port
 */


function generateQrcode(port) {
  var ip = getIPv4IPAddress();

  if (!ip) {
    colorconsole.warn('### App Server ### 本机IP地址为空，无法通过WIFI调试');
    return;
  }

  var addr2 = "http://".concat(ip, ":").concat(port);
  colorconsole.info('');
  colorconsole.info("### App Server ### \u670D\u52A1\u5668\u5730\u5740:".concat(addr2));
  colorconsole.info('### App Server ### 请确保手机与App Server处于相同网段');
  colorconsole.info(''); // 输出二维码地址

  outputQRCodeOnTerminal(addr2); //startChrome(debugAddr)
}
/**
 * 命令行输出二维码
 * @param text
 */


function outputQRCodeOnTerminal(text) {
  console.info('');
  console.info("\u751F\u6210HTTP\u670D\u52A1\u5668\u7684\u4E8C\u7EF4\u7801: ".concat(text));

  _qrcodeTerminal["default"].generate(text, {
    small: true
  });
}
/**
 * 获取可用的端口号
 * @param port
 * @returns {Number}
 */


var getPort = function getPort(port) {
  return (// 处理自定义端口
    new Promise(function (resolve, reject) {
      var server = _net["default"].createServer().listen(port);

      server.on('error', function (err) {
        if (err.code === 'EADDRINUSE') {
          server.close();
          resolve(getPort(port + 1));
        } else {
          reject(err);
        }
      });
      server.on('listening', function () {
        server.close();
        resolve(port);
      });
    })
  );
};
/**
 * 遍历目录文件 同步方法
 * @param dir
 * @param files 收集的文件列表
 * @param isInWhiteListFilter 判断文件是否处于白名单的回调函数
 */


exports.getPort = getPort;

function traverseDirSync(dir, files, isInWhiteListFilter) {
  var list = _fs["default"].readdirSync(dir);

  list.forEach(function (file) {
    file = _path["default"].join(dir, file);

    var stat = _fs["default"].statSync(file);

    if (stat && stat.isDirectory()) {
      traverseDirSync(file, files, isInWhiteListFilter);
    } else {
      if (typeof isInWhiteListFilter === 'function' && !isInWhiteListFilter(file) || typeof isInWhiteListFilter !== 'function') {
        files.push(file);
      }
    }
  });
}
/**
 * 获取开发者项目中的项目名称
 */


function getProjectName() {
  try {
    var pathProject = process.cwd(); // 平级目录 没有src文件夹

    var pathManifest = _path["default"].join(pathProject, 'manifest.json');

    var contManifest = require(pathManifest);

    return contManifest && contManifest["package"] || 'Bundle';
  } catch (err) {
    colorconsole.error("### App Server ### \u83B7\u53D6\u9879\u76EE\u540D\u79F0\u51FA\u9519\uFF1A".concat(err.message));
  }
}
/**
 * 获取服务器端的IP
 */


function getIPv4IPAddress() {
  var ifaces = _os["default"].networkInterfaces();

  var result;

  for (var prop in ifaces) {
    if (Object.prototype.hasOwnProperty.call(ifaces, prop)) {
      var iface = ifaces[prop];
      iface.every(function (eachAlias) {
        if (eachAlias.family === 'IPv4' && !eachAlias.internal && eachAlias.address !== '127.0.0.1') {
          result = eachAlias;
          return false;
        }

        return true;
      });

      if (result !== void 0) {
        break;
      }
    }
  }

  return result && result.address;
}
/**
 * 获取请求信息
 * @param request
 * @returns {{clientIp: (any|*|string), sn, linkMode}}
 */


function getClientFromRequest(request) {
  var clientIp = getClientIPAddress(request);
  var serverIp = getIPv4IPAddress();
  var sn = request.header['device-serial-number'];
  var linkMode = LINK_MODE.NULL;

  if (clientIp === '127.0.0.1' && sn) {
    linkMode = LINK_MODE.ADB;
  } else if (clientIp !== '127.0.0.1' && clientIp !== serverIp) {
    linkMode = LINK_MODE.WIFI;
  }

  return {
    clientIp: clientIp,
    sn: sn,
    linkMode: linkMode
  };
}
/**
 * 发送ws事件
 */


function emitWSEvent(wsSocket, evName, data) {
  wsSocket.sockets.emit(evName, data || {});
}
/**
 * 提供insepctor页面的打开地址
 * @param opt
 * @param opt.ws  手机端的websocket的地址
 * @param opt.serverPort 调试器server的所用端口号
 * @returns {string}
 */


function getInspectorUrl(opt) {
  var ws = opt.ws,
      serverPort = opt.serverPort;
  var urlDomain = "http://".concat(getServerIPAndPort(serverPort));
  var urlPath = '/inspector/inspector.html';
  var urlQueryStr = "?ws=".concat(encodeURI(ws), "&remoteFrontend=true&dockSide=undocked");
  return "".concat(urlDomain).concat(urlPath).concat(urlQueryStr);
}
/**
 * 获取请求信息
 * @param request
 * @returns {{sn, linkMode, ws, application, devicePort}}
 */


function getDebugInfoFromRequest(request) {
  var _getClientFromRequest = getClientFromRequest(request),
      sn = _getClientFromRequest.sn,
      linkMode = _getClientFromRequest.linkMode;

  var _request$body = request.body,
      ws = _request$body.ws,
      application = _request$body.application;
  var devicePort = ws.split(':')[1].split('/')[0];
  return {
    sn: sn,
    linkMode: linkMode,
    ws: ws,
    application: application,
    devicePort: devicePort
  };
}
/**
 * 开启一个chrome进程
 */


function startChrome(url) {
  return _chromeSimpleLauncher["default"].launch(url);
}

var SOURCE = {
  // 启动hapserver的类型
  IDE: 'ide'
};
exports.SOURCE = SOURCE;