
const { spawn } = require('child_process')
const fs = require('fs')
const path = require('path')

const debuglog = require(`./debuglog`)
const lookupChromeLinux = require('./sniffer-linux')
const lookupChromeMac = require('./sniffer-mac')
const lookupChromeWindows = require('./sniffer-windows')

const lookupMap = {
  'win32': lookupChromeWindows,
  'darwin': lookupChromeMac,
  'linux': lookupChromeLinux
}

let chromeArgs = [
  '--enable-devtools-experiments',
  '--no-first-run'
]

/**
 * 启动chrome进程
 * @param url   需要开启的url
 * @param options
 * @param options.chromePath {string}  可选， "path/to/chrome_binary"
 * @returns {Promise.<*>}
 */
function launch (url, options) {
  debuglog(`launch(): start: ${JSON.stringify(options)}`)
  options = options || {}

  // 设定chrome的--user-data-dir参数
  setupUserProfileArg(options)

  const launchPromise = new Promise((resolve, reject) => {
    lookupForChromePath(options)
      .then((chromeBinaryPath) => {
        const chromeProcess = launchChrome(chromeBinaryPath, url, options)
        resolve(chromeProcess)
      }, (err) => {
        reject(err)
      })
  }).catch(err => {
    return err
  })

  return launchPromise
}

function lookupForChromePath (options) {
  return new Promise((resolve, reject) => {
    let { chromePath } = options
    // 处理--chrome-path参数的情况
    if (chromePath) {
      try {
        chromePath = path.resolve(chromePath)
        fs.accessSync(chromePath)
        return resolve(chromePath)
      }
      catch (err) {
        return reject(err)
      }
    }
    else {
      // 处理需要嗅探的情况
      const lookupMethod = lookupMap[process.platform]
      lookupMethod().then(value => {
        if (value instanceof Error) {
          return reject(value)
        }

        return resolve(value)
      })
    }
  })
}

/**
 * 如需要单独启动一个chrome窗口，需指定user-data-dir;
 * 目录为chrome launcher同一个目录;
 */
function setupUserProfileArg () {
  chromeArgs = chromeArgs.concat([
    `--user-data-dir=${path.resolve(__dirname, '.devProfile')}`
  ])
}

/**
 * 嗅探位置之后，spawn chrome进程，定义onExit.
 */
function launchChrome (filePath, url, options) {
  let child
  const argsUrl = chromeArgs.concat([url])
  debuglog(`打开chrome浏览器，参数: ${argsUrl.join(' ')}`)
  try {
    child = spawn(filePath, argsUrl)
  }
  catch (err) {
    onLaunchChromeError(err)
    return err
  }
  child.on('error', onLaunchChromeError)
  child.on('exit', onChromeExit)
  return child

  function onLaunchChromeError (err) {
    debuglog(`chrome进程出错: ${err.message}`)
    typeof options.onError === 'function' && options.onError(err)
  }

  function onChromeExit (code, signal) {
    debuglog(`chrome进程退出。(退出代码${code})`)
    typeof options.onExit === 'function' && options.onExit()
  }
}

module.exports = {
  launch
}
